package Elastijk;
use strict;
use warnings;
our $VERSION = "0.03";

use JSON ();
use URI::Escape qw(uri_escape);
use Hijk;

our $JSON = JSON->new->utf8;

sub _build_hijk_request_args {
    my $args = $_[0];
    my ($path, $qs, $uri_param);
    $path = "/". join("/", grep { defined } @{$args}{qw(index type command)});
    if ($args->{uri_param}) {
        $qs =  join('&', map { uri_escape($_) . "=" . uri_escape($args->{uri_param}{$_}) } keys %{$args->{uri_param}});
    }

    return {
        method => $args->{method} || 'GET',
        host   => $args->{host}   || 'localhost',
        port   => $args->{port}   || '9200',
        !$path ?() :(
            path   => $path
        ),
        !$qs ?() :(
            query_string => $qs
        ),
        !$args->{body} ?() :(
            body => (ref($args->{body}) ? $JSON->encode($args->{body}) : $args->{body})
        ),
    }
}

sub request {
    my $arg = $_[0];
    if ($arg->{body}) {
        $arg = {%{$_[0]}};
        $arg->{body} = $JSON->encode( $arg->{body} );
    }
    my ($status, $res_body) = request_raw($arg);
    $res_body = $res_body ? $JSON->decode($res_body) : undef;
    return $status, $res_body;
}

sub request_raw {
    my $args = _build_hijk_request_args($_[0]);
    my $res = Hijk::request($args);
    return $res->{status}, $res->{body};
}

sub new {
    shift;
    require Elastijk::oo;
    return Elastijk::oo->new(@_);
}

1;

=encoding utf-8

=head1 NAME

Elastijk - A specialized ElasticSearch client.

=head1 SYNOPSIS

    use Elastijk;

    my ($status, $response) = Elastijk::request({
        host => "localhost",
        port => "9200",
        method => "GET",

        index => "blog",
        type => "article",
        command => "_search",

        uri_param => { search_type => "dfs_query_then_fetch" }
        body => {
            query => { match => { "body" => "cpan" } }
        }
    });

    if ($status eq "200") {
        for my $hit (@{ $response->{hits}{hits} }) {
            say $hit->{url};
        }
    }

=head1 DESCRIPTION

Elastijk is a ElasticSearch client library. It uses L<Hijk>, a HTTP client that
implements a tiny subset of HTTP/1.1 just enough to talk to ElasticSearch via
HTTP.

Elastijk provided low-level functions that are almost identical as using HTTP
client, and a object-oriented sugar-layer to make it easier to use. The following
documentation describe the object-oriented uses first, then the functions.

=head2 OBJECT PROPERTIES

An Elastijk object does not use object frameworks, but just follow the perl
convention. Object is a blessed hash, while all key-value pairs in the hash are
the properties. Users could break the pacaking and modify those values, but it
is fine. All key-value pairs are shallow-copied from `new` method:

    my $es = Elastijk->new(
        host => "es1.example.com",
        port => "9200"
    );

Here's a full list of key-value pairs that are consumed:

    host  => Str "localhost"
    port  => Str "9200"
    index => Str (optional)
    type  => Str (optional)

The values for C<index> and C<type> act like a "default" value and they are only
used in methods that could use them. Those methods should also takes new values
for index or type and overrieds the defaults.

=head1 OBJECT METHODS

=head2 search( $param1 => $value1, $param2 => $value2, ... )

This method encapsulate <request body search|http://www.elasticsearch.org/guide/en/elasticsearch/reference/current/search-request-body.html>

The arguments are key-value pairs from the API documents.

=head2 uri_search ( ... )

This method encapsulate C<uri search|http://www.elasticsearch.org/guide/en/elasticsearch/reference/current/search-uri-request.html>

The arguments are key-value pairs from the API documents.

=head2 exists( index => Str )

Check if the given index exists.

See also L<http://www.elasticsearch.org/guide/en/elasticsearch/reference/current/indices-exists.html>

=head2 exists( index => Str, type => Str )

Check if the given type exists.

See also L<http://www.elasticsearch.org/guide/en/elasticsearch/reference/current/indices-types-exists.html>

=head2 delete

=head1 FUNCTIONS

=head2 Elastijk::request( $args :HashRef ) : ($status :Int, $response :HashRef)

Making a request to the ElasticSearch server specified in C<$args>. It returns 2
values. C<$status> is the HTTP status code of the response, and the C<$response>
decoded as HashRef. ElasticSearch API always respond a single HashRef as JSON
text, this might or might not be changed in the future, if it is changed then
this function will be adjusted accordingly.

=head2 Elastijk::request_raw( $args :HashRef ) : ($status :Int, $response :Str)

Making a request to the ElasticSearch server specified in C<$args>. The main
difference between this function and C<Elastijk::request> is that
C<$args->{body}> s expected to be a String scalar, rather then a HashRef. And
the $response is not decoded from JSON. This function can be used if users wish
to use their own JSON parser to parse response, or if they wish to delay the
parsing to be done latter in some bulk-processing pipeline.

=head1 AUTHORS

=over 4

=item Kang-min Liu <gugod@gugod.org>

=item Borislav Nikolov <jack@sofialondonmoskva.com>

=back

=head1 COPYRIGHT

Copyright (c) 2013 Kang-min Liu C<< <gugod@gugod.org> >>.

=head1 LICENCE

The MIT License

=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE IS WITH
YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL
NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE SOFTWARE AS PERMITTED BY THE ABOVE LICENCE, BE
LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE
THE SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.

=cut
