package DBIx::Class::EncodedColumn::Crypt::PBKDF2;
# ABSTRACT: PBKDF2 support for DBIx::Class::EncodedColumn

use strict;
use warnings;

use Carp;
use Crypt::PBKDF2;

our $VERSION = '0.000001'; # TRIAL VERSION: generated by DZP::OurPkgVersion

sub make_encode_sub {
	my ( $class, $col, $args ) = @_;

	my $pbkdf2 = _get_pbkdf2_obj_from_args($args);

	return sub {
		my ( $plain_text ) = @_;
		return $pbkdf2->generate($plain_text);
	};
} ## end sub make_encode_sub


sub make_check_sub {
	my ( $class, $col, $args ) = @_;

	my $pbkdf2 = _get_pbkdf2_obj_from_args($args);

	return sub {
		my ( $self, $password ) = @_;
		croak 'Must supply a password to check' unless $password;
		return $pbkdf2->validate( $self->get_column($col), $password );
	};
} ## end sub make_check_sub

sub _get_pbkdf2_obj_from_args {
    my ( $args ) = @_;

	my %new_args = %$args;
# 	my %new_args;  ## We should probably do some validation of individual args
# 	$new_args{hash_class}   = $args->{hash_class}   if exists $args->{hash_class}; ## 'HMACSHA1', 'HMACSHA3'
# 	$new_args{hash_args}    = $args->{hash_args}    if exists $args->{hash_args};  ##  { sha_size => 512 }
# 	$new_args{iterations}   = $args->{iterations}   if exists $args->{iterations}; ## 1000
# 	$new_args{output_len}   = $args->{output_len}   if exists $args->{output_len}; ## 20
# 	$new_args{salt_len}     = $args->{salt_len}     if exists $args->{salt_len};   ## 4
# 	$new_args{encoding}     = $args->{encoding}     if exists $args->{encoding};   ## 'ldap'
# 	$new_args{length_limit} = $args->{length_limit} if exists $args->{length_limit};

	my $pbkdf2 = Crypt::PBKDF2->new(%new_args); 
	croak sprintf( "failed to create pbkdf2: %s\n", $_ ) unless $pbkdf2;

	return $pbkdf2;
}


1;

=pod

=encoding UTF-8

=head1 NAME

DBIx::Class::EncodedColumn::Crypt::PBKDF2 - PBKDF2 support for DBIx::Class::EncodedColumn

=head1 VERSION

version 0.000001

=head1 SYNOPSIS

  __PACKAGE__->add_columns(
      'password' => {
          data_type           => 'text',
          encode_column       => 1,
          encode_class        => 'Crypt::PBKDF2',
          encode_args         => {
              hash_class => 'HMACSHA1',
              iterations => 1000
          },
          encode_check_method => 'check_password',
      }
  )

=head1 DESCRIPTION

=head1 NAME

DBIx::Class::EncodedColumn::Crypt::PBKDF2 - PBKDF2 support for DBIx::Class::EncodedColumn

=head1 NAME

DBIx::Class::EncodedColumn::Crypt::PBKDF2

=head1 ARGUMENTS for C<encode_args> - same as args for C<Crypt::PBKDF2->new>

=head2 hash_class

B<Type:> String, B<Default:> HMACSHA1

The name of the default class that will provide PBKDF2's Pseudo-Random
Function (the backend hash). If the value starts with a C<+>, the C<+> will
be removed and the remainder will be taken as a fully-qualified package
name. Otherwise, the value will be appended to C<Crypt::PBKDF2::Hash::>.

=head2 hash_args

B<Type:> HashRef, B<Default:> {}

Arguments to be passed to the C<hash_class> constructor.

=head2 hasher

B<Type:> Object (must fulfill role L<Crypt::PBKDF2::Hash>), B<Default:> None.

It is also possible to provide a hash object directly; in this case the
C<hash_class> and C<hash_args> are ignored.

=head2 iterations

B<Type:> Integer, B<Default:> 1000.

The default number of iterations of the hashing function to use for the
C<generate> and C<PBKDF2> methods.

=head2 output_len

B<Type:> Integer.

The default size (in bytes, not bits) of the output hash. If a value isn't
provided, the output size depends on the C<hash_class>S< / >C<hasher>
selected, and will equal the output size of the backend hash (e.g. 20 bytes
for HMACSHA1).

=head2 salt_len

B<Type:> Integer, B<Default:> 4

The default salt length (in bytes) for the C<generate> method.

=head2 encoding

B<Type:> String (either "crypt" or "ldap"), B<Default:> "ldap"

The hash format to generate. The "ldap" format is intended to be compatible
with RFC2307, and looks like:

    {X-PBKDF2}HMACSHA1:AAAD6A:8ODUPA==:1HSdSVVwlWSZhbPGO7GIZ4iUbrk=

While the "crypt" format is similar to the format used by the C<crypt()>
function, except with more structured information in the second (salt) field.
It looks like:

    $PBKDF2$HMACSHA1:1000:4q9OTg==$9Pb6bCRgnct/dga+4v4Lyv8x31s=

Versions of this module up to 0.110461 generated the "crypt" format, so set
that if you want it. Current versions of this module will read either format,
but the "ldap" format is preferred.

=head2 length_limit

B<Type:> Integer

The maximum password length to allow, for generate and verify functions.
Allowing passwords of unlimited length can allow a denial-of-service attack
in which an attacker asks the server to validate very large passwords.

For compatibility this attribute is unset by default, but it is recommended
to set it to a reasonably small value like 100 -- large enough that users
aren't discouraged from having secure passwords, but small enough to limit
the computation needed to validate any one password.

=head1 METHODS

=head2 make_encode_sub($column_name, \%encode_args)

Returns a coderef that accepts a plaintext value and returns an
encoded value.

=head2 make_check_sub($column_name, \%encode_args)

Returns a coderef that when given the row object and a plaintext value
will return a boolean if the plaintext matches the encoded value. This
is typically used for password authentication.

=head1 SEE ALSO

=over 4

=item *

L<DBIx::Class::EncodedColumn>

=item *

L<Crypt::PBKDF2>

=back

=head1 AUTHOR

Charlie Garrison <garrison@zeta.org.au>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by Charlie Garrison.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut

__END__;
