/*
 * Copyright (C) 2012 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VanDine <ken.vandine@canonical.com>
 */

/*
 * build with:
 * valac -X -I../src -X -L../src/.libs -X -lfriends ../src/friends-0.1.vapi --basedir ./ --pkg glib-2.0 --pkg dee-1.0  example-vala.vala -o example-vala
 */

public class Example
{
  public Friends.Dispatcher dispatcher;
  public bool waiting = false;

  public Example (MainLoop loop)
  {
      try {
          dispatcher = new Friends.Dispatcher ();
      } catch (GLib.IOError e) {
          warning (e.message);
          dispatcher = null;
      }
  }

  public void run_example (string[] args)
  {
    if (args.length < 2) {
        PrintUsage ();
        return;
    }
    switch (args[1]) {
        case "--refresh":
            stdout.printf ("Refreshing friends\n");
            try {
                dispatcher.refresh ();
            } catch (GLib.IOError e) {
                warning (e.message);
            }
            return;

        case "--features":
            if (args.length < 3) {
                PrintUsage ();
                return;
            }
            stdout.printf ("Getting features for %s\n", args[2]);
            try {
                var features = dispatcher.features (args[2]);
                foreach (string feature in features)
                    stdout.printf ("%s\n", feature);
            } catch (GLib.IOError e) {
                warning (e.message);
            }
            return;

        case "--like":
            if (args.length < 3) {
                PrintUsage ();
                return;
            }
            waiting = true;
            for (int i=1; i<args.length; i++) {
                if (args[i] == "--like")
                    if (args[i+1] != null && args[i+2] != null) {
                        uint acct = int.parse (args[i+1]);
                        string id = args[i+2];
                        stdout.printf ("Liking: %s\n", id);
                        dispatcher.like_async.begin (
                            acct, id, (obj, res) => {
                                string result = "";
                                try {
                                    dispatcher.like_async.end (res, out result);
                                } catch (GLib.IOError e) {
                                    warning (e.message);
                                }
                                stdout.printf ("Liked: %s\n", result);
                                waiting = false;
                            }
                        );
                    } else {
                        PrintUsage ();
                    }
            }
            return;

        case "--unlike":
            waiting = true;
            for (int i=1; i<args.length; i++) {
                if (args[i] == "--unlike")
                    if (args[i+1] != null && args[i+2] != null) {
                        uint acct = int.parse (args[i+1]);
                        string id = args[i+2];
                        stdout.printf ("Unliking: %s\n", id);
                        dispatcher.unlike_async.begin (
                            acct, id, (obj, res) => {
                                string result = "";
                                try {
                                    dispatcher.unlike_async.end (res, out result);
                                } catch (GLib.IOError e) {
                                    warning (e.message);
                                }
                                stdout.printf ("Unliked: %s\n", result);
                                waiting = false;
                            }
                        );
                    } else {
                        PrintUsage ();
                    }
            }
            return;

        case "--delete":
            waiting = true;
            for (int i=1; i<args.length; i++) {
                if (args[i] == "--delete")
                    if (args[i+1] != null && args[i+2] != null) {
                        uint acct = int.parse (args[i+1]);
                        string id = args[i+2];
                        stdout.printf ("Deleting: %s\n", id);
                        dispatcher.delete_async.begin (
                            acct, id, (obj, res) => {
                                string result = "";
                                try {
                                    dispatcher.delete_async.end (res, out result);
                                } catch (GLib.IOError e) {
                                    warning (e.message);
                                }
                                stdout.printf ("Deleted: %s\n", result);
                                waiting = false;
                            }
                        );
                    } else {
                        PrintUsage ();
                    }
            }
            return;

        case "--retweet":
            waiting = true;
            for (int i=1; i<args.length; i++) {
                if (args[i] == "--retweet")
                    if (args[i+1] != null && args[i+2] != null) {
                        uint acct = int.parse (args[i+1]);
                        string id = args[i+2];
                        stdout.printf ("Retweet...\n");
                        dispatcher.retweet_async.begin (
                            acct, id, (obj, res) => {
                                string result = "";
                                try {
                                    dispatcher.retweet_async.end (res, out result);
                                } catch (GLib.IOError e) {
                                    warning (e.message);
                                }
                                stdout.printf ("Retweeted: %s\n", result);
                                waiting = false;
                            }
                        );
                    } else {
                        PrintUsage ();
                    }
            }
            return;

        case "--upload":
            uint acct = int.parse (args[2]);
            string desc = args[3];
            string uri = args[4];
            stdout.printf ("Uploading %s\n", uri);
            string result_url = "";
            waiting = true;
            dispatcher.upload_async.begin (acct, uri, desc, (obj, res) => {
                try {
                    dispatcher.upload_async.end (res, out result_url);
                    stdout.printf ("Uploaded, resulting URL is %s\n", result_url);
                    waiting = false;
                } catch (IOError e) {
                    warning (e.message);
                    waiting = false;
                }
            });
            return;

        case "--send-message":
            if (args.length < 3) {
                PrintUsage ();
                return;
            }
            for (int i=1; i<args.length; i++) {
                if (args[i] == "--send-message")
                    if (args[i+1] != null && args[i+2] != null) {
                        uint acct = int.parse (args[i+1]);
                        string message = args[i+2];
                        stdout.printf ("Posting...\n");
                        string result = "";
                        waiting = true;
                        dispatcher.send_message_async.begin (
                            acct, message, (obj, res) => {
                                try {
                                    dispatcher.send_message_async.end (
                                        res, out result);
                                } catch (GLib.IOError e) {
                                    warning (e.message);
                                }
                                stdout.printf ("Sent: %s\n", result);
                                waiting = false;
                            }
                        );
                    } else {
                        PrintUsage ();
                    }
            }
            return;

        case "--send-reply":
            if (args.length < 3) {
                PrintUsage ();
                return;
            }
            for (int i=1; i<args.length; i++) {
                if (args[i] == "--send-reply")
                    if (args[i+1] != null && args[i+2] != null && args[i+3] != null) {
                        uint acct = int.parse (args[i+1]);
                        string message_id = args[i+2];
                        string message = args[i+3];
                        stdout.printf ("Replying...\n");
                        string result = "";
                        waiting = true;
                        dispatcher.send_reply_async.begin (
                            acct, message_id, message, (obj, res) => {
                                try {
                                    dispatcher.send_reply_async.end (
                                        res, out result);
                                } catch (GLib.IOError e) {
                                    warning (e.message);
                                }
                                stdout.printf ("Sent: %s\n", result);
                                waiting = false;
                            }
                        );
                    } else {
                        PrintUsage ();
                    }
            }
            return;

        case "--help":
            PrintUsage ();
            return;

        default:
            return;
    }
  }

  private static void PrintUsage () {
      string usage = "Usage:\n" +
            "  --help\t\t\t\t\t\tPrint this usage message.\n" +
            "  --refresh\t\t\t\t\t\tRefresh the friends-dispatcher\n" +
            "  --like [account id] [message_id]\t\t\tLike a post\n" +
            "  --unlike [account id] [message_id]\t\t\tUnlike a post\n" +
            "  --delete [account id] [message_id]\t\t\tDelete a post\n" +
            "  --retweet [account id] [message_id]\t\t\tRetweet\n" +
            "  --send-message [account id] [message]\t\t\tMessage to post\n" +
            "  --send-reply [account id] [message_id] [message]\tReply to post\n" +
            "  --upload [account id] [description] [URI]\tUpload phone\n" +
            " \n";

        stdout.printf ("%s\n".printf (usage));
  }
}

static void main (string[] args) {
    var loop = new MainLoop ();
    Example e = new Example (loop);
    Timeout.add (100, () => {
        e.run_example (args);
        Idle.add ( () => {
            if (!e.waiting)
                loop.quit ();
            return true;
        });
        return false;
    });
    loop.run ();
}
