from typing import Any
from .lava_job_definition import (
    generate_metadata,
    NUMBER_OF_ATTEMPTS_LAVA_BOOT,
    artifact_download_steps,
)


def generate_lava_yaml_payload(args) -> dict[str, Any]:
    values = generate_metadata(args)

    # URLs to our kernel rootfs to boot from, both generated by the base
    # container build

    nfsrootfs = {
        "url": f"{args.rootfs_url_prefix}/lava-rootfs.tar.zst",
        "compression": "zstd",
    }

    fastboot_deploy_nfs = {
        "timeout": {"minutes": 10},
        "to": "nfs",
        "nfsrootfs": nfsrootfs,
    }

    fastboot_deploy_prepare = {
        "timeout": {"minutes": 5},
        "to": "downloads",
        "os": "oe",
        "images": {
            "kernel": {
                "url": f"{args.kernel_url_prefix}/{args.kernel_image_name}",
            },
        },
        "postprocess": {
            "docker": {
                "image": "registry.gitlab.collabora.com/lava/health-check-docker",
                "steps": [
                    f"cat Image.gz {args.dtb_filename}.dtb > Image.gz+dtb",
                    "mkbootimg --kernel Image.gz+dtb"
                    + ' --cmdline "root=/dev/nfs rw nfsroot=$NFS_SERVER_IP:$NFS_ROOTFS,tcp,hard rootwait ip=dhcp init=/init"'
                    + " --pagesize 4096 --base 0x80000000 -o boot.img",
                ],
            }
        },
    }
    if args.kernel_image_type:
        fastboot_deploy_prepare["images"]["kernel"]["type"] = args.kernel_image_type
    if args.dtb_filename:
        fastboot_deploy_prepare["images"]["dtb"] = {
            "url": f"{args.kernel_url_prefix}/{args.dtb_filename}.dtb"
        }

    tftp_deploy = {
        "timeout": {"minutes": 5},
        "to": "tftp",
        "os": "oe",
        "kernel": {
            "url": f"{args.kernel_url_prefix}/{args.kernel_image_name}",
        },
        "nfsrootfs": nfsrootfs,
    }
    if args.kernel_image_type:
        tftp_deploy["kernel"]["type"] = args.kernel_image_type
    if args.dtb_filename:
        tftp_deploy["dtb"] = {
            "url": f"{args.kernel_url_prefix}/{args.dtb_filename}.dtb"
        }

    fastboot_deploy = {
        "timeout": {"minutes": 2},
        "to": "fastboot",
        "docker": {
            "image": "registry.gitlab.collabora.com/lava/health-check-docker",
        },
        "images": {
            "boot": {"url": "downloads://boot.img"},
        },
    }

    fastboot_boot = {
        "timeout": {"minutes": 2},
        "docker": {"image": "registry.gitlab.collabora.com/lava/health-check-docker"},
        "failure_retry": NUMBER_OF_ATTEMPTS_LAVA_BOOT,
        "method": args.boot_method,
        "prompts": ["lava-shell:"],
        "commands": ["set_active a"],
    }

    tftp_boot = {
        "failure_retry": NUMBER_OF_ATTEMPTS_LAVA_BOOT,
        "method": args.boot_method,
        "prompts": ["lava-shell:"],
        "commands": "nfs",
    }

    # skeleton test definition: only declaring each job as a single 'test'
    # since LAVA's test parsing is not useful to us
    run_steps = []
    test = {
        "timeout": {"minutes": args.job_timeout_min},
        "failure_retry": 1,
        "definitions": [
            {
                "name": "mesa",
                "from": "inline",
                "lava-signal": "kmsg",
                "path": "inline/mesa.yaml",
                "repository": {
                    "metadata": {
                        "name": "mesa",
                        "description": "Mesa test plan",
                        "os": ["oe"],
                        "scope": ["functional"],
                        "format": "Lava-Test Test Definition 1.0",
                    },
                    "run": {"steps": run_steps},
                },
            }
        ],
    }

    # job execution script:
    #   - inline .gitlab-ci/common/init-stage1.sh
    #   - fetch and unpack per-pipeline build artifacts from build job
    #   - fetch and unpack per-job environment from lava-submit.sh
    #   - exec .gitlab-ci/common/init-stage2.sh

    with open(args.first_stage_init, "r") as init_sh:
        run_steps += [
            x.rstrip() for x in init_sh if not x.startswith("#") and x.rstrip()
        ]
    # We cannot distribute the Adreno 660 shader firmware inside rootfs,
    # since the license isn't bundled inside the repository
    if args.device_type == "sm8350-hdk":
        run_steps.append(
            "curl -L --retry 4 -f --retry-all-errors --retry-delay 60 "
            + "https://github.com/allahjasif1990/hdk888-firmware/raw/main/a660_zap.mbn "
            + '-o "/lib/firmware/qcom/sm8350/a660_zap.mbn"'
        )

    run_steps += artifact_download_steps(args)

    run_steps += [
        f"mkdir -p {args.ci_project_dir}",
        f"curl {args.build_url} | tar --zstd -x -C {args.ci_project_dir}",
        # Sleep a bit to give time for bash to dump shell xtrace messages into
        # console which may cause interleaving with LAVA_SIGNAL_STARTTC in some
        # devices like a618.
        "sleep 1",
        # Putting CI_JOB name as the testcase name, it may help LAVA farm
        # maintainers with monitoring
        f"lava-test-case 'mesa-ci_{args.mesa_job_name}' --shell /init-stage2.sh",
    ]

    if args.boot_method == "fastboot":
        values["actions"] = [
            {"deploy": fastboot_deploy_nfs},
            {"deploy": fastboot_deploy_prepare},
            {"deploy": fastboot_deploy},
            {"boot": fastboot_boot},
            {"test": test},
        ]
    else:  # tftp
        values["actions"] = [
            {"deploy": tftp_deploy},
            {"boot": tftp_boot},
            {"test": test},
        ]

    return values
