// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __reg_comp__
#define __reg_comp__

/** \file
 *  Wrapper for regular expression matching. */

#include <string>
#include "lexertl/state_machine.hpp"
#include "Puma/ErrorStream.h"

namespace Puma {

/** \class RegComp RegComp.h Puma/RegComp.h
 *  Instances of this class encapsulate a regular expression
 *  and the match mechanism. The implementation compiles the
 *  expression when a string is matched for the first time.
 *  In case of an error, a message can be printed on an
 *  ErrorStream object. */

class RegComp {
  std::string _expr;          // The regular expression.
  std::string _error_msg;     // Error message in case of problems
  bool _compiled;             // determines whether _sm is valid
  lexertl::state_machine _sm; // internal: lexertl compiled regular expression
        
public:
  // Constructors, etc.
  RegComp (const std::string &expr);
  RegComp (const RegComp &rc);
  RegComp &operator = (const RegComp &rc);
        
  /** Compile the regular expression.
   *  \param expr The regular expression string, which should be compiled
   *  \return 'false' in case of an error.
   *          Use comperror to get the message. */
  bool compile (const string &expr);

  /**  Match against a string.
   *  \param str String to be matched.
   *  \return 'true' if the regular expression matches the string. */
  bool match (const std::string & str);
        
  /**  To be called on errors.
   *  \param err ErrorStream on which the error message should be printed. */
  void comperror (ErrorStream &err) const;

  /** Check whether there was an error
   * \return true in case of an error; use comperror to print it. */
  bool error () const { return !_error_msg.empty (); }
};
 
 
} // namespace Puma

#endif /* __reg_comp__ */
