-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/dbi/adbc/gnu-db-adbc-resultset.adb,v $
--  Description     : Ada Database Objects - Result Object                   --
--  Author          : Michael Erdmann                                        --
--  Created         : 18.1.2002                                              --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2002/03/04 20:02:17 $                           --
--  Status          : $State: Exp $                                          --
--                                                                           --
--  Copyright (C) 2002 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  This software is implemented to work with GNAT, the GNU Ada compiler.    --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  Only Linux / MySQL                                                       --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--               purl:/net/michael.erdmann                                   --
--                                                                           --
-------------------------------------------------------------------------------
with Unchecked_Deallocation;
with Ada.Strings.Unbounded;                    use Ada.Strings.Unbounded;
with Ada.Text_IO;                              use Ada.Text_IO;

with GNU.DB.ADBC.Row;                          use GNU.DB.ADBC.Row;
with GNU.DB.ADBC.Driver;                       use GNU.DB.ADBC.Driver;

package body GNU.DB.ADBC.Resultset is

   Version : constant String :=
      "$Id: gnu-db-adbc-resultset.adb,v 1.6 2002/03/04 20:02:17 merdmann Exp $";

   subtype ST_Index is Resultset_ID range Null_ID+1..Resultset_ID'Last;

   ST : array( ST_Index ) of Handle := (others => Null_Handle);

   ---====================================================================---
   ---===                O B J  E C T     D A T A                      ===---
   ---===                                                              ===---
   ---=== This section contains all declarations of data structures    ===---
   ---=== to implement one instance of the connection object           ===---
   ---===                                                              ===---
   ---====================================================================---
   type Row_List_Element;
   type Row_List_Element_Access is access Row_List_Element;

   type Row_List_Element is record
         Data     : Row.Handle              := null;
         Next     : Row_List_Element_Access := null;
         Previous : Row_List_Element_Access := null;
      end record;

   type Attribute_Array is
      array( Positive range 1..Max_Number_Of_Attributes ) of Unbounded_String;

   procedure Free is
      new Unchecked_Deallocation( Row_List_Element, Row_List_Element_Access);

   -----------------
   -- Object_Data --
   -----------------
   type Object_Data is record
         Id               : Resultset_ID    := Null_ID;
         Nbr_Of_Records   : Natural         := 0;

         Mode             : Fetch_Mode_Type := Next;
         Read_Position    : Natural         := 0;

         Attribute_Next   : Positive        := Attribute_Array'First;
         Attribute_Name   : Attribute_Array := (others => Null_Unbounded_String);

         Result_Data      : Row_List_Element_Access := null;
         Result_Last      : Row_List_Element_Access := null;
         Next_Read        : Row_List_Element_Access := null;
      end record;

   ---=====================================================================---
   ---===         L O C A L   S U P P O R T   P R O C E D U R E S       ===---
   ---===                                                               ===---
   ---=====================================================================---

   ---=====================================================================---
   ---===             C O M P O  N E N T    I N T E R F A C E           ===---
   ---=====================================================================---

   ---=====================================================================---
   ---===           A T T R I B U T E    F U N C T I O N S              ===---
   ---=====================================================================---

   ----------
   -- Mode --
   ----------
   procedure Mode(
      This  : in Resultset_ID;
      Value : in Fetch_Mode_Type ) is
      Data  : Object_Data_Access renames  ST(This).Data;
   begin
      Data.Mode := Value;
   end Mode;

   ----------
   -- Mode --
   ----------
   function Mode(
      This  : in Resultset_ID ) return Fetch_Mode_Type is
      Data  : Object_Data_Access renames  ST(This).Data;
   begin
      return Data.Mode;
   end Mode;

   ----------------------
   -- Number_Of_Record --
   ----------------------
   procedure Number_Of_Records(
      This  : in Resultset_ID;
      Value : in Natural ) is
      Data  : Object_Data_Access renames  ST(This).Data;
   begin
      Data.Nbr_Of_Records := Value;
   end Number_Of_Records;

   ----------------------
   -- Number_Of_Record --
   ----------------------
   function Number_Of_Records(
      This  : in Resultset_ID ) return Natural is
      Data  : Object_Data_Access renames  ST(This).Data;
   begin
      return Data.Nbr_Of_Records;
   end Number_Of_Records;

   --------
   -- ID --
   --------
   function ID(
      This : in Object'Class ) return Resultset_ID is
   begin
      return This.Data.Id;
   end ID;

   ---------------
   -- Attribute --
   ---------------
   procedure Attribute(
      This  : in Resultset_ID;
      Name  : in String ) is
      Data  : Object_Data_Access renames  ST(This).Data;
      Next  : Positive renames Data.Attribute_Next;
   begin
      if Next in Data.Attribute_Name'Range then
         Data.Attribute_Name( Next ) := To_Unbounded_String(Name);
         Next := Next + 1;
      else
         raise Attribute_Overflow;
      end if;
   end Attribute;

   ---------------
   -- Attribute --
   ---------------
   function Attribute(
      This  : in Resultset_ID;
      Nbr   : in Positive ) return String is
      Data  : Object_Data_Access renames  ST(This).Data;
   begin
      if Nbr in Data.Attribute_Name'Range then
         return To_String( Data.Attribute_Name( Nbr ) );
      else
         raise Attribute_Not_Existing;
      end if;
   end Attribute;

   ------------------------
   -- Attrbiute_Position --
   ------------------------
   function Attribute_Position(
      This  : in Resultset_ID;
      Name  : in String ) return Natural is
      Data  : Object_Data_Access renames  ST(This).Data;
   begin
      for I in 1..Data.Attribute_Next loop
         if To_String(Data.Attribute_Name(I)) = Name then
            return I;
         end if;
      end loop;

      raise Attribute_Not_Existing;
   end Attribute_Position;

   -------------------
   -- Driver_Handle --
   -------------------
   function Driver_Handle(
      This  : in Resultset_ID ) return Driver.Handle is
   begin
      return Driver_Handle( ST(This).Stmt );
   end Driver_Handle;

   ---=====================================================================---
   ---===                        M E T H O D S                          ===---
   ---=====================================================================---
   --------------
   -- Allocate --
   --------------
   function Allocate(
      Set : in Handle ) return Resultset_ID is
      -- store a handle in the statement table. This will be called by the
      -- driver to insert.
   begin
      for I in ST'Range loop
         if ST(I) = Null_Handle then
            ST(I) := Set;
            if Set.Data = null then
               Set.Data := new Object_Data;
               Set.Data.Id := I;
               Set.Data.Result_Data := null;
               Set.Data.Result_Last := null;
               Set.Data.Next_Read   := null;
            end if;

            Add_Resultset( Set.Stmt, I );
            return I;
         end if;
      end loop;

      raise Resultset_Table_Overflow;
   end Allocate;

   ----------------
   -- Deallocate --
   ----------------
   procedure Deallocate(
      Id  : in Resultset_ID ) is
      -- deallocate the id from the resulset table
      procedure Free is
            new Unchecked_Deallocation( Object_Data, Object_Data_Access);

      Data : Object_Data_Access ;
      P    : Row_List_Element_Access;
   begin
      pragma Assert( Id in ST'Range );

      if ST(Id) /= null then
         Data := ST(Id).Data;
         if Data /= null then
            P := Data.Result_Data;
            while P /= null loop
               declare
                  Q : Row_List_Element_Access := P;
               begin
                  P := P.Next;
                  Free(Q);
               end;
            end loop;

            Statement.Delete_Resultset( ST(Id).Stmt, Id );
            if Driver_Handle( St(Id).Stmt ) /= null then
               Delete_Resultset( Driver_Handle( ST(Id).Stmt).all, Id );
            end if;
            ST(Id) := null;
            Free( Data );
         end if;
      end if;
   end Deallocate;

   ----------------
   -- Get_Handle --
   ----------------
   function Get_Handle(
      Id  : in Resultset_ID ) return Handle is
      -- return the handle for a given resultset_ID;
   begin
      return ST(Id);
   end Get_Handle;

   -----------
   -- Fetch --
   -----------
   procedure Fetch(
      This    : in Resultset_ID;
      Mode    : in Fetch_Mode_Type := Next ) is
      --
      -- fetch next element in result set. If there is no next
      -- element in the rwo cache load the entry from the data base.
      --
      -- R.1 - If a new fetchmode is give store it.
      -- R.2 - depending on the current fetchmode select the first, the
      --       next, previous or last position.
      -- R.3 - If no records have been loaded we assume a first time
      --       read and the head of the list is set
      -- R.4 - if the current resport has no next record, we assume
      --       that a record has to be fetched in sequence.
      -- R.5 - if the record has already been loaded take it from the
      --       memory.
      --
      Data    : Object_Data_Access renames  ST(This).Data;
      Readpos : Row_List_Element_Access renames Data.Next_Read ;

      function Fetch_Record(
         Mode    : in Fetch_Mode_Type ) return Row_List_Element_Access is
         P       : Row_List_Element_Access := new Row_List_Element;
         Current : Row.Handle              := null;
      begin
         pragma Debug( Put_Line("Fetching, mode =" & Fetch_Mode_Type'Image(Mode)));
         Fetch( ST(This).all, Current, Mode );
         Retrive_Host_Variables(ST(This).Stmt, Current );

         P.Data := Current;
         P.Next := null;
         P.Previous := null;

         return P;
      end Fetch_Record;

   begin
      if Mode /= Next and then Mode /= Data.Mode then            -- R.1
         Data.Mode := Mode;
      end if;

      case Data.Mode is                                          -- R.2
         when First =>
            if Data.Result_Data = null then                      -- R.3
               declare
                  P : Row_List_Element_Access := Fetch_Record(First);
               begin
                  Readpos          := P;
                  Data.Result_Last := P;
               end ;
            else
               Readpos := Data.Result_Data;
            end if;

         when Next =>
            if End_Of_Result( This ) then
               raise End_Of_Resultset_Exception;
            end if;

            if Data.Result_Data = null then                      -- R.3
               declare
                  P : Row_List_Element_Access := Fetch_Record(First);
               begin
                  Data.Result_Last := P;
                  Data.Result_Data := P;
                  P.Next           := null;
                  Readpos          := P;
               end ;
            elsif Readpos.Next = null then                      -- R.4
               declare
                  P : Row_List_Element_Access := Fetch_Record(Next);
               begin
                  -- link into the list of loaded rows
                  P.Previous   := Readpos;
                  Readpos.Next := P;

                  Data.Result_Last := P;
                  Readpos          := P;
               end ;
            else                                                -- R.5
               Readpos := Readpos.Next ;
            end if;
            Data.Read_Position := Data.Read_Position + 1;

         when Previous =>
            if Begin_Of_Result( This ) then
               raise Begin_Of_Resultset_Exception;
            end if;
            Readpos := Readpos.Previous;
            Data.Read_Position := Data.Read_Position - 1;
         when Last =>
            Readpos := Data.Result_Last;
            Data.Read_Position := Data.Nbr_Of_Records;
      end case;
   end Fetch;

   -------------------
   -- End_Of_Result --
   -------------------
   function End_Of_Result(
      This : in Resultset_ID ) return Boolean is
      Data : Object_Data_Access renames  ST(This).Data;
   begin
      return Data.Read_Position = Data.Nbr_Of_Records;
   end End_Of_Result;

   ---------------------
   -- Begin_Of_Result --
   ---------------------
   function Begin_Of_Result(
      This : in Resultset_ID ) return Boolean is
      Data : Object_Data_Access renames  ST(This).Data;
   begin
      return Data.Read_Position = 1;
   end Begin_Of_Result;

   ------------
   -- Domain --
   ------------
   package body String_Domain is

      Domain_Number     : constant Natural := Attribute_Position(Result,Name);
      Is_Null_Indicator : Boolean := False;
      -----------
      -- Value --
      -----------
      function Value(
         Trim   : in Boolean := False ) return String is
         -- retrieve the value of the specified domain by dispatching
         -- into the GNU.DB.ADBC.Row package.
         Length : Natural ;
         V      : String( 1..Size ) := (others =>' ');
         Row    : ADBC.Row.Handle renames  ST(Result).Data.Next_Read.Data;
      begin
         Get( Row.all, Domain_Number, V, Length);
         if Length > 0 then
            Is_Null_Indicator := False;
         else
            Is_Null_Indicator := True;
         end if;
         return V;
      end Value;

      ---------
      -- Set --
      ---------
      procedure Set( Value : in String ) is
      begin
         null;
      end Set;

      -------------
      -- Is_Null --
      -------------
      function Is_Null return Boolean is
      begin
         return Is_Null_Indicator;
      end Is_Null;

   end String_Domain;

   --------------------
   -- Numeric_Domain --
   --------------------
   package body Integer_Domain is

      Domain_Number     : constant Natural := Attribute_Position(Result, Name);
      Is_Null_Indicator : Boolean := False;
      -----------
      -- Value --
      -----------
      function Value return Number_Type  is
         -- retrieve the value of the specified domain
         Row   : ADBC.Row.Handle renames  ST(Result).Data.Next_Read.Data;
         Value : Integer := Get( Row.all, Domain_Number );
      begin
         return Number_Type( Value );
      end Value;

      ---------
      -- Set --
      ---------
      procedure Set( Value : in Number_Type ) is
      begin
         null;
      end Set;

      -------------
      -- Is_Null --
      -------------
      function Is_Null return Boolean is
      begin
         return Is_Null_Indicator;
      end Is_Null;

   end Integer_Domain;

end GNU.DB.ADBC.Resultset;

