------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2006 Maxim Reznik                                         --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

with OCI.Thick.Statements;
with Ada.Strings.Unbounded;

package OCI.Thick.Notifications.DB_Change is

   type Change_Kinds is
     (None, Startup, Shutdown, Shutdown_Any,
      Drop_Db, Deregistration, Object_Change);

   type Operation_Kinds is mod 256;

   All_Rows : constant Operation_Kinds;
   Insert   : constant Operation_Kinds;
   Update   : constant Operation_Kinds;
   Delete   : constant Operation_Kinds;
   Alter    : constant Operation_Kinds;
   Drop     : constant Operation_Kinds;

   type Row is record
      Id   : Ada.Strings.Unbounded.Unbounded_String;
      Kind : Operation_Kinds;
   end record;

   type Row_Array is array (Positive range <>) of Row;

   type Change_Descriptor is limited private;

   function Change_Kind (Object : Change_Descriptor) return Change_Kinds;
   function Table_Count (Object : Change_Descriptor) return Natural;

   function Table_Name
     (Object : Change_Descriptor;
      Index  : Positive) return String;

   function Table_Operation
     (Object : Change_Descriptor;
      Index  : Positive) return Operation_Kinds;

   function Table_Rows
     (Object : Change_Descriptor;
      Index  : Positive) return Row_Array;

   type Change_Subscription is new Base_Subscription with private;

   procedure Create
     (Item        : in out Change_Subscription;
      Need_Rowids : in Boolean := False);

   procedure Associate
     (Item        : in out Change_Subscription;
      Stmt        : in out Statements.Statement);
   --  Associate also executes the statement.

   procedure Notify
     (Item   : in out Change_Subscription;
      Data   : in     Change_Descriptor);

private
   All_Rows : constant Operation_Kinds := OCI.Lib.OCI_OPCODE_ALLROWS;
   Insert   : constant Operation_Kinds := OCI.Lib.OCI_OPCODE_INSERT;
   Update   : constant Operation_Kinds := OCI.Lib.OCI_OPCODE_UPDATE;
   Delete   : constant Operation_Kinds := OCI.Lib.OCI_OPCODE_DELETE;
   Alter    : constant Operation_Kinds := OCI.Lib.OCI_OPCODE_ALTER;
   Drop     : constant Operation_Kinds := OCI.Lib.OCI_OPCODE_DROP;

   type Change_Descriptor is limited record
      Descriptor : OCI.Lib.DVoid;
   end record;

   type Change_Subscription is new Base_Subscription with null record;

   procedure Internal_Notify
     (Item       : in out Change_Subscription;
      Descriptor : in     OCI.Lib.DVoid;
      Buffer     : in     Buffer_Type);

end OCI.Thick.Notifications.DB_Change;
