/*
 *     gtkatlantic - the gtk+ monopd client, enjoy network monopoly games
 *
 *
 *  Copyright © 2002-2015 Sylvain Rochet
 *
 *  gtkatlantic is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; see the file COPYING. If not, see
 *  <http://www.gnu.org/licenses/>.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#include <png.h>
#include "readpng.h"


gint8 readpng_read(gchar *path, _png_imagedata *png_imagedata) {
	FILE *infile;
	gint8 sig[8];
	png_uint_32 width, height;
	gint32 bit_depth;
	gint32 color_type;
	png_byte channels;
	png_structp structp = NULL;
	png_infop infop = NULL;
	gint32 x, y;
	png_bytepp row_pointers;
	guint8 *tmp, *ptr_in, *ptr_out;
	gint32 rowbytes;
	gint8 ret = 0;

	if ((infile = fopen(path, "rb")) == NULL) {
		fprintf(stderr, "can't open file: %s\n", path);
		ret = -1;
		goto free_and_return;
	}

	memset(sig, 0, sizeof(sig));
	if (fread(sig, 1, sizeof(sig), infile) != sizeof(sig)) {
		fprintf(stderr, "failed to read file: %s\n", path);
		ret = -1;
		goto free_and_return;
	}
	if (png_sig_cmp((png_bytep)sig, 0, sizeof(sig))) {
		fprintf(stderr, "not a valid PNG file: %s\n", path);
		ret = -1;
		goto free_and_return;
	}

	structp = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if (!structp) {
		/* very unlikely */
		ret = -1;
		goto free_and_return;
	}

	infop = png_create_info_struct(structp);
	if (!infop) {
		/* very unlikely */
		ret = -1;
		goto free_and_return;
	}

	if (setjmp(png_jmpbuf(structp))) {
		/* very unlikely */
		ret = -1;
		goto free_and_return;
	}

	png_init_io(structp, infile);
	png_set_sig_bytes(structp, sizeof(sig));

	png_read_info(structp, infop);

	png_get_IHDR(structp, infop, &width, &height, &bit_depth, &color_type, NULL, NULL, NULL);

	if (color_type == PNG_COLOR_TYPE_PALETTE) {
		png_set_expand(structp);
	}

	if (color_type == PNG_COLOR_TYPE_GRAY  &&  bit_depth < 8) {
		png_set_expand(structp);
	}

	if (png_get_valid(structp, infop, PNG_INFO_tRNS)) {
		png_set_expand(structp);
	}

	if (bit_depth == 16) {
		png_set_strip_16(structp);
	}

	if (color_type == PNG_COLOR_TYPE_GRAY  ||  color_type == PNG_COLOR_TYPE_GRAY_ALPHA) {
		png_set_gray_to_rgb(structp);
	}

	png_read_update_info(structp, infop);

	channels = png_get_channels(structp, infop);
	if (channels != 3  &&  channels != 4) {
		/* very unlikely */
		ret = -1;
		goto free_and_return;
	}

	memset(png_imagedata, 0, sizeof(_png_imagedata));
	png_imagedata->width = width;
	png_imagedata->height = height;

	rowbytes = png_get_rowbytes(structp, infop);

	row_pointers = (png_bytepp)g_malloc(height * sizeof(png_bytep));
	tmp = g_malloc(height * rowbytes);
	for (y = 0; y < (gint32)height; y++) {
		row_pointers[y] = (png_bytep)(tmp + y * rowbytes);
	}

	png_read_image(structp, row_pointers);
	png_read_end(structp, NULL);

	png_imagedata->rgba = ptr_out = g_malloc(width * height * 4);
	if (channels == 4) {
		for (y = 0; y < (gint32)height; y++) {
			memcpy(ptr_out, row_pointers[y], width*4);
			ptr_out += width*4;
		}
	}
	else /* if (channels == 3) */ {
		for (y = 0; y < (gint32)height; y++) {
			ptr_in = row_pointers[y];
			for (x = 0; x < (gint32)width; x++) {
				memcpy(ptr_out, ptr_in, 3);
				ptr_in += 3;
				ptr_out += 3;
				*ptr_out++ = 0xff; /* alpha */
			}
		}
	}

	g_free(tmp);
	g_free(row_pointers);

free_and_return:
	if (infile) {
		fclose(infile);
	}

	if (structp && infop) {
		png_destroy_read_struct(&structp, &infop, NULL);
	} else if(structp) {
		png_destroy_read_struct(&structp, NULL, NULL);
	}

	return ret;
}


gint32 readpng_get_width(_png_imagedata *png_imagedata) {
	return png_imagedata->width;
}


gint32 readpng_get_height(_png_imagedata *png_imagedata) {
	return png_imagedata->height;
}


guint8 *readpng_get_rgba(_png_imagedata *png_imagedata)  {

	if (png_imagedata->rgba != NULL) {
		png_imagedata->rgba_needed = TRUE;
		return png_imagedata->rgba;
	}
	return NULL;
}

guint8 *readpng_get_rgba_crop(_png_imagedata *png_imagedata, gint32 x, gint32 y, gint32 width, gint32 height)  {
	gint32 i;
	guint8 *in, *rout, *out;

	if(x + width  > png_imagedata->width)   return NULL;
	if(y + height > png_imagedata->height)  return NULL;

	in = png_imagedata->rgba + (y * png_imagedata->width + x) * 4,
	rout = out = g_malloc0(width * height * 4);

	for (i = 0 ; i < height ; i++) {
		memcpy(out, in, width * 4);
		out += width * 4;
		in += png_imagedata->width * 4;
	}

	return rout;
}


void readpng_cleanup(_png_imagedata *png_imagedata)  {

	if(png_imagedata->rgba != NULL  &&  !png_imagedata->rgba_needed) {
		g_free(png_imagedata->rgba);
		png_imagedata->rgba = NULL;
	}
}
