package tests::GroupFieldTest;

use strict;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::GroupField;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();
    $self->init_i18n();

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_locale();
    $self->set_up_test_schema();

    my $spec = new Lire::ReportSpec();
    $spec->superservice( 'test' );

    $self->{'i18n_groupfield'} =
      new Lire::GroupField( 'name' => 'file',
                            'report_spec' => $spec,
                            'i18n_domain' => 'lire-test',
                            'label' => 'JPEG Files' );

    return $self;
}

sub tear_down {
    my $self = shift->SUPER::tear_down();

    $self->tear_down_locale();

    return $self;
}

sub test_new {
    my $self = $_[0];

    my $spec = new Lire::ReportSpec();
    $spec->superservice( 'test' );
    my $group_field = new Lire::GroupField( 'name' => 'file',
                                            'report_spec' => $spec,
                                            'label' => 'JPEG Files' );

    $self->assert_not_null( $group_field, "new() returned undef" );
    $self->assert( UNIVERSAL::isa( $group_field, 'Lire::GroupField' ),
                   'new() returned something other than a Lire::GroupField instance' );

    $self->assert_died( sub { new Lire::GroupField() },
                        qr/missing \'name\' parameter/,
                      );
    $self->assert_died( sub { new Lire::GroupField( 'name' => 'tata toto' ) },
                        qr/invalid \'name\' parameter: /,
                      );
    $self->assert_died( sub { new Lire::GroupField( 'name' => 'file') },
                        qr/missing \'report_spec\' parameter/,
                      );
    $self->assert_died( sub { new Lire::GroupField( 'name' => 'file',
                                                    'report_spec' => {} ) },
                        qr/\'report_spec\' parameter should be a \'Lire::ReportSpec\' instance, not \'HASH/,
                      );
    $self->assert_died( sub { new Lire::GroupField( 'name' => 'wawa',
                                                    'report_spec' => $spec,
                                                    'label' => 'wawalabel'
                                                  ) },
                        qr/'wawa' field name is not defined in the specification's schemas/,
                      );

    $self->assert_died( sub { new Lire::GroupField( 'name' => 'file_size',
                                                    'report_spec' => $spec,
                                                    'label' => 'wawalabel'
                                                  ) },
                        qr/'file_size' field is of type 'bytes'. Use Lire::Rangegroup for this type/,
                      );
    $self->assert_died( sub { new Lire::GroupField( 'name' => 'time_start',
                                                    'report_spec' => $spec,
                                                    'label' => 'wawalabel'
                                                  ) },
                        qr/'time_start' field is of type 'timestamp'. Use Lire::Timeslot or Lire::Timegroup for this type/,
                      );

    $self->assert_str_equals( 'file', $group_field->{'name'} );
    $self->assert_str_equals( 'JPEG Files', $group_field->{'label'} );
    $self->assert_str_equals( 'lire', $group_field->{'i18n_domain'} );
}

sub test_Lire::GroupField::label_i18n {
    my $self = $_[0];

    my $group_field = $self->{'i18n_groupfield'};
    $self->assert_str_equals( 'JPEG Files', $group_field->label() );
    $self->set_locale( 'fr_CA.iso8859-1' );
    $self->assert_str_equals( 'Fichiers JPEG', $group_field->label() );
    $group_field->{'label'} = 'Downloads';
    $self->assert_str_equals( 'Tlchargements', $group_field->label() );
}

sub test_label_nolabel_i18n {
    my $self = $_[0];

    my $group_field = $self->{'i18n_groupfield'};
    delete $group_field->{'label'};
    $self->assert_str_equals( 'File', $group_field->label() );
    $self->set_locale( 'fr_CA.iso8859-1' );
    $self->assert_str_equals( 'Fichier', $group_field->label() );
}

1;
