<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:exslt="http://exslt.org/common" 
                xmlns:func="http://exslt.org/functions"
                xmlns:debug="http://wohlberg.net/xml/debug"
                extension-element-prefixes="exslt func"
                version="1.0">

<!-- 
     This file contains templates for constructing output for use in
     debugging the defaults expansion code. 

     Copyright © 2005-2007 Brendt Wohlberg <photoml@wohlberg.net>

     This is free software; you can redistribute it and/or modify it 
     under the terms of version 2 of the GNU General Public License 
     at http://www.gnu.org/licenses/gpl-2.0.txt.

     This software is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
     GNU General Public License for more details.
-->



<!-- *****************************************************************
     Extension function calling template with name debug-node-set.
     ***************************************************************** -->
  <func:function name="debug:debug-node-set">
    <xsl:param name="nodes" select="/.."/>
    <xsl:param name="label" select="''"/>
    <xsl:param name="txtdf" select="false()"/>
    <xsl:param name="indnt" select="'  '"/>

    <xsl:variable name="result">
      <xsl:call-template name="debug-node-set">
        <xsl:with-param name="nodes" select="$nodes"/>
        <xsl:with-param name="label" select="$label"/>
        <xsl:with-param name="txtdf" select="$txtdf"/>
        <xsl:with-param name="indnt" select="$indnt"/>
      </xsl:call-template>
    </xsl:variable>

    <func:result select="$result"/>
  </func:function>



<!-- *****************************************************************

     Construct a representation of a node set for use in debugging. 
     The parameters are 'nodes', which is the node set for which the
     debugging representation is desired, 'label', which is an initial
     label string written before the node set representation, 'txtdf',
     which is a flag indicating whether all text nodes should be
     included or only text nodes which contain non-whitespace content,
     and 'indnt', which is a string containing the number of spaces
     for the initial indentation

     Note: When using this template to display the content of a result
     tree fragment converted into a node set by using exslt:node-set,
     parameter "nodes" should have a value of "exslt:node-set($var)/*", 
     as opposed to "$var" when the variable is itself a node set.
     ***************************************************************** -->
  <xsl:template name="debug-node-set">
    <xsl:param name="nodes" select="/.."/>
    <xsl:param name="label" select="''"/>
    <xsl:param name="txtdf" select="false()"/>
    <xsl:param name="indnt" select="'  '"/>

    <xsl:if test="$label != ''">
      <xsl:value-of select="$label"/>
      <xsl:value-of select="$indnt"/>
      <xsl:text>[Node set contains </xsl:text>
      <xsl:value-of select="count($nodes)"/>
      <xsl:choose>
        <xsl:when test="count($nodes) = 1">
          <xsl:text> node</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text> nodes</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:text>, including </xsl:text>
      <xsl:value-of select="count($nodes[self::*])"/>
      <xsl:choose>
        <xsl:when test="count($nodes[self::*]) = 1">
          <xsl:text> element</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text> elements</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:text>]&#10;</xsl:text>
    </xsl:if>
    <xsl:apply-templates select="$nodes[self::*]|$nodes[self::text()]|
                                 $nodes[self::comment()]|
                                 $nodes[self::processing-instruction()]"
                         mode="debug-node-set-mode">
      <xsl:with-param name="indnt" select="concat($indnt,'  ')"/>
      <xsl:with-param name="txtdf" select="$txtdf"/>
    </xsl:apply-templates>
  </xsl:template>



<!-- *****************************************************************
     Construct the node set debugging representation of an element.
     ***************************************************************** -->
  <xsl:template match="*" mode="debug-node-set-mode">
    <xsl:param name="txtdf" select="false()"/>
    <xsl:param name="indnt" select="'  '"/>

    <xsl:value-of select="$indnt"/>
    <xsl:value-of select="name(.)"/>
    <xsl:apply-templates select="@*" mode="debug-node-set-mode"/>
    <xsl:text>&#10;</xsl:text>
    <xsl:apply-templates select="*|text()|comment()|processing-instruction()"
                         mode="debug-node-set-mode">
      <xsl:with-param name="indnt" select="concat($indnt,'  ')"/>
      <xsl:with-param name="txtdf" select="$txtdf"/>
    </xsl:apply-templates>
  </xsl:template>



<!-- *****************************************************************
     Construct the node set debugging representation of an attribute.
     ***************************************************************** -->
  <xsl:template match="@*" mode="debug-node-set-mode">
    <xsl:param name="indnt" select="'  '"/>

    <xsl:text> [</xsl:text>
    <xsl:value-of select="name(.)"/>
    <xsl:text>=</xsl:text>
    <xsl:value-of select="."/>
    <xsl:text>]</xsl:text>
  </xsl:template>



<!-- *****************************************************************
     Construct the node set debugging representation of a text node.
     ***************************************************************** -->
  <xsl:template match="text()" mode="debug-node-set-mode">
    <xsl:param name="txtdf" select="false()"/>
    <xsl:param name="indnt" select="'  '"/>
 
    <xsl:if test="boolean($txtdf) or translate(normalize-space(.),' ','')!=''">
      <xsl:value-of select="$indnt"/>
      <xsl:text>[text: </xsl:text>
      <xsl:value-of select="translate(substring(.,1,20),'&#10;',' ')"/>
      <xsl:text>]&#10;</xsl:text>
    </xsl:if>
  </xsl:template>



<!-- *****************************************************************
     Construct the node set debugging representation of a comment.
     ***************************************************************** -->
  <xsl:template match="comment()" mode="debug-node-set-mode">
    <xsl:param name="indnt" select="'  '"/>

    <xsl:value-of select="$indnt"/>
    <xsl:text>[comment: </xsl:text>
    <xsl:value-of select="translate(substring(.,1,20),'&#10;',' ')"/>
    <xsl:text>]&#10;</xsl:text>
  </xsl:template>



<!-- *****************************************************************
     Construct the node set debugging representation of a processing 
     instruction.
     ***************************************************************** -->
  <xsl:template match="processing-instruction()" mode="debug-node-set-mode">
    <xsl:param name="indnt" select="'  '"/>

    <xsl:value-of select="$indnt"/>
    <xsl:text>[pi: </xsl:text>
    <xsl:value-of select="name(.)"/>
    <xsl:text> </xsl:text>
    <xsl:value-of select="translate(substring(.,1,20),'&#10;',' ')"/>
    <xsl:text>]&#10;</xsl:text>
  </xsl:template>
  

</xsl:stylesheet>
