<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                version="1.0">

<!-- 
     This file contains templates for constructing an HTML index of
     roll, sheet, digital, and digimage elements using the summary
     details extracted by preindex.xsl.

     Copyright © 2005,2011 Brendt Wohlberg <photoml@wohlberg.net>

     This is free software; you can redistribute it and/or modify it 
     under the terms of version 2 of the GNU General Public License 
     at http://www.gnu.org/licenses/gpl-2.0.txt.

     This software is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
     GNU General Public License for more details.
-->

  <xsl:output method="xml" indent="yes" encoding="utf-8"
    doctype-public="-//W3C//DTD XHTML 1.0 Strict//EN"
    doctype-system="http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd"/>

  <!-- Stylesheet parameter determining the primary font size -->
  <xsl:param name="font-size">11</xsl:param>
   <!-- Stylesheet parameter determining whether to group by year -->
   <xsl:param name="year-group" select="false"/>

  <!-- Key matching all sheet elements with specific year of start date -->
  <xsl:key name="sheetyear" match="sheet" use="substring(startdate,1,4)"/>

  <!-- Key matching all roll elements with specific year of start date -->
  <xsl:key name="rollyear" match="roll" use="substring(startdate,1,4)"/>

   <!-- Key matching all digital elements with specific year of start date -->
  <xsl:key name="digyear" match="digital" use="substring(startdate,1,4)"/>

  <!-- Key matching all digimage elements with specific group-id -->
  <xsl:key name="digigid" match="digimage" use="@group-id"/>


  <!-- Template matching document root. -->
  <xsl:template match="/">
    <html>
      <head>
        <title>Photo Metadata Index</title>
        <style type="text/css">
          body  {
          margin-left: 3%;
          margin-right: 3%;
          font-family: sans-serif;
          font-size: <xsl:value-of select="$font-size"/>pt;
          color: black;
          background: white;
          }
          table { 
          margin-left: auto;
          margin-right: auto;
          border: solid black 1px;
          border-spacing: 5px;
          border-collapse: collapse;
          }
          tr, td {
          background-color: rgb(245,245,245);
          border: solid black 1px;
          padding: 4px;
          }
          th { 
          background-color: rgb(230,230,230);
          border: solid black 1px;
          padding: 4px;
          }
          a {
          text-decoration: none;
          color: #2e5c89;
          } 
          a:hover {
          color: #247cd2;
          }
          .year {
          background-color: rgb(237,237,237);
          }
          .center {
          text-align : center;
          }
          .title {
          text-align: center;
          font-size: 170%;
          padding-bottom: 0.4em;
          }
          .spacer {
          padding-bottom: 3em;
          }
        </style>
      </head>
      <body>
        <xsl:if test="//sheet">
          <xsl:call-template name="sheet"/>
          <div class="spacer"/>
        </xsl:if>
        <xsl:if test="//roll">
          <xsl:call-template name="roll"/>
          <div class="spacer"/>
        </xsl:if>
        <xsl:if test="//digital">
          <xsl:call-template name="digital"/>
          <div class="spacer"/>
        </xsl:if>
        <xsl:if test="//digimage">
          <xsl:call-template name="digimage"/>
        </xsl:if>
      </body>
    </html>
  </xsl:template>


  <!-- Write index of sheet elements. -->
  <xsl:template name="sheet">
    <div class="title">Sheet Index</div>
    <table>
      <tr>
        <th>ID</th>
        <th>Date</th>
        <th>Film</th>
        <th>Camera</th>
      </tr>
      <xsl:choose>
        <xsl:when test="boolean($year-group)">
          <xsl:for-each select="//sheet[generate-id(.) =
                                    generate-id(key('sheetyear',
                                    substring(startdate,1,4))[1])]">
            <xsl:sort select="substring(startdate,1,4)"/>
            <tr>
              <td colspan="4" align="center" class="year">
                <b><xsl:value-of select="substring(startdate,1,4)"/></b>
              </td>
            </tr>
            <xsl:for-each select="key('sheetyear', substring(startdate,1,4))">
              <xsl:sort select="@id"/>
              <xsl:call-template name="sheet-row"/>
            </xsl:for-each>
          </xsl:for-each>
        </xsl:when>
        <xsl:otherwise>
          <xsl:for-each select="//sheet">
            <xsl:sort select="@id"/>
            <xsl:call-template name="sheet-row"/>
          </xsl:for-each>
        </xsl:otherwise>
      </xsl:choose>
    </table>
  </xsl:template>
   

   <!-- Write a single sheet table row. -->
  <xsl:template name="sheet-row">
    <tr>
      <td>
        <a href="{concat(parent::file/@html,@aid)}">
          <xsl:value-of select="@id"/>
        </a>
      </td>
      <td><xsl:value-of select="startdate"/></td>
      <td><xsl:apply-templates select="film"/></td>
      <td><xsl:apply-templates select="body"/></td>
    </tr>
  </xsl:template>


  <!-- Write index of roll elements. -->
  <xsl:template name="roll">
    <div class="title">Roll Index</div>
    <table>
      <tr>
        <th>ID</th>
        <th>Start</th>
        <th>End</th>
        <th>Film</th>
        <th>Camera</th>
      </tr>
      <xsl:choose>
        <xsl:when test="boolean($year-group)">
          <xsl:for-each select="//roll[generate-id(.) =
                                    generate-id(key('rollyear',
                                    substring(startdate,1,4))[1])]">
            <xsl:sort select="substring(startdate,1,4)"/>
            <tr>
              <td colspan="5" align="center" class="year">
                <b><xsl:value-of select="substring(startdate,1,4)"/></b>
              </td>
            </tr>
            <xsl:for-each select="key('rollyear', substring(startdate,1,4))">
              <xsl:sort select="@id"/>
              <xsl:call-template name="roll-row"/>
            </xsl:for-each>
          </xsl:for-each>
        </xsl:when>
        <xsl:otherwise>
          <xsl:for-each select="//roll">
            <xsl:sort select="@id"/>
            <xsl:call-template name="roll-row"/>
          </xsl:for-each>
        </xsl:otherwise>
      </xsl:choose>
    </table>
  </xsl:template>


  <!-- Write a single roll table row. -->
  <xsl:template name="roll-row">
    <tr>
      <td>
        <a href="{concat(parent::file/@html,@aid)}">
          <xsl:value-of select="@id"/>
        </a>
      </td>
      <td><xsl:value-of select="startdate"/></td>
      <td><xsl:value-of select="enddate"/></td>
      <td><xsl:apply-templates select="film"/></td>
      <td><xsl:apply-templates select="body"/></td>
    </tr>
  </xsl:template>
  

  <!-- Handle film child of sheet or roll elements. -->
  <xsl:template match="film">
    <xsl:value-of select="make"/>
    <xsl:text> </xsl:text>
    <xsl:value-of select="name"/>
    <xsl:if test="speed">
      (ISO <xsl:value-of select="speed"/>)
    </xsl:if>
  </xsl:template>


  <!-- Handle camera body description. -->
  <xsl:template match="body">
    <xsl:value-of select="concat(make,' ',model)"/>
  </xsl:template>


  <!-- Write index of digital elements. -->
  <xsl:template name="digital">
    <div class="title">Digital Index</div>
    <table>
      <tr>
        <th>ID</th>
        <th>Start</th>
        <th>End</th>
        <th>Camera</th>
      </tr>
      <xsl:choose>
        <xsl:when test="boolean($year-group)">
          <xsl:for-each select="//digital[generate-id(.) =
                                generate-id(key('digyear',
                                substring(startdate,1,4))[1])]">
            <xsl:sort select="substring(startdate,1,4)"/>
            <tr>
              <td colspan="4" align="center" class="year">
                <b><xsl:value-of select="substring(startdate,1,4)"/></b>
              </td>
            </tr>
            <xsl:for-each select="key('digyear', substring(startdate,1,4))">
              <xsl:sort select="@id"/>
              <xsl:call-template name="digital-row"/>
            </xsl:for-each>
          </xsl:for-each>
        </xsl:when>
        <xsl:otherwise>
          <xsl:for-each select="//digital">
            <xsl:sort select="@id"/>
            <xsl:call-template name="digital-row"/>
          </xsl:for-each>
        </xsl:otherwise>
      </xsl:choose>
    </table>
  </xsl:template>


   <!-- Write a single digital table row. -->
  <xsl:template name="digital-row">
    <tr>
      <td>
        <a href="{concat(parent::file/@html,@aid)}">
          <xsl:value-of select="@id"/>
        </a>
      </td>
      <td><xsl:value-of select="startdate"/></td>
      <td><xsl:value-of select="enddate"/></td>
      <td><xsl:apply-templates select="body"/></td>
    </tr>
  </xsl:template>


  <!-- Write index of digimage elements. -->
  <xsl:template name="digimage">
    <div class="title">Digimage Index</div>
    <table>
      <tr>
        <th>ID</th>
        <th>Frames</th>
      </tr>
      <!-- Iterate over each unique digimage group-id -->
      <xsl:for-each select="//digimage[generate-id(.)=generate-id(key('digigid',@group-id)[1])]">
        <xsl:sort select="@group-id"/>
        <xsl:variable name="gid" select="@group-id"/>
        <tr>
          <td><xsl:value-of select="$gid"/></td>
          <td>
            <!-- Iterate over each file containing a digimage with $gid as
                 its group-id -->
            <xsl:for-each select="//file[child::digimage[@group-id=$gid]]">
              <xsl:sort select="@html"/>
              <xsl:text>[</xsl:text>
              <a href="{concat(@html,'#digimage-',$gid)}">
                <xsl:for-each select="digimage[@group-id=$gid]">
                  <xsl:sort select="@frame-id"/>
                  <xsl:variable name="fid" select="@frame-id"/>
                  <xsl:if test="not(following-sibling::digimage[@group-id=$gid][@frame-id=$fid])">
                    <xsl:value-of select="$fid"/>
                    <xsl:if test="not(position() = last())">
                      <xsl:text>, </xsl:text>
                    </xsl:if>
                  </xsl:if>
                </xsl:for-each>
              </a>
              <xsl:text>] </xsl:text>
            </xsl:for-each>
          </td>
        </tr>
      </xsl:for-each>
    </table>
  </xsl:template>


</xsl:stylesheet>
