% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tximport.R
\name{tximport}
\alias{tximport}
\title{Import transcript-level abundances and counts
for transcript- and gene-level analysis packages}
\usage{
tximport(files, type = c("none", "salmon", "sailfish", "alevin",
  "kallisto", "rsem", "stringtie"), txIn = TRUE, txOut = FALSE,
  countsFromAbundance = c("no", "scaledTPM", "lengthScaledTPM",
  "dtuScaledTPM"), tx2gene = NULL, varReduce = FALSE,
  dropInfReps = FALSE, infRepStat = NULL, ignoreTxVersion = FALSE,
  ignoreAfterBar = FALSE, geneIdCol, txIdCol, abundanceCol, countsCol,
  lengthCol, importer = NULL, existenceOptional = FALSE,
  sparse = FALSE, sparseThreshold = 1, readLength = 75,
  forceSlow = FALSE)
}
\arguments{
\item{files}{a character vector of filenames for the transcript-level abundances}

\item{type}{character, the type of software used to generate the abundances.
Options are "salmon", "sailfish", "alevin", "kallisto", "rsem", "stringtie", or "none".
This argument is used to autofill the arguments below (geneIdCol, etc.)
"none" means that the user will specify these columns.}

\item{txIn}{logical, whether the incoming files are transcript level (default TRUE)}

\item{txOut}{logical, whether the function should just output
transcript-level (default FALSE)}

\item{countsFromAbundance}{character, either "no" (default), "scaledTPM",
"lengthScaledTPM", or "dtuScaledTPM".
Whether to generate estimated counts using abundance estimates:
\itemize{
  \item scaled up to library size (scaledTPM),
  \item scaled using the average transcript length over samples
        and then the library size (lengthScaledTPM), or
  \item scaled using the median transcript length among isoforms of a gene,
        and then the library size (dtuScaledTPM). 
}
dtuScaledTPM is designed for DTU analysis in combination with \code{txOut=TRUE},
and it requires specifing a \code{tx2gene} data.frame.
dtuScaledTPM works such that within a gene, values from all samples and
all transcripts get scaled by the same fixed median transcript length.
If using scaledTPM, lengthScaledTPM, or geneLengthScaledTPM, 
the counts are no longer correlated across samples with transcript length,
and so the length offset matrix should not be used.}

\item{tx2gene}{a two-column data.frame linking transcript id (column 1)
to gene id (column 2).
the column names are not relevant, but this column order must be used. 
this argument is required for gene-level summarization for methods
that provides transcript-level estimates only
(kallisto, Salmon, Sailfish)}

\item{varReduce}{whether to reduce per-sample inferential replicates
information into a matrix of sample variances \code{variance} (default FALSE).
alevin computes inferential variance by default for bootstrap
inferential replicates, so this argument is ignored/not necessary}

\item{dropInfReps}{whether to skip reading in inferential replicates
(default FALSE). For alevin, \code{tximport} will still read in the
inferential variance matrix if it exists}

\item{infRepStat}{a function to re-compute counts and abundances from the
inferential replicates, e.g. \code{matrixStats::rowMedians} to re-compute counts 
as the median of the inferential replicates. The order of operations is:
first counts are re-computed, then abundances are re-computed.
Following this, if \code{countsFromAbundance} is not "no",
\code{tximport} will again re-compute counts from the re-computed abundances.
\code{infRepStat} should operate on rows of a matrix. (default is NULL)}

\item{ignoreTxVersion}{logical, whether to split the tx id on the '.' character
to remove version information to facilitate matching with the tx id in \code{tx2gene}
(default FALSE)}

\item{ignoreAfterBar}{logical, whether to split the tx id on the '|' character
to facilitate matching with the tx id in \code{tx2gene} (default FALSE)}

\item{geneIdCol}{name of column with gene id. if missing, the \code{tx2gene} argument can be used}

\item{txIdCol}{name of column with tx id}

\item{abundanceCol}{name of column with abundances (e.g. TPM or FPKM)}

\item{countsCol}{name of column with estimated counts}

\item{lengthCol}{name of column with feature length information}

\item{importer}{a function used to read in the files}

\item{existenceOptional}{logical, should tximport not check if files exist before attempting
import (default FALSE, meaning files must exist according to \code{file.exists})}

\item{sparse}{logical, whether to try to import data sparsely (default is FALSE).
Initial implementation for \code{txOut=TRUE}, \code{countsFromAbundance="no"}
or \code{"scaledTPM"}, no inferential replicates. Only counts matrix
is returned (and abundance matrix if using \code{"scaledTPM"})}

\item{sparseThreshold}{the minimum threshold for including a count as a
non-zero count during sparse import (default is 1)}

\item{readLength}{numeric, the read length used to calculate counts from
StringTie's output of coverage. Default value (from StringTie) is 75.
The formula used to calculate counts is:
\code{cov * transcript length / read length}}

\item{forceSlow}{logical, argument used for testing. Will force the use of
the slower R code for importing alevin, even if \code{fishpond}
library is installed. Default is FALSE}
}
\value{
a simple list containing matrices: abundance, counts, length.
Another list element 'countsFromAbundance' carries through
the character argument used in the tximport call.
The length matrix contains the average transcript length for each
gene which can be used as an offset for gene-level analysis.
If detected, and \code{txOut=TRUE}, inferential replicates for
each sample will be imported and stored as a list of matrices,
itself an element \code{infReps} in the returned list.
An exception is alevin, in which the \code{infReps} are a list
of bootstrap replicate matrices, where each matrix has
genes as rows and cells as columns.
If \code{varReduce=TRUE} the inferential replicates will be summarized
according to the sample variance, and stored as a matrix \code{variance}.
alevin already computes the variance of the bootstrap inferential replicates
and so this is imported without needing to specify \code{varReduce=TRUE}
(note that alevin uses the 1/N variance estimator, so not the same as \code{var}).
}
\description{
\code{tximport} imports transcript-level estimates from various
external software and optionally summarizes abundances, counts,
and transcript lengths
to the gene-level (default) or outputs transcript-level matrices
(see \code{txOut} argument).
}
\details{
\code{tximport} will also load in information about inferential replicates --
a list of matrices of the Gibbs samples from the posterior, or bootstrap replicates,
per sample -- if these data are available in the expected locations relative to the \code{files}.
The inferential replicates, stored in \code{infReps} in the output list,
are on estimated counts, and therefore follow \code{counts} in the output list.
By setting \code{varReduce=TRUE}, the inferential replicate matrices
will be replaced by a single matrix with the sample variance per transcript/gene and per sample.

While \code{tximport} summarizes to the gene-level by default, 
the user can also perform the import and summarization steps manually,
by specifing \code{txOut=TRUE} and then using the function \code{summarizeToGene}.
Note however that this is equivalent to \code{tximport} with \code{txOut=FALSE} (the default).

Solutions to the error "tximport failed at summarizing to the gene-level":

\enumerate{
  \item provide a \code{tx2gene} data.frame linking transcripts to genes (more below)
  \item avoid gene-level summarization by specifying \code{txOut=TRUE}
  \item set \code{geneIdCol} to an appropriate column in the files
}

See \code{vignette('tximport')} for example code for generating a
\code{tx2gene} data.frame from a \code{TxDb} object.
Note that the \code{keys} and \code{select} functions used
to create the \code{tx2gene} object are documented
in the man page for \link[AnnotationDbi]{AnnotationDb-class} objects
in the AnnotationDbi package (TxDb inherits from AnnotationDb).
For further details on generating TxDb objects from various inputs
see \code{vignette('GenomicFeatures')} from the GenomicFeatures package.

For \code{type="alevin"} all arguments other than \code{files},
\code{dropInfReps}, and \code{forceSlow} are ignored,
and \code{files} should point to a single \code{quants_mat.gz} file,
in the directory structure created by the alevin software
(e.g. do not move the file or delete the other important files).
Note that importing alevin quantifications will be much faster by first
installing the \code{fishpond} package, which contains a
C++ importer for alevin's EDS format.
For alevin, \code{tximport} is importing the gene-by-cell matrix of counts,
as \code{txi$counts}, and effective lengths are not estimated.
\code{txi$variance} may also be imported if inferential replicates were
used, as well as inferential replicates if these were output by alevin.
Length correction should not be applied to datasets where there
is not an expected correlation of counts and feature length.
}
\examples{

# load data for demonstrating tximport
# note that the vignette shows more examples
# including how to read in files quickly using the readr package

library(tximportData)
dir <- system.file("extdata", package="tximportData")
samples <- read.table(file.path(dir,"samples.txt"), header=TRUE)
files <- file.path(dir,"salmon", samples$run, "quant.sf.gz")
names(files) <- paste0("sample",1:6)

# tx2gene links transcript IDs to gene IDs for summarization
tx2gene <- read.csv(file.path(dir, "tx2gene.gencode.v27.csv"))

txi <- tximport(files, type="salmon", tx2gene=tx2gene)

}
\references{
Charlotte Soneson, Michael I. Love, Mark D. Robinson (2015):
Differential analyses for RNA-seq: transcript-level estimates
improve gene-level inferences. F1000Research.
\url{http://dx.doi.org/10.12688/f1000research.7563.1}
}
