// Copyright 2015 Google Inc. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Autogenerated by gen.go, do not edit.

package cairo

import (
	"fmt"
	"io"
	"runtime"
	"unsafe"
)

/*
#cgo pkg-config: cairo
#include <cairo.h>
#if CAIRO_HAS_SVG_SURFACE
#include <cairo-svg.h>
#endif
#if CAIRO_HAS_XLIB_SURFACE
#include <cairo-xlib.h>
#endif
#include <stdlib.h>

int gocairoWriteFunc(int key, const unsigned char* data, unsigned int length);
int gocairoReadFunc(int key, const unsigned char* data, unsigned int length);

// A cairo_write_func_t for use in cairo_surface_write_to_png.
cairo_status_t gocairo_write_func(void *closure,
                                  const unsigned char *data,
                                  unsigned int length) {
  return gocairoWriteFunc(*(int*)closure, data, length)
    ? CAIRO_STATUS_SUCCESS
    : CAIRO_STATUS_WRITE_ERROR;
}

// A cairo_read_func_t for use in cairo_image_surface_create_from_png_stream.
cairo_status_t gocairo_read_func(void *closure,
                                 const unsigned char *data,
                                 unsigned int length) {
  return gocairoReadFunc(*(int*)closure, data, length)
    ? CAIRO_STATUS_SUCCESS
    : CAIRO_STATUS_WRITE_ERROR;
}
*/
import "C"

// Error implements the error interface.
func (s Status) Error() string {
	return C.GoString(C.cairo_status_to_string(C.cairo_status_t(s)))
}

// WriteToPNG encodes a Surface to an io.Writer as a PNG file.
func (surface *Surface) WriteToPNG(w io.Writer) error {
	data := writeClosure{w: w}
	key := goPointers.put(data)
	status := C.cairo_surface_write_to_png_stream((*C.cairo_surface_t)(surface.Ptr),
		(C.cairo_write_func_t)(unsafe.Pointer(C.gocairo_write_func)),
		unsafe.Pointer(&key))
	goPointers.clear(key)
	// TODO: which should we prefer between writeClosure.err and status?
	// Perhaps test against CAIRO_STATUS_WRITE_ERROR?  Needs a test case.
	return Status(status).toError()
}

// ImageSurfaceCreateFromPNGStream creates an ImageSurface from a stream of
// PNG data.
func ImageSurfaceCreateFromPNGStream(r io.Reader) (*ImageSurface, error) {
	data := readClosure{r: r}
	key := goPointers.put(data)
	surf := &ImageSurface{wrapSurface(C.cairo_image_surface_create_from_png_stream(
		(C.cairo_read_func_t)(unsafe.Pointer(C.gocairo_read_func)),
		unsafe.Pointer(&key)))}
	goPointers.clear(key)
	// TODO: which should we prefer between readClosure.err and status?
	// Perhaps test against CAIRO_STATUS_WRITE_ERROR?  Needs a test case.
	return surf, surf.status()
}

// PathIter creates an iterator over the segments within the path.
func (p *Path) Iter() *PathIter {
	return &PathIter{path: p, i: 0}
}

// PathIter iterates a Path.
type PathIter struct {
	path *Path
	i    C.int
}

// Next returns the next PathSegment, or returns nil at the end of the path.
func (pi *PathIter) Next() *PathSegment {
	if pi.i >= pi.path.Ptr.num_data {
		return nil
	}
	// path.data is an array of cairo_path_data_t, but the union makes
	// things complicated.
	dataArray := (*[1 << 30]C.cairo_path_data_t)(unsafe.Pointer(pi.path.Ptr.data))
	seg, ofs := decodePathSegment(unsafe.Pointer(&dataArray[pi.i]))
	pi.i += C.int(ofs)
	return seg
}

type ImageSurface struct {
	*Surface
}
type RecordingSurface struct {
	*Surface
}
type SurfaceObserver struct {
	*Surface
}
type ToyFontFace struct {
	*FontFace
}
type MeshPattern struct {
	*Pattern
}
type SVGSurface struct {
	*Surface
}
type XlibSurface struct {
	*Surface
}
type XlibDevice struct {
	*Device
}

// See cairo_version().
//
// C API documentation: http://cairographics.org/manual/cairo-Version-Information.html#cairo-version
func Version() int {
	ret := int(C.cairo_version())
	return ret
}

// See cairo_version_string().
//
// C API documentation: http://cairographics.org/manual/cairo-Version-Information.html#cairo-version-string
func VersionString() string {
	ret := C.GoString(C.cairo_version_string())
	return ret
}

// See cairo_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-t
type Context struct {
	Ptr *C.cairo_t
}

func freeContext(obj *Context) {
	C.cairo_destroy(obj.Ptr)
}
func wrapContext(p *C.cairo_t) *Context {
	ret := &Context{p}
	runtime.SetFinalizer(ret, freeContext)
	return ret
}

// Wrap a C cairo_t* found from some external source as a *Context.  The Go side will destroy the reference when it's no longer used.
func WrapContext(p unsafe.Pointer) *Context {
	return wrapContext((*C.cairo_t)(p))
}

// Construct a Context from a C cairo_t* found from some exernal source.  It is the caller's responsibility to ensure the pointer lives.
func BorrowContext(p unsafe.Pointer) *Context {
	return &Context{(*C.cairo_t)(p)}
}

// See cairo_surface_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-t
type Surface struct {
	Ptr *C.cairo_surface_t
}

func freeSurface(obj *Surface) {
	C.cairo_surface_destroy(obj.Ptr)
}
func wrapSurface(p *C.cairo_surface_t) *Surface {
	ret := &Surface{p}
	runtime.SetFinalizer(ret, freeSurface)
	return ret
}

// Wrap a C cairo_surface_t* found from some external source as a *Surface.  The Go side will destroy the reference when it's no longer used.
func WrapSurface(p unsafe.Pointer) *Surface {
	return wrapSurface((*C.cairo_surface_t)(p))
}

// Construct a Surface from a C cairo_surface_t* found from some exernal source.  It is the caller's responsibility to ensure the pointer lives.
func BorrowSurface(p unsafe.Pointer) *Surface {
	return &Surface{(*C.cairo_surface_t)(p)}
}

// See cairo_device_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-t
type Device struct {
	Ptr *C.cairo_device_t
}

func freeDevice(obj *Device) {
	C.cairo_device_destroy(obj.Ptr)
}
func wrapDevice(p *C.cairo_device_t) *Device {
	ret := &Device{p}
	runtime.SetFinalizer(ret, freeDevice)
	return ret
}

// Wrap a C cairo_device_t* found from some external source as a *Device.  The Go side will destroy the reference when it's no longer used.
func WrapDevice(p unsafe.Pointer) *Device {
	return wrapDevice((*C.cairo_device_t)(p))
}

// Construct a Device from a C cairo_device_t* found from some exernal source.  It is the caller's responsibility to ensure the pointer lives.
func BorrowDevice(p unsafe.Pointer) *Device {
	return &Device{(*C.cairo_device_t)(p)}
}

// See cairo_matrix_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-t
type Matrix struct {
	Xx float64
	Yx float64
	Xy float64
	Yy float64
	X0 float64
	Y0 float64
}

// See cairo_pattern_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-t
type Pattern struct {
	Ptr *C.cairo_pattern_t
}

func freePattern(obj *Pattern) {
	C.cairo_pattern_destroy(obj.Ptr)
}
func wrapPattern(p *C.cairo_pattern_t) *Pattern {
	ret := &Pattern{p}
	runtime.SetFinalizer(ret, freePattern)
	return ret
}

// Wrap a C cairo_pattern_t* found from some external source as a *Pattern.  The Go side will destroy the reference when it's no longer used.
func WrapPattern(p unsafe.Pointer) *Pattern {
	return wrapPattern((*C.cairo_pattern_t)(p))
}

// Construct a Pattern from a C cairo_pattern_t* found from some exernal source.  It is the caller's responsibility to ensure the pointer lives.
func BorrowPattern(p unsafe.Pointer) *Pattern {
	return &Pattern{(*C.cairo_pattern_t)(p)}
}

// See cairo_status_t.
//
// C API documentation: http://cairographics.org/manual/cairo-Error-handling.html#cairo-status-t
type Status int

const (
	StatusSuccess                 Status = C.CAIRO_STATUS_SUCCESS
	StatusNoMemory                Status = C.CAIRO_STATUS_NO_MEMORY
	StatusInvalidRestore          Status = C.CAIRO_STATUS_INVALID_RESTORE
	StatusInvalidPopGroup         Status = C.CAIRO_STATUS_INVALID_POP_GROUP
	StatusNoCurrentPoint          Status = C.CAIRO_STATUS_NO_CURRENT_POINT
	StatusInvalidMatrix           Status = C.CAIRO_STATUS_INVALID_MATRIX
	StatusInvalidStatus           Status = C.CAIRO_STATUS_INVALID_STATUS
	StatusNullPointer             Status = C.CAIRO_STATUS_NULL_POINTER
	StatusInvalidString           Status = C.CAIRO_STATUS_INVALID_STRING
	StatusInvalidPathData         Status = C.CAIRO_STATUS_INVALID_PATH_DATA
	StatusReadError               Status = C.CAIRO_STATUS_READ_ERROR
	StatusWriteError              Status = C.CAIRO_STATUS_WRITE_ERROR
	StatusSurfaceFinished         Status = C.CAIRO_STATUS_SURFACE_FINISHED
	StatusSurfaceTypeMismatch     Status = C.CAIRO_STATUS_SURFACE_TYPE_MISMATCH
	StatusPatternTypeMismatch     Status = C.CAIRO_STATUS_PATTERN_TYPE_MISMATCH
	StatusInvalidContent          Status = C.CAIRO_STATUS_INVALID_CONTENT
	StatusInvalidFormat           Status = C.CAIRO_STATUS_INVALID_FORMAT
	StatusInvalidVisual           Status = C.CAIRO_STATUS_INVALID_VISUAL
	StatusFileNotFound            Status = C.CAIRO_STATUS_FILE_NOT_FOUND
	StatusInvalidDash             Status = C.CAIRO_STATUS_INVALID_DASH
	StatusInvalidDscComment       Status = C.CAIRO_STATUS_INVALID_DSC_COMMENT
	StatusInvalidIndex            Status = C.CAIRO_STATUS_INVALID_INDEX
	StatusClipNotRepresentable    Status = C.CAIRO_STATUS_CLIP_NOT_REPRESENTABLE
	StatusTempFileError           Status = C.CAIRO_STATUS_TEMP_FILE_ERROR
	StatusInvalidStride           Status = C.CAIRO_STATUS_INVALID_STRIDE
	StatusFontTypeMismatch        Status = C.CAIRO_STATUS_FONT_TYPE_MISMATCH
	StatusUserFontImmutable       Status = C.CAIRO_STATUS_USER_FONT_IMMUTABLE
	StatusUserFontError           Status = C.CAIRO_STATUS_USER_FONT_ERROR
	StatusNegativeCount           Status = C.CAIRO_STATUS_NEGATIVE_COUNT
	StatusInvalidClusters         Status = C.CAIRO_STATUS_INVALID_CLUSTERS
	StatusInvalidSlant            Status = C.CAIRO_STATUS_INVALID_SLANT
	StatusInvalidWeight           Status = C.CAIRO_STATUS_INVALID_WEIGHT
	StatusInvalidSize             Status = C.CAIRO_STATUS_INVALID_SIZE
	StatusUserFontNotImplemented  Status = C.CAIRO_STATUS_USER_FONT_NOT_IMPLEMENTED
	StatusDeviceTypeMismatch      Status = C.CAIRO_STATUS_DEVICE_TYPE_MISMATCH
	StatusDeviceError             Status = C.CAIRO_STATUS_DEVICE_ERROR
	StatusInvalidMeshConstruction Status = C.CAIRO_STATUS_INVALID_MESH_CONSTRUCTION
	StatusDeviceFinished          Status = C.CAIRO_STATUS_DEVICE_FINISHED
	StatusJbig2GlobalMissing      Status = C.CAIRO_STATUS_JBIG2_GLOBAL_MISSING
	StatusLastStatus              Status = C.CAIRO_STATUS_LAST_STATUS
)

// See cairo_content_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-content-t
type Content int

const (
	ContentColor      Content = C.CAIRO_CONTENT_COLOR
	ContentAlpha      Content = C.CAIRO_CONTENT_ALPHA
	ContentColorAlpha Content = C.CAIRO_CONTENT_COLOR_ALPHA
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i Content) String() string {
	switch i {
	case ContentColor:
		return "ContentColor"
	case ContentAlpha:
		return "ContentAlpha"
	case ContentColorAlpha:
		return "ContentColorAlpha"
	default:
		return fmt.Sprintf("Content(%d)", i)
	}
}

// See cairo_format_t.
//
// C API documentation: http://cairographics.org/manual/cairo-Image-Surfaces.html#cairo-format-t
type Format int

const (
	FormatInvalid  Format = C.CAIRO_FORMAT_INVALID
	FormatARGB32   Format = C.CAIRO_FORMAT_ARGB32
	FormatRGB24    Format = C.CAIRO_FORMAT_RGB24
	FormatA8       Format = C.CAIRO_FORMAT_A8
	FormatA1       Format = C.CAIRO_FORMAT_A1
	FormatRGB16565 Format = C.CAIRO_FORMAT_RGB16_565
	FormatRGB30    Format = C.CAIRO_FORMAT_RGB30
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i Format) String() string {
	switch i {
	case FormatInvalid:
		return "FormatInvalid"
	case FormatARGB32:
		return "FormatARGB32"
	case FormatRGB24:
		return "FormatRGB24"
	case FormatA8:
		return "FormatA8"
	case FormatA1:
		return "FormatA1"
	case FormatRGB16565:
		return "FormatRGB16565"
	case FormatRGB30:
		return "FormatRGB30"
	default:
		return fmt.Sprintf("Format(%d)", i)
	}
}

// See cairo_create().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-create
func Create(target *Surface) *Context {
	ret := wrapContext(C.cairo_create(target.Ptr))
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_save().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-save
func (cr *Context) Save() {
	C.cairo_save(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_restore().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-restore
func (cr *Context) Restore() {
	C.cairo_restore(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_push_group().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-push-group
func (cr *Context) PushGroup() {
	C.cairo_push_group(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_push_group_with_content().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-push-group-with-content
func (cr *Context) PushGroupWithContent(content Content) {
	C.cairo_push_group_with_content(cr.Ptr, C.cairo_content_t(content))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_pop_group().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-pop-group
func (cr *Context) PopGroup() *Pattern {
	ret := wrapPattern(C.cairo_pop_group(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_pop_group_to_source().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-pop-group-to-source
func (cr *Context) PopGroupToSource() {
	C.cairo_pop_group_to_source(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_operator_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-operator-t
type Operator int

const (
	OperatorClear         Operator = C.CAIRO_OPERATOR_CLEAR
	OperatorSource        Operator = C.CAIRO_OPERATOR_SOURCE
	OperatorOver          Operator = C.CAIRO_OPERATOR_OVER
	OperatorIn            Operator = C.CAIRO_OPERATOR_IN
	OperatorOut           Operator = C.CAIRO_OPERATOR_OUT
	OperatorAtop          Operator = C.CAIRO_OPERATOR_ATOP
	OperatorDest          Operator = C.CAIRO_OPERATOR_DEST
	OperatorDestOver      Operator = C.CAIRO_OPERATOR_DEST_OVER
	OperatorDestIn        Operator = C.CAIRO_OPERATOR_DEST_IN
	OperatorDestOut       Operator = C.CAIRO_OPERATOR_DEST_OUT
	OperatorDestAtop      Operator = C.CAIRO_OPERATOR_DEST_ATOP
	OperatorXOR           Operator = C.CAIRO_OPERATOR_XOR
	OperatorAdd           Operator = C.CAIRO_OPERATOR_ADD
	OperatorSaturate      Operator = C.CAIRO_OPERATOR_SATURATE
	OperatorMultiply      Operator = C.CAIRO_OPERATOR_MULTIPLY
	OperatorScreen        Operator = C.CAIRO_OPERATOR_SCREEN
	OperatorOverlay       Operator = C.CAIRO_OPERATOR_OVERLAY
	OperatorDarken        Operator = C.CAIRO_OPERATOR_DARKEN
	OperatorLighten       Operator = C.CAIRO_OPERATOR_LIGHTEN
	OperatorColorDodge    Operator = C.CAIRO_OPERATOR_COLOR_DODGE
	OperatorColorBurn     Operator = C.CAIRO_OPERATOR_COLOR_BURN
	OperatorHardLight     Operator = C.CAIRO_OPERATOR_HARD_LIGHT
	OperatorSoftLight     Operator = C.CAIRO_OPERATOR_SOFT_LIGHT
	OperatorDifference    Operator = C.CAIRO_OPERATOR_DIFFERENCE
	OperatorExclusion     Operator = C.CAIRO_OPERATOR_EXCLUSION
	OperatorHslHue        Operator = C.CAIRO_OPERATOR_HSL_HUE
	OperatorHslSaturation Operator = C.CAIRO_OPERATOR_HSL_SATURATION
	OperatorHslColor      Operator = C.CAIRO_OPERATOR_HSL_COLOR
	OperatorHslLuminosity Operator = C.CAIRO_OPERATOR_HSL_LUMINOSITY
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i Operator) String() string {
	switch i {
	case OperatorClear:
		return "OperatorClear"
	case OperatorSource:
		return "OperatorSource"
	case OperatorOver:
		return "OperatorOver"
	case OperatorIn:
		return "OperatorIn"
	case OperatorOut:
		return "OperatorOut"
	case OperatorAtop:
		return "OperatorAtop"
	case OperatorDest:
		return "OperatorDest"
	case OperatorDestOver:
		return "OperatorDestOver"
	case OperatorDestIn:
		return "OperatorDestIn"
	case OperatorDestOut:
		return "OperatorDestOut"
	case OperatorDestAtop:
		return "OperatorDestAtop"
	case OperatorXOR:
		return "OperatorXOR"
	case OperatorAdd:
		return "OperatorAdd"
	case OperatorSaturate:
		return "OperatorSaturate"
	case OperatorMultiply:
		return "OperatorMultiply"
	case OperatorScreen:
		return "OperatorScreen"
	case OperatorOverlay:
		return "OperatorOverlay"
	case OperatorDarken:
		return "OperatorDarken"
	case OperatorLighten:
		return "OperatorLighten"
	case OperatorColorDodge:
		return "OperatorColorDodge"
	case OperatorColorBurn:
		return "OperatorColorBurn"
	case OperatorHardLight:
		return "OperatorHardLight"
	case OperatorSoftLight:
		return "OperatorSoftLight"
	case OperatorDifference:
		return "OperatorDifference"
	case OperatorExclusion:
		return "OperatorExclusion"
	case OperatorHslHue:
		return "OperatorHslHue"
	case OperatorHslSaturation:
		return "OperatorHslSaturation"
	case OperatorHslColor:
		return "OperatorHslColor"
	case OperatorHslLuminosity:
		return "OperatorHslLuminosity"
	default:
		return fmt.Sprintf("Operator(%d)", i)
	}
}

// See cairo_set_operator().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-operator
func (cr *Context) SetOperator(op Operator) {
	C.cairo_set_operator(cr.Ptr, C.cairo_operator_t(op))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_source().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-source
func (cr *Context) SetSource(source *Pattern) {
	C.cairo_set_source(cr.Ptr, source.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_source_rgb().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-source-rgb
func (cr *Context) SetSourceRGB(red, green, blue float64) {
	C.cairo_set_source_rgb(cr.Ptr, C.double(red), C.double(green), C.double(blue))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_source_rgba().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-source-rgba
func (cr *Context) SetSourceRGBA(red, green, blue, alpha float64) {
	C.cairo_set_source_rgba(cr.Ptr, C.double(red), C.double(green), C.double(blue), C.double(alpha))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_source_surface().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-source-surface
func (cr *Context) SetSourceSurface(surface *Surface, x, y float64) {
	C.cairo_set_source_surface(cr.Ptr, surface.Ptr, C.double(x), C.double(y))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_tolerance().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-tolerance
func (cr *Context) SetTolerance(tolerance float64) {
	C.cairo_set_tolerance(cr.Ptr, C.double(tolerance))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_antialias_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-antialias-t
type Antialias int

const (
	AntialiasDefault  Antialias = C.CAIRO_ANTIALIAS_DEFAULT
	AntialiasNone     Antialias = C.CAIRO_ANTIALIAS_NONE
	AntialiasGray     Antialias = C.CAIRO_ANTIALIAS_GRAY
	AntialiasSubpixel Antialias = C.CAIRO_ANTIALIAS_SUBPIXEL
	AntialiasFast     Antialias = C.CAIRO_ANTIALIAS_FAST
	AntialiasGood     Antialias = C.CAIRO_ANTIALIAS_GOOD
	AntialiasBest     Antialias = C.CAIRO_ANTIALIAS_BEST
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i Antialias) String() string {
	switch i {
	case AntialiasDefault:
		return "AntialiasDefault"
	case AntialiasNone:
		return "AntialiasNone"
	case AntialiasGray:
		return "AntialiasGray"
	case AntialiasSubpixel:
		return "AntialiasSubpixel"
	case AntialiasFast:
		return "AntialiasFast"
	case AntialiasGood:
		return "AntialiasGood"
	case AntialiasBest:
		return "AntialiasBest"
	default:
		return fmt.Sprintf("Antialias(%d)", i)
	}
}

// See cairo_set_antialias().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-antialias
func (cr *Context) SetAntialias(antialias Antialias) {
	C.cairo_set_antialias(cr.Ptr, C.cairo_antialias_t(antialias))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_fill_rule_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-fill-rule-t
type FillRule int

const (
	FillRuleWinding FillRule = C.CAIRO_FILL_RULE_WINDING
	FillRuleEvenOdd FillRule = C.CAIRO_FILL_RULE_EVEN_ODD
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i FillRule) String() string {
	switch i {
	case FillRuleWinding:
		return "FillRuleWinding"
	case FillRuleEvenOdd:
		return "FillRuleEvenOdd"
	default:
		return fmt.Sprintf("FillRule(%d)", i)
	}
}

// See cairo_set_fill_rule().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-fill-rule
func (cr *Context) SetFillRule(fillRule FillRule) {
	C.cairo_set_fill_rule(cr.Ptr, C.cairo_fill_rule_t(fillRule))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_line_width().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-line-width
func (cr *Context) SetLineWidth(width float64) {
	C.cairo_set_line_width(cr.Ptr, C.double(width))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_line_cap_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-line-cap-t
type LineCap int

const (
	LineCapButt   LineCap = C.CAIRO_LINE_CAP_BUTT
	LineCapRound  LineCap = C.CAIRO_LINE_CAP_ROUND
	LineCapSquare LineCap = C.CAIRO_LINE_CAP_SQUARE
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i LineCap) String() string {
	switch i {
	case LineCapButt:
		return "LineCapButt"
	case LineCapRound:
		return "LineCapRound"
	case LineCapSquare:
		return "LineCapSquare"
	default:
		return fmt.Sprintf("LineCap(%d)", i)
	}
}

// See cairo_set_line_cap().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-line-cap
func (cr *Context) SetLineCap(lineCap LineCap) {
	C.cairo_set_line_cap(cr.Ptr, C.cairo_line_cap_t(lineCap))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_line_join_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-line-join-t
type LineJoin int

const (
	LineJoinMiter LineJoin = C.CAIRO_LINE_JOIN_MITER
	LineJoinRound LineJoin = C.CAIRO_LINE_JOIN_ROUND
	LineJoinBevel LineJoin = C.CAIRO_LINE_JOIN_BEVEL
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i LineJoin) String() string {
	switch i {
	case LineJoinMiter:
		return "LineJoinMiter"
	case LineJoinRound:
		return "LineJoinRound"
	case LineJoinBevel:
		return "LineJoinBevel"
	default:
		return fmt.Sprintf("LineJoin(%d)", i)
	}
}

// See cairo_set_line_join().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-line-join
func (cr *Context) SetLineJoin(lineJoin LineJoin) {
	C.cairo_set_line_join(cr.Ptr, C.cairo_line_join_t(lineJoin))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_dash().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-dash
func (cr *Context) SetDash(dashes []float64, offset float64) {
	C.cairo_set_dash(cr.Ptr, (*C.double)(sliceBytes(unsafe.Pointer(&dashes))), C.int(len(dashes)), C.double(offset))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_miter_limit().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-set-miter-limit
func (cr *Context) SetMiterLimit(limit float64) {
	C.cairo_set_miter_limit(cr.Ptr, C.double(limit))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_translate().
//
// C API documentation: http://cairographics.org/manual/cairo-Transformations.html#cairo-translate
func (cr *Context) Translate(tx, ty float64) {
	C.cairo_translate(cr.Ptr, C.double(tx), C.double(ty))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_scale().
//
// C API documentation: http://cairographics.org/manual/cairo-Transformations.html#cairo-scale
func (cr *Context) Scale(sx, sy float64) {
	C.cairo_scale(cr.Ptr, C.double(sx), C.double(sy))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_rotate().
//
// C API documentation: http://cairographics.org/manual/cairo-Transformations.html#cairo-rotate
func (cr *Context) Rotate(angle float64) {
	C.cairo_rotate(cr.Ptr, C.double(angle))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_transform().
//
// C API documentation: http://cairographics.org/manual/cairo-Transformations.html#cairo-transform
func (cr *Context) Transform(matrix *Matrix) {
	C.cairo_transform(cr.Ptr, (*C.cairo_matrix_t)(unsafe.Pointer(matrix)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_matrix().
//
// C API documentation: http://cairographics.org/manual/cairo-Transformations.html#cairo-set-matrix
func (cr *Context) SetMatrix(matrix *Matrix) {
	C.cairo_set_matrix(cr.Ptr, (*C.cairo_matrix_t)(unsafe.Pointer(matrix)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_identity_matrix().
//
// C API documentation: http://cairographics.org/manual/cairo-Transformations.html#cairo-identity-matrix
func (cr *Context) IdentityMatrix() {
	C.cairo_identity_matrix(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_user_to_device().
//
// C API documentation: http://cairographics.org/manual/cairo-Transformations.html#cairo-user-to-device
func (cr *Context) UserToDevice(x, y *float64) {
	C.cairo_user_to_device(cr.Ptr, (*C.double)(unsafe.Pointer(x)), (*C.double)(unsafe.Pointer(y)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_user_to_device_distance().
//
// C API documentation: http://cairographics.org/manual/cairo-Transformations.html#cairo-user-to-device-distance
func (cr *Context) UserToDeviceDistance(dx, dy *float64) {
	C.cairo_user_to_device_distance(cr.Ptr, (*C.double)(unsafe.Pointer(dx)), (*C.double)(unsafe.Pointer(dy)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_device_to_user().
//
// C API documentation: http://cairographics.org/manual/cairo-Transformations.html#cairo-device-to-user
func (cr *Context) DeviceToUser(x, y *float64) {
	C.cairo_device_to_user(cr.Ptr, (*C.double)(unsafe.Pointer(x)), (*C.double)(unsafe.Pointer(y)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_device_to_user_distance().
//
// C API documentation: http://cairographics.org/manual/cairo-Transformations.html#cairo-device-to-user-distance
func (cr *Context) DeviceToUserDistance(dx, dy *float64) {
	C.cairo_device_to_user_distance(cr.Ptr, (*C.double)(unsafe.Pointer(dx)), (*C.double)(unsafe.Pointer(dy)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_new_path().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-new-path
func (cr *Context) NewPath() {
	C.cairo_new_path(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_move_to().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-move-to
func (cr *Context) MoveTo(x, y float64) {
	C.cairo_move_to(cr.Ptr, C.double(x), C.double(y))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_new_sub_path().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-new-sub-path
func (cr *Context) NewSubPath() {
	C.cairo_new_sub_path(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_line_to().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-line-to
func (cr *Context) LineTo(x, y float64) {
	C.cairo_line_to(cr.Ptr, C.double(x), C.double(y))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_curve_to().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-curve-to
func (cr *Context) CurveTo(x1, y1, x2, y2, x3, y3 float64) {
	C.cairo_curve_to(cr.Ptr, C.double(x1), C.double(y1), C.double(x2), C.double(y2), C.double(x3), C.double(y3))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_arc().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-arc
func (cr *Context) Arc(xc, yc, radius, angle1, angle2 float64) {
	C.cairo_arc(cr.Ptr, C.double(xc), C.double(yc), C.double(radius), C.double(angle1), C.double(angle2))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_arc_negative().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-arc-negative
func (cr *Context) ArcNegative(xc, yc, radius, angle1, angle2 float64) {
	C.cairo_arc_negative(cr.Ptr, C.double(xc), C.double(yc), C.double(radius), C.double(angle1), C.double(angle2))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_rel_move_to().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-rel-move-to
func (cr *Context) RelMoveTo(dx, dy float64) {
	C.cairo_rel_move_to(cr.Ptr, C.double(dx), C.double(dy))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_rel_line_to().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-rel-line-to
func (cr *Context) RelLineTo(dx, dy float64) {
	C.cairo_rel_line_to(cr.Ptr, C.double(dx), C.double(dy))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_rel_curve_to().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-rel-curve-to
func (cr *Context) RelCurveTo(dx1, dy1, dx2, dy2, dx3, dy3 float64) {
	C.cairo_rel_curve_to(cr.Ptr, C.double(dx1), C.double(dy1), C.double(dx2), C.double(dy2), C.double(dx3), C.double(dy3))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_rectangle().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-rectangle
func (cr *Context) Rectangle(x, y, width, height float64) {
	C.cairo_rectangle(cr.Ptr, C.double(x), C.double(y), C.double(width), C.double(height))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_close_path().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-close-path
func (cr *Context) ClosePath() {
	C.cairo_close_path(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_path_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-path-extents
func (cr *Context) PathExtents() (float64, float64, float64, float64) {
	var x1 C.double
	var y1 C.double
	var x2 C.double
	var y2 C.double

	C.cairo_path_extents(cr.Ptr, &x1, &y1, &x2, &y2)
	if err := cr.status(); err != nil {
		panic(err)
	}
	return float64(x1), float64(y1), float64(x2), float64(y2)
}

// See cairo_paint().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-paint
func (cr *Context) Paint() {
	C.cairo_paint(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_paint_with_alpha().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-paint-with-alpha
func (cr *Context) PaintWithAlpha(alpha float64) {
	C.cairo_paint_with_alpha(cr.Ptr, C.double(alpha))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_mask().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-mask
func (cr *Context) Mask(pattern *Pattern) {
	C.cairo_mask(cr.Ptr, pattern.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_mask_surface().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-mask-surface
func (cr *Context) MaskSurface(surface *Surface, surfaceX, surfaceY float64) {
	C.cairo_mask_surface(cr.Ptr, surface.Ptr, C.double(surfaceX), C.double(surfaceY))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_stroke().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-stroke
func (cr *Context) Stroke() {
	C.cairo_stroke(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_stroke_preserve().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-stroke-preserve
func (cr *Context) StrokePreserve() {
	C.cairo_stroke_preserve(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_fill().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-fill
func (cr *Context) Fill() {
	C.cairo_fill(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_fill_preserve().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-fill-preserve
func (cr *Context) FillPreserve() {
	C.cairo_fill_preserve(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_copy_page().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-copy-page
func (cr *Context) CopyPage() {
	C.cairo_copy_page(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_show_page().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-show-page
func (cr *Context) ShowPage() {
	C.cairo_show_page(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_in_stroke().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-in-stroke
func (cr *Context) InStroke(x, y float64) bool {
	ret := C.cairo_in_stroke(cr.Ptr, C.double(x), C.double(y)) != 0
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_in_fill().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-in-fill
func (cr *Context) InFill(x, y float64) bool {
	ret := C.cairo_in_fill(cr.Ptr, C.double(x), C.double(y)) != 0
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_in_clip().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-in-clip
func (cr *Context) InClip(x, y float64) bool {
	ret := C.cairo_in_clip(cr.Ptr, C.double(x), C.double(y)) != 0
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_stroke_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-stroke-extents
func (cr *Context) StrokeExtents() (float64, float64, float64, float64) {
	var x1 C.double
	var y1 C.double
	var x2 C.double
	var y2 C.double

	C.cairo_stroke_extents(cr.Ptr, &x1, &y1, &x2, &y2)
	if err := cr.status(); err != nil {
		panic(err)
	}
	return float64(x1), float64(y1), float64(x2), float64(y2)
}

// See cairo_fill_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-fill-extents
func (cr *Context) FillExtents() (float64, float64, float64, float64) {
	var x1 C.double
	var y1 C.double
	var x2 C.double
	var y2 C.double

	C.cairo_fill_extents(cr.Ptr, &x1, &y1, &x2, &y2)
	if err := cr.status(); err != nil {
		panic(err)
	}
	return float64(x1), float64(y1), float64(x2), float64(y2)
}

// See cairo_reset_clip().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-reset-clip
func (cr *Context) ResetClip() {
	C.cairo_reset_clip(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_clip().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-clip
func (cr *Context) Clip() {
	C.cairo_clip(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_clip_preserve().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-clip-preserve
func (cr *Context) ClipPreserve() {
	C.cairo_clip_preserve(cr.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_clip_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-clip-extents
func (cr *Context) ClipExtents() (float64, float64, float64, float64) {
	var x1 C.double
	var y1 C.double
	var x2 C.double
	var y2 C.double

	C.cairo_clip_extents(cr.Ptr, &x1, &y1, &x2, &y2)
	if err := cr.status(); err != nil {
		panic(err)
	}
	return float64(x1), float64(y1), float64(x2), float64(y2)
}

// See cairo_rectangle_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-rectangle-t
type Rectangle struct {
	X      float64
	Y      float64
	Width  float64
	Height float64
}

// See cairo_scaled_font_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-t
type ScaledFont struct {
	Ptr *C.cairo_scaled_font_t
}

func freeScaledFont(obj *ScaledFont) {
	C.cairo_scaled_font_destroy(obj.Ptr)
}
func wrapScaledFont(p *C.cairo_scaled_font_t) *ScaledFont {
	ret := &ScaledFont{p}
	runtime.SetFinalizer(ret, freeScaledFont)
	return ret
}

// Wrap a C cairo_scaled_font_t* found from some external source as a *ScaledFont.  The Go side will destroy the reference when it's no longer used.
func WrapScaledFont(p unsafe.Pointer) *ScaledFont {
	return wrapScaledFont((*C.cairo_scaled_font_t)(p))
}

// Construct a ScaledFont from a C cairo_scaled_font_t* found from some exernal source.  It is the caller's responsibility to ensure the pointer lives.
func BorrowScaledFont(p unsafe.Pointer) *ScaledFont {
	return &ScaledFont{(*C.cairo_scaled_font_t)(p)}
}

// See cairo_font_face_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-face-t.html#cairo-font-face-t
type FontFace struct {
	Ptr *C.cairo_font_face_t
}

func freeFontFace(obj *FontFace) {
	C.cairo_font_face_destroy(obj.Ptr)
}
func wrapFontFace(p *C.cairo_font_face_t) *FontFace {
	ret := &FontFace{p}
	runtime.SetFinalizer(ret, freeFontFace)
	return ret
}

// Wrap a C cairo_font_face_t* found from some external source as a *FontFace.  The Go side will destroy the reference when it's no longer used.
func WrapFontFace(p unsafe.Pointer) *FontFace {
	return wrapFontFace((*C.cairo_font_face_t)(p))
}

// Construct a FontFace from a C cairo_font_face_t* found from some exernal source.  It is the caller's responsibility to ensure the pointer lives.
func BorrowFontFace(p unsafe.Pointer) *FontFace {
	return &FontFace{(*C.cairo_font_face_t)(p)}
}

// See cairo_glyph_t.
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-glyph-t
type Glyph struct {
	Index uint32
	X     float64
	Y     float64
}

// See cairo_text_cluster_flags_t.
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-text-cluster-flags-t
type TextClusterFlags int

const (
	TextClusterFlagBackward TextClusterFlags = C.CAIRO_TEXT_CLUSTER_FLAG_BACKWARD
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i TextClusterFlags) String() string {
	switch i {
	case TextClusterFlagBackward:
		return "TextClusterFlagBackward"
	default:
		return fmt.Sprintf("TextClusterFlags(%d)", i)
	}
}

// See cairo_text_extents_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-text-extents-t
type TextExtents struct {
	XBearing float64
	YBearing float64
	Width    float64
	Height   float64
	XAdvance float64
	YAdvance float64
}

// See cairo_font_extents_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-font-extents-t
type FontExtents struct {
	Ascent      float64
	Descent     float64
	Height      float64
	MaxXAdvance float64
	MaxYAdvance float64
}

// See cairo_font_slant_t.
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-font-slant-t
type FontSlant int

const (
	FontSlantNormal  FontSlant = C.CAIRO_FONT_SLANT_NORMAL
	FontSlantItalic  FontSlant = C.CAIRO_FONT_SLANT_ITALIC
	FontSlantOblique FontSlant = C.CAIRO_FONT_SLANT_OBLIQUE
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i FontSlant) String() string {
	switch i {
	case FontSlantNormal:
		return "FontSlantNormal"
	case FontSlantItalic:
		return "FontSlantItalic"
	case FontSlantOblique:
		return "FontSlantOblique"
	default:
		return fmt.Sprintf("FontSlant(%d)", i)
	}
}

// See cairo_font_weight_t.
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-font-weight-t
type FontWeight int

const (
	FontWeightNormal FontWeight = C.CAIRO_FONT_WEIGHT_NORMAL
	FontWeightBold   FontWeight = C.CAIRO_FONT_WEIGHT_BOLD
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i FontWeight) String() string {
	switch i {
	case FontWeightNormal:
		return "FontWeightNormal"
	case FontWeightBold:
		return "FontWeightBold"
	default:
		return fmt.Sprintf("FontWeight(%d)", i)
	}
}

// See cairo_subpixel_order_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-subpixel-order-t
type SubpixelOrder int

const (
	SubpixelOrderDefault SubpixelOrder = C.CAIRO_SUBPIXEL_ORDER_DEFAULT
	SubpixelOrderRGB     SubpixelOrder = C.CAIRO_SUBPIXEL_ORDER_RGB
	SubpixelOrderBGR     SubpixelOrder = C.CAIRO_SUBPIXEL_ORDER_BGR
	SubpixelOrderVRGB    SubpixelOrder = C.CAIRO_SUBPIXEL_ORDER_VRGB
	SubpixelOrderVBGR    SubpixelOrder = C.CAIRO_SUBPIXEL_ORDER_VBGR
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i SubpixelOrder) String() string {
	switch i {
	case SubpixelOrderDefault:
		return "SubpixelOrderDefault"
	case SubpixelOrderRGB:
		return "SubpixelOrderRGB"
	case SubpixelOrderBGR:
		return "SubpixelOrderBGR"
	case SubpixelOrderVRGB:
		return "SubpixelOrderVRGB"
	case SubpixelOrderVBGR:
		return "SubpixelOrderVBGR"
	default:
		return fmt.Sprintf("SubpixelOrder(%d)", i)
	}
}

// See cairo_hint_style_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-hint-style-t
type HintStyle int

const (
	HintStyleDefault HintStyle = C.CAIRO_HINT_STYLE_DEFAULT
	HintStyleNone    HintStyle = C.CAIRO_HINT_STYLE_NONE
	HintStyleSlight  HintStyle = C.CAIRO_HINT_STYLE_SLIGHT
	HintStyleMedium  HintStyle = C.CAIRO_HINT_STYLE_MEDIUM
	HintStyleFull    HintStyle = C.CAIRO_HINT_STYLE_FULL
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i HintStyle) String() string {
	switch i {
	case HintStyleDefault:
		return "HintStyleDefault"
	case HintStyleNone:
		return "HintStyleNone"
	case HintStyleSlight:
		return "HintStyleSlight"
	case HintStyleMedium:
		return "HintStyleMedium"
	case HintStyleFull:
		return "HintStyleFull"
	default:
		return fmt.Sprintf("HintStyle(%d)", i)
	}
}

// See cairo_hint_metrics_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-hint-metrics-t
type HintMetrics int

const (
	HintMetricsDefault HintMetrics = C.CAIRO_HINT_METRICS_DEFAULT
	HintMetricsOff     HintMetrics = C.CAIRO_HINT_METRICS_OFF
	HintMetricsOn      HintMetrics = C.CAIRO_HINT_METRICS_ON
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i HintMetrics) String() string {
	switch i {
	case HintMetricsDefault:
		return "HintMetricsDefault"
	case HintMetricsOff:
		return "HintMetricsOff"
	case HintMetricsOn:
		return "HintMetricsOn"
	default:
		return fmt.Sprintf("HintMetrics(%d)", i)
	}
}

// See cairo_font_options_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-t
type FontOptions struct {
	Ptr *C.cairo_font_options_t
}

func freeFontOptions(obj *FontOptions) {
	C.cairo_font_options_destroy(obj.Ptr)
}
func wrapFontOptions(p *C.cairo_font_options_t) *FontOptions {
	ret := &FontOptions{p}
	runtime.SetFinalizer(ret, freeFontOptions)
	return ret
}

// Wrap a C cairo_font_options_t* found from some external source as a *FontOptions.  The Go side will destroy the reference when it's no longer used.
func WrapFontOptions(p unsafe.Pointer) *FontOptions {
	return wrapFontOptions((*C.cairo_font_options_t)(p))
}

// Construct a FontOptions from a C cairo_font_options_t* found from some exernal source.  It is the caller's responsibility to ensure the pointer lives.
func BorrowFontOptions(p unsafe.Pointer) *FontOptions {
	return &FontOptions{(*C.cairo_font_options_t)(p)}
}

// See cairo_font_options_create().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-create
func FontOptionsCreate() *FontOptions {
	ret := wrapFontOptions(C.cairo_font_options_create())
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_font_options_copy().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-copy
func (original *FontOptions) Copy() *FontOptions {
	ret := wrapFontOptions(C.cairo_font_options_copy(original.Ptr))
	if err := original.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_font_options_status().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-status
func (options *FontOptions) status() error {
	ret := Status(C.cairo_font_options_status(options.Ptr)).toError()
	return ret
}

// See cairo_font_options_merge().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-merge
func (options *FontOptions) Merge(other *FontOptions) {
	C.cairo_font_options_merge(options.Ptr, other.Ptr)
	if err := options.status(); err != nil {
		panic(err)
	}
}

// See cairo_font_options_equal().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-equal
func (options *FontOptions) Equal(other *FontOptions) bool {
	ret := C.cairo_font_options_equal(options.Ptr, other.Ptr) != 0
	if err := options.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_font_options_hash().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-hash
func (options *FontOptions) Hash() uint32 {
	ret := uint32(C.cairo_font_options_hash(options.Ptr))
	if err := options.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_font_options_set_antialias().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-set-antialias
func (options *FontOptions) SetAntialias(antialias Antialias) {
	C.cairo_font_options_set_antialias(options.Ptr, C.cairo_antialias_t(antialias))
	if err := options.status(); err != nil {
		panic(err)
	}
}

// See cairo_font_options_get_antialias().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-get-antialias
func (options *FontOptions) GetAntialias() Antialias {
	ret := Antialias(C.cairo_font_options_get_antialias(options.Ptr))
	if err := options.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_font_options_set_subpixel_order().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-set-subpixel-order
func (options *FontOptions) SetSubpixelOrder(subpixelOrder SubpixelOrder) {
	C.cairo_font_options_set_subpixel_order(options.Ptr, C.cairo_subpixel_order_t(subpixelOrder))
	if err := options.status(); err != nil {
		panic(err)
	}
}

// See cairo_font_options_get_subpixel_order().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-get-subpixel-order
func (options *FontOptions) GetSubpixelOrder() SubpixelOrder {
	ret := SubpixelOrder(C.cairo_font_options_get_subpixel_order(options.Ptr))
	if err := options.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_font_options_set_hint_style().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-set-hint-style
func (options *FontOptions) SetHintStyle(hintStyle HintStyle) {
	C.cairo_font_options_set_hint_style(options.Ptr, C.cairo_hint_style_t(hintStyle))
	if err := options.status(); err != nil {
		panic(err)
	}
}

// See cairo_font_options_get_hint_style().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-get-hint-style
func (options *FontOptions) GetHintStyle() HintStyle {
	ret := HintStyle(C.cairo_font_options_get_hint_style(options.Ptr))
	if err := options.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_font_options_set_hint_metrics().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-set-hint-metrics
func (options *FontOptions) SetHintMetrics(hintMetrics HintMetrics) {
	C.cairo_font_options_set_hint_metrics(options.Ptr, C.cairo_hint_metrics_t(hintMetrics))
	if err := options.status(); err != nil {
		panic(err)
	}
}

// See cairo_font_options_get_hint_metrics().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-options-t.html#cairo-font-options-get-hint-metrics
func (options *FontOptions) GetHintMetrics() HintMetrics {
	ret := HintMetrics(C.cairo_font_options_get_hint_metrics(options.Ptr))
	if err := options.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_select_font_face().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-select-font-face
func (cr *Context) SelectFontFace(family string, slant FontSlant, weight FontWeight) {
	c_family := C.CString(family)
	defer C.free(unsafe.Pointer(c_family))
	C.cairo_select_font_face(cr.Ptr, c_family, C.cairo_font_slant_t(slant), C.cairo_font_weight_t(weight))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_font_size().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-set-font-size
func (cr *Context) SetFontSize(size float64) {
	C.cairo_set_font_size(cr.Ptr, C.double(size))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_font_matrix().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-set-font-matrix
func (cr *Context) SetFontMatrix(matrix *Matrix) {
	C.cairo_set_font_matrix(cr.Ptr, (*C.cairo_matrix_t)(unsafe.Pointer(matrix)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_get_font_matrix().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-get-font-matrix
func (cr *Context) GetFontMatrix(matrix *Matrix) {
	C.cairo_get_font_matrix(cr.Ptr, (*C.cairo_matrix_t)(unsafe.Pointer(matrix)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_font_options().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-set-font-options
func (cr *Context) SetFontOptions(options *FontOptions) {
	C.cairo_set_font_options(cr.Ptr, options.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_get_font_options().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-get-font-options
func (cr *Context) GetFontOptions(options *FontOptions) {
	C.cairo_get_font_options(cr.Ptr, options.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_set_font_face().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-set-font-face
func (cr *Context) SetFontFace(fontFace *FontFace) {
	C.cairo_set_font_face(cr.Ptr, fontFace.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_get_font_face().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-get-font-face
func (cr *Context) GetFontFace() *FontFace {
	ret := wrapFontFace(C.cairo_get_font_face(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_set_scaled_font().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-set-scaled-font
func (cr *Context) SetScaledFont(scaledFont *ScaledFont) {
	C.cairo_set_scaled_font(cr.Ptr, scaledFont.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_get_scaled_font().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-get-scaled-font
func (cr *Context) GetScaledFont() *ScaledFont {
	ret := wrapScaledFont(C.cairo_get_scaled_font(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_show_text().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-show-text
func (cr *Context) ShowText(utf8 string) {
	c_utf8 := C.CString(utf8)
	defer C.free(unsafe.Pointer(c_utf8))
	C.cairo_show_text(cr.Ptr, c_utf8)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_show_glyphs().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-show-glyphs
func (cr *Context) ShowGlyphs(glyphs []Glyph) {
	C.cairo_show_glyphs(cr.Ptr, (*C.cairo_glyph_t)(sliceBytes(unsafe.Pointer(&glyphs))), C.int(len(glyphs)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_text_path().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-text-path
func (cr *Context) TextPath(utf8 string) {
	c_utf8 := C.CString(utf8)
	defer C.free(unsafe.Pointer(c_utf8))
	C.cairo_text_path(cr.Ptr, c_utf8)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_glyph_path().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-glyph-path
func (cr *Context) GlyphPath(glyphs []Glyph) {
	C.cairo_glyph_path(cr.Ptr, (*C.cairo_glyph_t)(sliceBytes(unsafe.Pointer(&glyphs))), C.int(len(glyphs)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_text_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-text-extents
func (cr *Context) TextExtents(utf8 string, extents *TextExtents) {
	c_utf8 := C.CString(utf8)
	defer C.free(unsafe.Pointer(c_utf8))
	C.cairo_text_extents(cr.Ptr, c_utf8, (*C.cairo_text_extents_t)(unsafe.Pointer(extents)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_glyph_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-glyph-extents
func (cr *Context) GlyphExtents(glyphs []Glyph, extents *TextExtents) {
	C.cairo_glyph_extents(cr.Ptr, (*C.cairo_glyph_t)(sliceBytes(unsafe.Pointer(&glyphs))), C.int(len(glyphs)), (*C.cairo_text_extents_t)(unsafe.Pointer(extents)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_font_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-font-extents
func (cr *Context) FontExtents(extents *FontExtents) {
	C.cairo_font_extents(cr.Ptr, (*C.cairo_font_extents_t)(unsafe.Pointer(extents)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_font_face_status().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-face-t.html#cairo-font-face-status
func (fontFace *FontFace) status() error {
	ret := Status(C.cairo_font_face_status(fontFace.Ptr)).toError()
	return ret
}

// See cairo_font_type_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-face-t.html#cairo-font-type-t
type FontType int

const (
	FontTypeToy    FontType = C.CAIRO_FONT_TYPE_TOY
	FontTypeFt     FontType = C.CAIRO_FONT_TYPE_FT
	FontTypeWin32  FontType = C.CAIRO_FONT_TYPE_WIN32
	FontTypeQuartz FontType = C.CAIRO_FONT_TYPE_QUARTZ
	FontTypeUser   FontType = C.CAIRO_FONT_TYPE_USER
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i FontType) String() string {
	switch i {
	case FontTypeToy:
		return "FontTypeToy"
	case FontTypeFt:
		return "FontTypeFt"
	case FontTypeWin32:
		return "FontTypeWin32"
	case FontTypeQuartz:
		return "FontTypeQuartz"
	case FontTypeUser:
		return "FontTypeUser"
	default:
		return fmt.Sprintf("FontType(%d)", i)
	}
}

// See cairo_font_face_get_type().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-font-face-t.html#cairo-font-face-get-type
func (fontFace *FontFace) GetType() FontType {
	ret := FontType(C.cairo_font_face_get_type(fontFace.Ptr))
	if err := fontFace.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_scaled_font_create().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-create
func ScaledFontCreate(fontFace *FontFace, fontMatrix, ctm *Matrix, options *FontOptions) *ScaledFont {
	ret := wrapScaledFont(C.cairo_scaled_font_create(fontFace.Ptr, (*C.cairo_matrix_t)(unsafe.Pointer(fontMatrix)), (*C.cairo_matrix_t)(unsafe.Pointer(ctm)), options.Ptr))
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_scaled_font_status().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-status
func (scaledFont *ScaledFont) status() error {
	ret := Status(C.cairo_scaled_font_status(scaledFont.Ptr)).toError()
	return ret
}

// See cairo_scaled_font_get_type().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-get-type
func (scaledFont *ScaledFont) GetType() FontType {
	ret := FontType(C.cairo_scaled_font_get_type(scaledFont.Ptr))
	if err := scaledFont.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_scaled_font_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-extents
func (scaledFont *ScaledFont) Extents(extents *FontExtents) {
	C.cairo_scaled_font_extents(scaledFont.Ptr, (*C.cairo_font_extents_t)(unsafe.Pointer(extents)))
	if err := scaledFont.status(); err != nil {
		panic(err)
	}
}

// See cairo_scaled_font_text_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-text-extents
func (scaledFont *ScaledFont) TextExtents(utf8 string, extents *TextExtents) {
	c_utf8 := C.CString(utf8)
	defer C.free(unsafe.Pointer(c_utf8))
	C.cairo_scaled_font_text_extents(scaledFont.Ptr, c_utf8, (*C.cairo_text_extents_t)(unsafe.Pointer(extents)))
	if err := scaledFont.status(); err != nil {
		panic(err)
	}
}

// See cairo_scaled_font_glyph_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-glyph-extents
func (scaledFont *ScaledFont) GlyphExtents(glyphs []Glyph, extents *TextExtents) {
	C.cairo_scaled_font_glyph_extents(scaledFont.Ptr, (*C.cairo_glyph_t)(sliceBytes(unsafe.Pointer(&glyphs))), C.int(len(glyphs)), (*C.cairo_text_extents_t)(unsafe.Pointer(extents)))
	if err := scaledFont.status(); err != nil {
		panic(err)
	}
}

// See cairo_scaled_font_get_font_face().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-get-font-face
func (scaledFont *ScaledFont) GetFontFace() *FontFace {
	ret := wrapFontFace(C.cairo_scaled_font_get_font_face(scaledFont.Ptr))
	if err := scaledFont.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_scaled_font_get_font_matrix().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-get-font-matrix
func (scaledFont *ScaledFont) GetFontMatrix(fontMatrix *Matrix) {
	C.cairo_scaled_font_get_font_matrix(scaledFont.Ptr, (*C.cairo_matrix_t)(unsafe.Pointer(fontMatrix)))
	if err := scaledFont.status(); err != nil {
		panic(err)
	}
}

// See cairo_scaled_font_get_ctm().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-get-ctm
func (scaledFont *ScaledFont) GetCTM(ctm *Matrix) {
	C.cairo_scaled_font_get_ctm(scaledFont.Ptr, (*C.cairo_matrix_t)(unsafe.Pointer(ctm)))
	if err := scaledFont.status(); err != nil {
		panic(err)
	}
}

// See cairo_scaled_font_get_scale_matrix().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-get-scale-matrix
func (scaledFont *ScaledFont) GetScaleMatrix(scaleMatrix *Matrix) {
	C.cairo_scaled_font_get_scale_matrix(scaledFont.Ptr, (*C.cairo_matrix_t)(unsafe.Pointer(scaleMatrix)))
	if err := scaledFont.status(); err != nil {
		panic(err)
	}
}

// See cairo_scaled_font_get_font_options().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-scaled-font-get-font-options
func (scaledFont *ScaledFont) GetFontOptions(options *FontOptions) {
	C.cairo_scaled_font_get_font_options(scaledFont.Ptr, options.Ptr)
	if err := scaledFont.status(); err != nil {
		panic(err)
	}
}

// See cairo_toy_font_face_create().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-toy-font-face-create
func ToyFontFaceCreate(family string, slant FontSlant, weight FontWeight) *ToyFontFace {
	c_family := C.CString(family)
	defer C.free(unsafe.Pointer(c_family))
	ret := &ToyFontFace{wrapFontFace(C.cairo_toy_font_face_create(c_family, C.cairo_font_slant_t(slant), C.cairo_font_weight_t(weight)))}
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_toy_font_face_get_family().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-toy-font-face-get-family
func (fontFace *ToyFontFace) GetFamily() string {
	ret := C.GoString(C.cairo_toy_font_face_get_family(fontFace.Ptr))
	if err := fontFace.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_toy_font_face_get_slant().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-toy-font-face-get-slant
func (fontFace *ToyFontFace) GetSlant() FontSlant {
	ret := FontSlant(C.cairo_toy_font_face_get_slant(fontFace.Ptr))
	if err := fontFace.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_toy_font_face_get_weight().
//
// C API documentation: http://cairographics.org/manual/cairo-text.html#cairo-toy-font-face-get-weight
func (fontFace *ToyFontFace) GetWeight() FontWeight {
	ret := FontWeight(C.cairo_toy_font_face_get_weight(fontFace.Ptr))
	if err := fontFace.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_user_font_face_create().
//
// C API documentation: http://cairographics.org/manual/cairo-User-Fonts.html#cairo-user-font-face-create
func UserFontFaceCreate() *FontFace {
	ret := wrapFontFace(C.cairo_user_font_face_create())
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_operator().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-operator
func (cr *Context) GetOperator() Operator {
	ret := Operator(C.cairo_get_operator(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_source().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-source
func (cr *Context) GetSource() *Pattern {
	ret := wrapPattern(C.cairo_get_source(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_tolerance().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-tolerance
func (cr *Context) GetTolerance() float64 {
	ret := float64(C.cairo_get_tolerance(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_antialias().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-antialias
func (cr *Context) GetAntialias() Antialias {
	ret := Antialias(C.cairo_get_antialias(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_has_current_point().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-has-current-point
func (cr *Context) HasCurrentPoint() bool {
	ret := C.cairo_has_current_point(cr.Ptr) != 0
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_current_point().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-get-current-point
func (cr *Context) GetCurrentPoint() (float64, float64) {
	var x C.double
	var y C.double

	C.cairo_get_current_point(cr.Ptr, &x, &y)
	if err := cr.status(); err != nil {
		panic(err)
	}
	return float64(x), float64(y)
}

// See cairo_get_fill_rule().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-fill-rule
func (cr *Context) GetFillRule() FillRule {
	ret := FillRule(C.cairo_get_fill_rule(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_line_width().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-line-width
func (cr *Context) GetLineWidth() float64 {
	ret := float64(C.cairo_get_line_width(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_line_cap().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-line-cap
func (cr *Context) GetLineCap() LineCap {
	ret := LineCap(C.cairo_get_line_cap(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_line_join().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-line-join
func (cr *Context) GetLineJoin() LineJoin {
	ret := LineJoin(C.cairo_get_line_join(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_miter_limit().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-miter-limit
func (cr *Context) GetMiterLimit() float64 {
	ret := float64(C.cairo_get_miter_limit(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_dash_count().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-dash-count
func (cr *Context) GetDashCount() int {
	ret := int(C.cairo_get_dash_count(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_dash().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-dash
func (cr *Context) GetDash(dashes, offset *float64) {
	C.cairo_get_dash(cr.Ptr, (*C.double)(unsafe.Pointer(dashes)), (*C.double)(unsafe.Pointer(offset)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_get_matrix().
//
// C API documentation: http://cairographics.org/manual/cairo-Transformations.html#cairo-get-matrix
func (cr *Context) GetMatrix(matrix *Matrix) {
	C.cairo_get_matrix(cr.Ptr, (*C.cairo_matrix_t)(unsafe.Pointer(matrix)))
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_get_target().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-target
func (cr *Context) GetTarget() *Surface {
	ret := wrapSurface(C.cairo_get_target(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_get_group_target().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-get-group-target
func (cr *Context) GetGroupTarget() *Surface {
	ret := wrapSurface(C.cairo_get_group_target(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_path_data_type_t.
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-path-data-type-t
type PathDataType int

const (
	PathMoveTo    PathDataType = C.CAIRO_PATH_MOVE_TO
	PathLineTo    PathDataType = C.CAIRO_PATH_LINE_TO
	PathCurveTo   PathDataType = C.CAIRO_PATH_CURVE_TO
	PathClosePath PathDataType = C.CAIRO_PATH_CLOSE_PATH
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i PathDataType) String() string {
	switch i {
	case PathMoveTo:
		return "PathMoveTo"
	case PathLineTo:
		return "PathLineTo"
	case PathCurveTo:
		return "PathCurveTo"
	case PathClosePath:
		return "PathClosePath"
	default:
		return fmt.Sprintf("PathDataType(%d)", i)
	}
}

// See cairo_path_t.
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-path-t
type Path struct {
	Ptr *C.cairo_path_t
}

func freePath(obj *Path) {
	C.cairo_path_destroy(obj.Ptr)
}
func wrapPath(p *C.cairo_path_t) *Path {
	ret := &Path{p}
	runtime.SetFinalizer(ret, freePath)
	return ret
}

// Wrap a C cairo_path_t* found from some external source as a *Path.  The Go side will destroy the reference when it's no longer used.
func WrapPath(p unsafe.Pointer) *Path {
	return wrapPath((*C.cairo_path_t)(p))
}

// Construct a Path from a C cairo_path_t* found from some exernal source.  It is the caller's responsibility to ensure the pointer lives.
func BorrowPath(p unsafe.Pointer) *Path {
	return &Path{(*C.cairo_path_t)(p)}
}

// See cairo_copy_path().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-copy-path
func (cr *Context) CopyPath() *Path {
	ret := wrapPath(C.cairo_copy_path(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_copy_path_flat().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-copy-path-flat
func (cr *Context) CopyPathFlat() *Path {
	ret := wrapPath(C.cairo_copy_path_flat(cr.Ptr))
	if err := cr.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_append_path().
//
// C API documentation: http://cairographics.org/manual/cairo-Paths.html#cairo-append-path
func (cr *Context) AppendPath(path *Path) {
	C.cairo_append_path(cr.Ptr, path.Ptr)
	if err := cr.status(); err != nil {
		panic(err)
	}
}

// See cairo_status().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-t.html#cairo-status
func (cr *Context) status() error {
	ret := Status(C.cairo_status(cr.Ptr)).toError()
	return ret
}

// See cairo_device_type_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-type-t
type DeviceType int

const (
	DeviceTypeDRM     DeviceType = C.CAIRO_DEVICE_TYPE_DRM
	DeviceTypeGL      DeviceType = C.CAIRO_DEVICE_TYPE_GL
	DeviceTypeScript  DeviceType = C.CAIRO_DEVICE_TYPE_SCRIPT
	DeviceTypeXCB     DeviceType = C.CAIRO_DEVICE_TYPE_XCB
	DeviceTypeXlib    DeviceType = C.CAIRO_DEVICE_TYPE_XLIB
	DeviceTypeXML     DeviceType = C.CAIRO_DEVICE_TYPE_XML
	DeviceTypeCOGL    DeviceType = C.CAIRO_DEVICE_TYPE_COGL
	DeviceTypeWin32   DeviceType = C.CAIRO_DEVICE_TYPE_WIN32
	DeviceTypeInvalid DeviceType = C.CAIRO_DEVICE_TYPE_INVALID
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i DeviceType) String() string {
	switch i {
	case DeviceTypeDRM:
		return "DeviceTypeDRM"
	case DeviceTypeGL:
		return "DeviceTypeGL"
	case DeviceTypeScript:
		return "DeviceTypeScript"
	case DeviceTypeXCB:
		return "DeviceTypeXCB"
	case DeviceTypeXlib:
		return "DeviceTypeXlib"
	case DeviceTypeXML:
		return "DeviceTypeXML"
	case DeviceTypeCOGL:
		return "DeviceTypeCOGL"
	case DeviceTypeWin32:
		return "DeviceTypeWin32"
	case DeviceTypeInvalid:
		return "DeviceTypeInvalid"
	default:
		return fmt.Sprintf("DeviceType(%d)", i)
	}
}

// See cairo_device_get_type().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-get-type
func (device *Device) GetType() DeviceType {
	ret := DeviceType(C.cairo_device_get_type(device.Ptr))
	if err := device.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_device_status().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-status
func (device *Device) status() error {
	ret := Status(C.cairo_device_status(device.Ptr)).toError()
	return ret
}

// See cairo_device_acquire().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-acquire
func (device *Device) Acquire() error {
	ret := Status(C.cairo_device_acquire(device.Ptr)).toError()
	if err := device.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_device_release().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-release
func (device *Device) Release() {
	C.cairo_device_release(device.Ptr)
	if err := device.status(); err != nil {
		panic(err)
	}
}

// See cairo_device_flush().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-flush
func (device *Device) Flush() {
	C.cairo_device_flush(device.Ptr)
	if err := device.status(); err != nil {
		panic(err)
	}
}

// See cairo_device_finish().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-finish
func (device *Device) Finish() {
	C.cairo_device_finish(device.Ptr)
	if err := device.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_create_similar().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-create-similar
func (other *Surface) CreateSimilar(content Content, width, height int) *Surface {
	ret := wrapSurface(C.cairo_surface_create_similar(other.Ptr, C.cairo_content_t(content), C.int(width), C.int(height)))
	if err := other.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_surface_create_similar_image().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-create-similar-image
func (other *Surface) CreateSimilarImage(format Format, width, height int) *Surface {
	ret := wrapSurface(C.cairo_surface_create_similar_image(other.Ptr, C.cairo_format_t(format), C.int(width), C.int(height)))
	if err := other.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_surface_unmap_image().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-unmap-image
func (surface *Surface) UnmapImage(image *Surface) {
	C.cairo_surface_unmap_image(surface.Ptr, image.Ptr)
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_create_for_rectangle().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-create-for-rectangle
func (target *Surface) CreateForRectangle(x, y, width, height float64) *Surface {
	ret := wrapSurface(C.cairo_surface_create_for_rectangle(target.Ptr, C.double(x), C.double(y), C.double(width), C.double(height)))
	if err := target.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_surface_observer_mode_t.
type SurfaceObserverMode int

const (
	SurfaceObserverNormal           SurfaceObserverMode = C.CAIRO_SURFACE_OBSERVER_NORMAL
	SurfaceObserverRecordOperations SurfaceObserverMode = C.CAIRO_SURFACE_OBSERVER_RECORD_OPERATIONS
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i SurfaceObserverMode) String() string {
	switch i {
	case SurfaceObserverNormal:
		return "SurfaceObserverNormal"
	case SurfaceObserverRecordOperations:
		return "SurfaceObserverRecordOperations"
	default:
		return fmt.Sprintf("SurfaceObserverMode(%d)", i)
	}
}

// See cairo_surface_create_observer().
func (target *Surface) CreateObserver(mode SurfaceObserverMode) *SurfaceObserver {
	ret := &SurfaceObserver{wrapSurface(C.cairo_surface_create_observer(target.Ptr, C.cairo_surface_observer_mode_t(mode)))}
	if err := target.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_surface_observer_elapsed().
func (surface *SurfaceObserver) Elapsed() float64 {
	ret := float64(C.cairo_surface_observer_elapsed(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_device_observer_elapsed().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-observer-elapsed
func (device *Device) ObserverElapsed() float64 {
	ret := float64(C.cairo_device_observer_elapsed(device.Ptr))
	if err := device.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_device_observer_paint_elapsed().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-observer-paint-elapsed
func (device *Device) ObserverPaintElapsed() float64 {
	ret := float64(C.cairo_device_observer_paint_elapsed(device.Ptr))
	if err := device.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_device_observer_mask_elapsed().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-observer-mask-elapsed
func (device *Device) ObserverMaskElapsed() float64 {
	ret := float64(C.cairo_device_observer_mask_elapsed(device.Ptr))
	if err := device.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_device_observer_fill_elapsed().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-observer-fill-elapsed
func (device *Device) ObserverFillElapsed() float64 {
	ret := float64(C.cairo_device_observer_fill_elapsed(device.Ptr))
	if err := device.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_device_observer_stroke_elapsed().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-observer-stroke-elapsed
func (device *Device) ObserverStrokeElapsed() float64 {
	ret := float64(C.cairo_device_observer_stroke_elapsed(device.Ptr))
	if err := device.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_device_observer_glyphs_elapsed().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-device-t.html#cairo-device-observer-glyphs-elapsed
func (device *Device) ObserverGlyphsElapsed() float64 {
	ret := float64(C.cairo_device_observer_glyphs_elapsed(device.Ptr))
	if err := device.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_surface_finish().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-finish
func (surface *Surface) Finish() {
	C.cairo_surface_finish(surface.Ptr)
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_get_device().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-get-device
func (surface *Surface) GetDevice() *Device {
	ret := wrapDevice(C.cairo_surface_get_device(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_surface_status().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-status
func (surface *Surface) status() error {
	ret := Status(C.cairo_surface_status(surface.Ptr)).toError()
	return ret
}

// See cairo_surface_type_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-type-t
type SurfaceType int

const (
	SurfaceTypeImage         SurfaceType = C.CAIRO_SURFACE_TYPE_IMAGE
	SurfaceTypePDF           SurfaceType = C.CAIRO_SURFACE_TYPE_PDF
	SurfaceTypePS            SurfaceType = C.CAIRO_SURFACE_TYPE_PS
	SurfaceTypeXlib          SurfaceType = C.CAIRO_SURFACE_TYPE_XLIB
	SurfaceTypeXCB           SurfaceType = C.CAIRO_SURFACE_TYPE_XCB
	SurfaceTypeGlitz         SurfaceType = C.CAIRO_SURFACE_TYPE_GLITZ
	SurfaceTypeQuartz        SurfaceType = C.CAIRO_SURFACE_TYPE_QUARTZ
	SurfaceTypeWin32         SurfaceType = C.CAIRO_SURFACE_TYPE_WIN32
	SurfaceTypeBeos          SurfaceType = C.CAIRO_SURFACE_TYPE_BEOS
	SurfaceTypeDirectfb      SurfaceType = C.CAIRO_SURFACE_TYPE_DIRECTFB
	SurfaceTypeSVG           SurfaceType = C.CAIRO_SURFACE_TYPE_SVG
	SurfaceTypeOS2           SurfaceType = C.CAIRO_SURFACE_TYPE_OS2
	SurfaceTypeWin32Printing SurfaceType = C.CAIRO_SURFACE_TYPE_WIN32_PRINTING
	SurfaceTypeQuartzImage   SurfaceType = C.CAIRO_SURFACE_TYPE_QUARTZ_IMAGE
	SurfaceTypeScript        SurfaceType = C.CAIRO_SURFACE_TYPE_SCRIPT
	SurfaceTypeQt            SurfaceType = C.CAIRO_SURFACE_TYPE_QT
	SurfaceTypeRecording     SurfaceType = C.CAIRO_SURFACE_TYPE_RECORDING
	SurfaceTypeVG            SurfaceType = C.CAIRO_SURFACE_TYPE_VG
	SurfaceTypeGL            SurfaceType = C.CAIRO_SURFACE_TYPE_GL
	SurfaceTypeDRM           SurfaceType = C.CAIRO_SURFACE_TYPE_DRM
	SurfaceTypeTee           SurfaceType = C.CAIRO_SURFACE_TYPE_TEE
	SurfaceTypeXML           SurfaceType = C.CAIRO_SURFACE_TYPE_XML
	SurfaceTypeSkia          SurfaceType = C.CAIRO_SURFACE_TYPE_SKIA
	SurfaceTypeSubsurface    SurfaceType = C.CAIRO_SURFACE_TYPE_SUBSURFACE
	SurfaceTypeCOGL          SurfaceType = C.CAIRO_SURFACE_TYPE_COGL
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i SurfaceType) String() string {
	switch i {
	case SurfaceTypeImage:
		return "SurfaceTypeImage"
	case SurfaceTypePDF:
		return "SurfaceTypePDF"
	case SurfaceTypePS:
		return "SurfaceTypePS"
	case SurfaceTypeXlib:
		return "SurfaceTypeXlib"
	case SurfaceTypeXCB:
		return "SurfaceTypeXCB"
	case SurfaceTypeGlitz:
		return "SurfaceTypeGlitz"
	case SurfaceTypeQuartz:
		return "SurfaceTypeQuartz"
	case SurfaceTypeWin32:
		return "SurfaceTypeWin32"
	case SurfaceTypeBeos:
		return "SurfaceTypeBeos"
	case SurfaceTypeDirectfb:
		return "SurfaceTypeDirectfb"
	case SurfaceTypeSVG:
		return "SurfaceTypeSVG"
	case SurfaceTypeOS2:
		return "SurfaceTypeOS2"
	case SurfaceTypeWin32Printing:
		return "SurfaceTypeWin32Printing"
	case SurfaceTypeQuartzImage:
		return "SurfaceTypeQuartzImage"
	case SurfaceTypeScript:
		return "SurfaceTypeScript"
	case SurfaceTypeQt:
		return "SurfaceTypeQt"
	case SurfaceTypeRecording:
		return "SurfaceTypeRecording"
	case SurfaceTypeVG:
		return "SurfaceTypeVG"
	case SurfaceTypeGL:
		return "SurfaceTypeGL"
	case SurfaceTypeDRM:
		return "SurfaceTypeDRM"
	case SurfaceTypeTee:
		return "SurfaceTypeTee"
	case SurfaceTypeXML:
		return "SurfaceTypeXML"
	case SurfaceTypeSkia:
		return "SurfaceTypeSkia"
	case SurfaceTypeSubsurface:
		return "SurfaceTypeSubsurface"
	case SurfaceTypeCOGL:
		return "SurfaceTypeCOGL"
	default:
		return fmt.Sprintf("SurfaceType(%d)", i)
	}
}

// See cairo_surface_get_type().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-get-type
func (surface *Surface) GetType() SurfaceType {
	ret := SurfaceType(C.cairo_surface_get_type(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_surface_get_content().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-get-content
func (surface *Surface) GetContent() Content {
	ret := Content(C.cairo_surface_get_content(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_surface_supports_mime_type().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-supports-mime-type
func (surface *Surface) SupportsMimeType(mimeType string) bool {
	c_mimeType := C.CString(mimeType)
	defer C.free(unsafe.Pointer(c_mimeType))
	ret := C.cairo_surface_supports_mime_type(surface.Ptr, c_mimeType) != 0
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_surface_get_font_options().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-get-font-options
func (surface *Surface) GetFontOptions(options *FontOptions) {
	C.cairo_surface_get_font_options(surface.Ptr, options.Ptr)
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_flush().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-flush
func (surface *Surface) Flush() {
	C.cairo_surface_flush(surface.Ptr)
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_mark_dirty().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-mark-dirty
func (surface *Surface) MarkDirty() {
	C.cairo_surface_mark_dirty(surface.Ptr)
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_mark_dirty_rectangle().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-mark-dirty-rectangle
func (surface *Surface) MarkDirtyRectangle(x, y, width, height int) {
	C.cairo_surface_mark_dirty_rectangle(surface.Ptr, C.int(x), C.int(y), C.int(width), C.int(height))
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_set_device_scale().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-set-device-scale
func (surface *Surface) SetDeviceScale(xScale, yScale float64) {
	C.cairo_surface_set_device_scale(surface.Ptr, C.double(xScale), C.double(yScale))
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_get_device_scale().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-get-device-scale
func (surface *Surface) GetDeviceScale() (float64, float64) {
	var xScale C.double
	var yScale C.double

	C.cairo_surface_get_device_scale(surface.Ptr, &xScale, &yScale)
	if err := surface.status(); err != nil {
		panic(err)
	}
	return float64(xScale), float64(yScale)
}

// See cairo_surface_set_device_offset().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-set-device-offset
func (surface *Surface) SetDeviceOffset(xOffset, yOffset float64) {
	C.cairo_surface_set_device_offset(surface.Ptr, C.double(xOffset), C.double(yOffset))
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_get_device_offset().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-get-device-offset
func (surface *Surface) GetDeviceOffset() (float64, float64) {
	var xOffset C.double
	var yOffset C.double

	C.cairo_surface_get_device_offset(surface.Ptr, &xOffset, &yOffset)
	if err := surface.status(); err != nil {
		panic(err)
	}
	return float64(xOffset), float64(yOffset)
}

// See cairo_surface_set_fallback_resolution().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-set-fallback-resolution
func (surface *Surface) SetFallbackResolution(xPixelsPerInch, yPixelsPerInch float64) {
	C.cairo_surface_set_fallback_resolution(surface.Ptr, C.double(xPixelsPerInch), C.double(yPixelsPerInch))
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_get_fallback_resolution().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-get-fallback-resolution
func (surface *Surface) GetFallbackResolution() (float64, float64) {
	var xPixelsPerInch C.double
	var yPixelsPerInch C.double

	C.cairo_surface_get_fallback_resolution(surface.Ptr, &xPixelsPerInch, &yPixelsPerInch)
	if err := surface.status(); err != nil {
		panic(err)
	}
	return float64(xPixelsPerInch), float64(yPixelsPerInch)
}

// See cairo_surface_copy_page().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-copy-page
func (surface *Surface) CopyPage() {
	C.cairo_surface_copy_page(surface.Ptr)
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_show_page().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-show-page
func (surface *Surface) ShowPage() {
	C.cairo_surface_show_page(surface.Ptr)
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_surface_has_show_text_glyphs().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-surface-t.html#cairo-surface-has-show-text-glyphs
func (surface *Surface) HasShowTextGlyphs() bool {
	ret := C.cairo_surface_has_show_text_glyphs(surface.Ptr) != 0
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_image_surface_create().
//
// C API documentation: http://cairographics.org/manual/cairo-Image-Surfaces.html#cairo-image-surface-create
func ImageSurfaceCreate(format Format, width, height int) *ImageSurface {
	ret := &ImageSurface{wrapSurface(C.cairo_image_surface_create(C.cairo_format_t(format), C.int(width), C.int(height)))}
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_format_stride_for_width().
//
// C API documentation: http://cairographics.org/manual/cairo-Image-Surfaces.html#cairo-format-stride-for-width
func (format Format) StrideForWidth(width int) int {
	ret := int(C.cairo_format_stride_for_width(C.cairo_format_t(format), C.int(width)))
	return ret
}

// See cairo_image_surface_get_data().
//
// C API documentation: http://cairographics.org/manual/cairo-Image-Surfaces.html#cairo-image-surface-get-data
func (i *ImageSurface) Data() []byte {
	buf := C.cairo_image_surface_get_data(i.Ptr)
	return C.GoBytes(unsafe.Pointer(buf), C.int(i.GetStride()*i.GetHeight()))
}

// See cairo_image_surface_get_format().
//
// C API documentation: http://cairographics.org/manual/cairo-Image-Surfaces.html#cairo-image-surface-get-format
func (surface *ImageSurface) GetFormat() Format {
	ret := Format(C.cairo_image_surface_get_format(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_image_surface_get_width().
//
// C API documentation: http://cairographics.org/manual/cairo-Image-Surfaces.html#cairo-image-surface-get-width
func (surface *ImageSurface) GetWidth() int {
	ret := int(C.cairo_image_surface_get_width(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_image_surface_get_height().
//
// C API documentation: http://cairographics.org/manual/cairo-Image-Surfaces.html#cairo-image-surface-get-height
func (surface *ImageSurface) GetHeight() int {
	ret := int(C.cairo_image_surface_get_height(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_image_surface_get_stride().
//
// C API documentation: http://cairographics.org/manual/cairo-Image-Surfaces.html#cairo-image-surface-get-stride
func (surface *ImageSurface) GetStride() int {
	ret := int(C.cairo_image_surface_get_stride(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_recording_surface_create().
//
// C API documentation: http://cairographics.org/manual/cairo-Recording-Surfaces.html#cairo-recording-surface-create
func RecordingSurfaceCreate(content Content, extents *Rectangle) *RecordingSurface {
	ret := &RecordingSurface{wrapSurface(C.cairo_recording_surface_create(C.cairo_content_t(content), (*C.cairo_rectangle_t)(unsafe.Pointer(extents))))}
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_recording_surface_ink_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-Recording-Surfaces.html#cairo-recording-surface-ink-extents
func (surface *RecordingSurface) InkExtents() (float64, float64, float64, float64) {
	var x0 C.double
	var y0 C.double
	var width C.double
	var height C.double

	C.cairo_recording_surface_ink_extents(surface.Ptr, &x0, &y0, &width, &height)
	if err := surface.status(); err != nil {
		panic(err)
	}
	return float64(x0), float64(y0), float64(width), float64(height)
}

// See cairo_recording_surface_get_extents().
//
// C API documentation: http://cairographics.org/manual/cairo-Recording-Surfaces.html#cairo-recording-surface-get-extents
func (surface *RecordingSurface) GetExtents(extents *Rectangle) bool {
	ret := C.cairo_recording_surface_get_extents(surface.Ptr, (*C.cairo_rectangle_t)(unsafe.Pointer(extents))) != 0
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_pattern_create_rgb().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-create-rgb
func PatternCreateRGB(red, green, blue float64) *Pattern {
	ret := wrapPattern(C.cairo_pattern_create_rgb(C.double(red), C.double(green), C.double(blue)))
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_pattern_create_rgba().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-create-rgba
func PatternCreateRGBA(red, green, blue, alpha float64) *Pattern {
	ret := wrapPattern(C.cairo_pattern_create_rgba(C.double(red), C.double(green), C.double(blue), C.double(alpha)))
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_pattern_create_for_surface().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-create-for-surface
func PatternCreateForSurface(surface *Surface) *Pattern {
	ret := wrapPattern(C.cairo_pattern_create_for_surface(surface.Ptr))
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_pattern_create_linear().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-create-linear
func PatternCreateLinear(x0, y0, x1, y1 float64) *Pattern {
	ret := wrapPattern(C.cairo_pattern_create_linear(C.double(x0), C.double(y0), C.double(x1), C.double(y1)))
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_pattern_create_radial().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-create-radial
func PatternCreateRadial(cx0, cy0, radius0, cx1, cy1, radius1 float64) *Pattern {
	ret := wrapPattern(C.cairo_pattern_create_radial(C.double(cx0), C.double(cy0), C.double(radius0), C.double(cx1), C.double(cy1), C.double(radius1)))
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_pattern_create_mesh().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-create-mesh
func PatternCreateMesh() *Pattern {
	ret := wrapPattern(C.cairo_pattern_create_mesh())
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_pattern_status().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-status
func (pattern *Pattern) status() error {
	ret := Status(C.cairo_pattern_status(pattern.Ptr)).toError()
	return ret
}

// See cairo_pattern_type_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-type-t
type PatternType int

const (
	PatternTypeSolid        PatternType = C.CAIRO_PATTERN_TYPE_SOLID
	PatternTypeSurface      PatternType = C.CAIRO_PATTERN_TYPE_SURFACE
	PatternTypeLinear       PatternType = C.CAIRO_PATTERN_TYPE_LINEAR
	PatternTypeRadial       PatternType = C.CAIRO_PATTERN_TYPE_RADIAL
	PatternTypeMesh         PatternType = C.CAIRO_PATTERN_TYPE_MESH
	PatternTypeRasterSource PatternType = C.CAIRO_PATTERN_TYPE_RASTER_SOURCE
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i PatternType) String() string {
	switch i {
	case PatternTypeSolid:
		return "PatternTypeSolid"
	case PatternTypeSurface:
		return "PatternTypeSurface"
	case PatternTypeLinear:
		return "PatternTypeLinear"
	case PatternTypeRadial:
		return "PatternTypeRadial"
	case PatternTypeMesh:
		return "PatternTypeMesh"
	case PatternTypeRasterSource:
		return "PatternTypeRasterSource"
	default:
		return fmt.Sprintf("PatternType(%d)", i)
	}
}

// See cairo_pattern_get_type().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-get-type
func (pattern *Pattern) GetType() PatternType {
	ret := PatternType(C.cairo_pattern_get_type(pattern.Ptr))
	if err := pattern.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_pattern_add_color_stop_rgb().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-add-color-stop-rgb
func (pattern *Pattern) AddColorStopRGB(offset, red, green, blue float64) {
	C.cairo_pattern_add_color_stop_rgb(pattern.Ptr, C.double(offset), C.double(red), C.double(green), C.double(blue))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_pattern_add_color_stop_rgba().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-add-color-stop-rgba
func (pattern *Pattern) AddColorStopRGBA(offset, red, green, blue, alpha float64) {
	C.cairo_pattern_add_color_stop_rgba(pattern.Ptr, C.double(offset), C.double(red), C.double(green), C.double(blue), C.double(alpha))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_mesh_pattern_begin_patch().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-mesh-pattern-begin-patch
func (pattern *MeshPattern) BeginPatch() {
	C.cairo_mesh_pattern_begin_patch(pattern.Ptr)
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_mesh_pattern_end_patch().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-mesh-pattern-end-patch
func (pattern *MeshPattern) EndPatch() {
	C.cairo_mesh_pattern_end_patch(pattern.Ptr)
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_mesh_pattern_curve_to().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-mesh-pattern-curve-to
func (pattern *MeshPattern) CurveTo(x1, y1, x2, y2, x3, y3 float64) {
	C.cairo_mesh_pattern_curve_to(pattern.Ptr, C.double(x1), C.double(y1), C.double(x2), C.double(y2), C.double(x3), C.double(y3))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_mesh_pattern_line_to().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-mesh-pattern-line-to
func (pattern *MeshPattern) LineTo(x, y float64) {
	C.cairo_mesh_pattern_line_to(pattern.Ptr, C.double(x), C.double(y))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_mesh_pattern_move_to().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-mesh-pattern-move-to
func (pattern *MeshPattern) MoveTo(x, y float64) {
	C.cairo_mesh_pattern_move_to(pattern.Ptr, C.double(x), C.double(y))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_mesh_pattern_set_control_point().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-mesh-pattern-set-control-point
func (pattern *MeshPattern) SetControlPoint(pointNum int, x, y float64) {
	C.cairo_mesh_pattern_set_control_point(pattern.Ptr, C.uint(pointNum), C.double(x), C.double(y))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_mesh_pattern_set_corner_color_rgb().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-mesh-pattern-set-corner-color-rgb
func (pattern *MeshPattern) SetCornerColorRGB(cornerNum int, red, green, blue float64) {
	C.cairo_mesh_pattern_set_corner_color_rgb(pattern.Ptr, C.uint(cornerNum), C.double(red), C.double(green), C.double(blue))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_mesh_pattern_set_corner_color_rgba().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-mesh-pattern-set-corner-color-rgba
func (pattern *MeshPattern) SetCornerColorRGBA(cornerNum int, red, green, blue, alpha float64) {
	C.cairo_mesh_pattern_set_corner_color_rgba(pattern.Ptr, C.uint(cornerNum), C.double(red), C.double(green), C.double(blue), C.double(alpha))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_pattern_set_matrix().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-set-matrix
func (pattern *Pattern) SetMatrix(matrix *Matrix) {
	C.cairo_pattern_set_matrix(pattern.Ptr, (*C.cairo_matrix_t)(unsafe.Pointer(matrix)))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_pattern_get_matrix().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-get-matrix
func (pattern *Pattern) GetMatrix(matrix *Matrix) {
	C.cairo_pattern_get_matrix(pattern.Ptr, (*C.cairo_matrix_t)(unsafe.Pointer(matrix)))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_extend_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-extend-t
type Extend int

const (
	ExtendNone    Extend = C.CAIRO_EXTEND_NONE
	ExtendRepeat  Extend = C.CAIRO_EXTEND_REPEAT
	ExtendReflect Extend = C.CAIRO_EXTEND_REFLECT
	ExtendPad     Extend = C.CAIRO_EXTEND_PAD
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i Extend) String() string {
	switch i {
	case ExtendNone:
		return "ExtendNone"
	case ExtendRepeat:
		return "ExtendRepeat"
	case ExtendReflect:
		return "ExtendReflect"
	case ExtendPad:
		return "ExtendPad"
	default:
		return fmt.Sprintf("Extend(%d)", i)
	}
}

// See cairo_pattern_set_extend().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-set-extend
func (pattern *Pattern) SetExtend(extend Extend) {
	C.cairo_pattern_set_extend(pattern.Ptr, C.cairo_extend_t(extend))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_pattern_get_extend().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-get-extend
func (pattern *Pattern) GetExtend() Extend {
	ret := Extend(C.cairo_pattern_get_extend(pattern.Ptr))
	if err := pattern.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_filter_t.
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-filter-t
type Filter int

const (
	FilterFast     Filter = C.CAIRO_FILTER_FAST
	FilterGood     Filter = C.CAIRO_FILTER_GOOD
	FilterBest     Filter = C.CAIRO_FILTER_BEST
	FilterNearest  Filter = C.CAIRO_FILTER_NEAREST
	FilterBilinear Filter = C.CAIRO_FILTER_BILINEAR
	FilterGaussian Filter = C.CAIRO_FILTER_GAUSSIAN
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i Filter) String() string {
	switch i {
	case FilterFast:
		return "FilterFast"
	case FilterGood:
		return "FilterGood"
	case FilterBest:
		return "FilterBest"
	case FilterNearest:
		return "FilterNearest"
	case FilterBilinear:
		return "FilterBilinear"
	case FilterGaussian:
		return "FilterGaussian"
	default:
		return fmt.Sprintf("Filter(%d)", i)
	}
}

// See cairo_pattern_set_filter().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-set-filter
func (pattern *Pattern) SetFilter(filter Filter) {
	C.cairo_pattern_set_filter(pattern.Ptr, C.cairo_filter_t(filter))
	if err := pattern.status(); err != nil {
		panic(err)
	}
}

// See cairo_pattern_get_filter().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-pattern-get-filter
func (pattern *Pattern) GetFilter() Filter {
	ret := Filter(C.cairo_pattern_get_filter(pattern.Ptr))
	if err := pattern.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_mesh_pattern_get_path().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-pattern-t.html#cairo-mesh-pattern-get-path
func (pattern *MeshPattern) GetPath(patchNum int) *Path {
	ret := wrapPath(C.cairo_mesh_pattern_get_path(pattern.Ptr, C.uint(patchNum)))
	if err := pattern.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_matrix_init_identity().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-init-identity
func (matrix *Matrix) InitIdentity() {
	C.cairo_matrix_init_identity((*C.cairo_matrix_t)(unsafe.Pointer(matrix)))
}

// See cairo_matrix_init_translate().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-init-translate
func (matrix *Matrix) InitTranslate(tx, ty float64) {
	C.cairo_matrix_init_translate((*C.cairo_matrix_t)(unsafe.Pointer(matrix)), C.double(tx), C.double(ty))
}

// See cairo_matrix_init_scale().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-init-scale
func (matrix *Matrix) InitScale(sx, sy float64) {
	C.cairo_matrix_init_scale((*C.cairo_matrix_t)(unsafe.Pointer(matrix)), C.double(sx), C.double(sy))
}

// See cairo_matrix_init_rotate().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-init-rotate
func (matrix *Matrix) InitRotate(radians float64) {
	C.cairo_matrix_init_rotate((*C.cairo_matrix_t)(unsafe.Pointer(matrix)), C.double(radians))
}

// See cairo_matrix_translate().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-translate
func (matrix *Matrix) Translate(tx, ty float64) {
	C.cairo_matrix_translate((*C.cairo_matrix_t)(unsafe.Pointer(matrix)), C.double(tx), C.double(ty))
}

// See cairo_matrix_scale().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-scale
func (matrix *Matrix) Scale(sx, sy float64) {
	C.cairo_matrix_scale((*C.cairo_matrix_t)(unsafe.Pointer(matrix)), C.double(sx), C.double(sy))
}

// See cairo_matrix_rotate().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-rotate
func (matrix *Matrix) Rotate(radians float64) {
	C.cairo_matrix_rotate((*C.cairo_matrix_t)(unsafe.Pointer(matrix)), C.double(radians))
}

// See cairo_matrix_invert().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-invert
func (matrix *Matrix) Invert() error {
	ret := Status(C.cairo_matrix_invert((*C.cairo_matrix_t)(unsafe.Pointer(matrix)))).toError()
	return ret
}

// See cairo_matrix_multiply().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-multiply
func (result *Matrix) Multiply(a, b *Matrix) {
	C.cairo_matrix_multiply((*C.cairo_matrix_t)(unsafe.Pointer(result)), (*C.cairo_matrix_t)(unsafe.Pointer(a)), (*C.cairo_matrix_t)(unsafe.Pointer(b)))
}

// See cairo_matrix_transform_distance().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-transform-distance
func (matrix *Matrix) TransformDistance(dx, dy *float64) {
	C.cairo_matrix_transform_distance((*C.cairo_matrix_t)(unsafe.Pointer(matrix)), (*C.double)(unsafe.Pointer(dx)), (*C.double)(unsafe.Pointer(dy)))
}

// See cairo_matrix_transform_point().
//
// C API documentation: http://cairographics.org/manual/cairo-cairo-matrix-t.html#cairo-matrix-transform-point
func (matrix *Matrix) TransformPoint(x, y *float64) {
	C.cairo_matrix_transform_point((*C.cairo_matrix_t)(unsafe.Pointer(matrix)), (*C.double)(unsafe.Pointer(x)), (*C.double)(unsafe.Pointer(y)))
}

// See cairo_region_t.
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-t
type Region struct {
	Ptr *C.cairo_region_t
}

func freeRegion(obj *Region) {
	C.cairo_region_destroy(obj.Ptr)
}
func wrapRegion(p *C.cairo_region_t) *Region {
	ret := &Region{p}
	runtime.SetFinalizer(ret, freeRegion)
	return ret
}

// Wrap a C cairo_region_t* found from some external source as a *Region.  The Go side will destroy the reference when it's no longer used.
func WrapRegion(p unsafe.Pointer) *Region {
	return wrapRegion((*C.cairo_region_t)(p))
}

// Construct a Region from a C cairo_region_t* found from some exernal source.  It is the caller's responsibility to ensure the pointer lives.
func BorrowRegion(p unsafe.Pointer) *Region {
	return &Region{(*C.cairo_region_t)(p)}
}

// See cairo_region_overlap_t.
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-overlap-t
type RegionOverlap int

const (
	RegionOverlapIn   RegionOverlap = C.CAIRO_REGION_OVERLAP_IN
	RegionOverlapOut  RegionOverlap = C.CAIRO_REGION_OVERLAP_OUT
	RegionOverlapPart RegionOverlap = C.CAIRO_REGION_OVERLAP_PART
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i RegionOverlap) String() string {
	switch i {
	case RegionOverlapIn:
		return "RegionOverlapIn"
	case RegionOverlapOut:
		return "RegionOverlapOut"
	case RegionOverlapPart:
		return "RegionOverlapPart"
	default:
		return fmt.Sprintf("RegionOverlap(%d)", i)
	}
}

// See cairo_region_create().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-create
func RegionCreate() *Region {
	ret := wrapRegion(C.cairo_region_create())
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_region_copy().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-copy
func (original *Region) Copy() *Region {
	ret := wrapRegion(C.cairo_region_copy(original.Ptr))
	if err := original.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_region_equal().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-equal
func (a *Region) Equal(b *Region) bool {
	ret := C.cairo_region_equal(a.Ptr, b.Ptr) != 0
	if err := a.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_region_status().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-status
func (region *Region) status() error {
	ret := Status(C.cairo_region_status(region.Ptr)).toError()
	return ret
}

// See cairo_region_num_rectangles().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-num-rectangles
func (region *Region) NumRectangles() int {
	ret := int(C.cairo_region_num_rectangles(region.Ptr))
	if err := region.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_region_is_empty().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-is-empty
func (region *Region) IsEmpty() bool {
	ret := C.cairo_region_is_empty(region.Ptr) != 0
	if err := region.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_region_contains_point().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-contains-point
func (region *Region) ContainsPoint(x, y int) bool {
	ret := C.cairo_region_contains_point(region.Ptr, C.int(x), C.int(y)) != 0
	if err := region.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_region_translate().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-translate
func (region *Region) Translate(dx, dy int) {
	C.cairo_region_translate(region.Ptr, C.int(dx), C.int(dy))
	if err := region.status(); err != nil {
		panic(err)
	}
}

// See cairo_region_subtract().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-subtract
func (dst *Region) Subtract(other *Region) error {
	ret := Status(C.cairo_region_subtract(dst.Ptr, other.Ptr)).toError()
	if err := dst.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_region_intersect().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-intersect
func (dst *Region) Intersect(other *Region) error {
	ret := Status(C.cairo_region_intersect(dst.Ptr, other.Ptr)).toError()
	if err := dst.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_region_union().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-union
func (dst *Region) Union(other *Region) error {
	ret := Status(C.cairo_region_union(dst.Ptr, other.Ptr)).toError()
	if err := dst.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_region_xor().
//
// C API documentation: http://cairographics.org/manual/cairo-Regions.html#cairo-region-xor
func (dst *Region) XOR(other *Region) error {
	ret := Status(C.cairo_region_xor(dst.Ptr, other.Ptr)).toError()
	if err := dst.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_svg_version_t.
//
// C API documentation: http://cairographics.org/manual/cairo-SVG-Surfaces.html#cairo-svg-version-t
type SVGVersion int

const (
	SVGVersion11 SVGVersion = C.CAIRO_SVG_VERSION_1_1
	SVGVersion12 SVGVersion = C.CAIRO_SVG_VERSION_1_2
)

// String implements the Stringer interface, which is used in places like fmt's %q.  For all enums like this it returns the Go name of the constant.
func (i SVGVersion) String() string {
	switch i {
	case SVGVersion11:
		return "SVGVersion11"
	case SVGVersion12:
		return "SVGVersion12"
	default:
		return fmt.Sprintf("SVGVersion(%d)", i)
	}
}

// See cairo_svg_surface_create().
//
// C API documentation: http://cairographics.org/manual/cairo-SVG-Surfaces.html#cairo-svg-surface-create
func SVGSurfaceCreate(filename string, widthInPoints, heightInPoints float64) *SVGSurface {
	c_filename := C.CString(filename)
	defer C.free(unsafe.Pointer(c_filename))
	ret := &SVGSurface{wrapSurface(C.cairo_svg_surface_create(c_filename, C.double(widthInPoints), C.double(heightInPoints)))}
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_svg_surface_restrict_to_version().
//
// C API documentation: http://cairographics.org/manual/cairo-SVG-Surfaces.html#cairo-svg-surface-restrict-to-version
func (surface *SVGSurface) RestrictToVersion(version SVGVersion) {
	C.cairo_svg_surface_restrict_to_version(surface.Ptr, C.cairo_svg_version_t(version))
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_svg_get_versions().
//
// C API documentation: http://cairographics.org/manual/cairo-SVG-Surfaces.html#cairo-svg-get-versions
func SVGGetVersions() []SVGVersion {
	var cVersionsPtr *C.cairo_svg_version_t
	var cNumVersions C.int
	C.cairo_svg_get_versions(&cVersionsPtr, &cNumVersions)
	slice := (*[1 << 30]C.cairo_svg_version_t)(unsafe.Pointer(cVersionsPtr))[:cNumVersions:cNumVersions]
	versions := make([]SVGVersion, cNumVersions)
	for i := 0; i < int(cNumVersions); i++ {
		versions[i] = SVGVersion(slice[i])
	}
	return versions
}

// See cairo_svg_version_to_string().
//
// C API documentation: http://cairographics.org/manual/cairo-SVG-Surfaces.html#cairo-svg-version-to-string
func (version SVGVersion) ToString() string {
	ret := C.GoString(C.cairo_svg_version_to_string(C.cairo_svg_version_t(version)))
	return ret
}

// See cairo_xlib_surface_create().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-surface-create
func XlibSurfaceCreate(dpy unsafe.Pointer, drawable uint64, visual unsafe.Pointer, width, height int) *XlibSurface {
	ret := &XlibSurface{wrapSurface(C.cairo_xlib_surface_create((*C.Display)(dpy), C.Drawable(drawable), (*C.Visual)(visual), C.int(width), C.int(height)))}
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_xlib_surface_create_for_bitmap().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-surface-create-for-bitmap
func XlibSurfaceCreateForBitmap(dpy unsafe.Pointer, bitmap uint64, screen unsafe.Pointer, width, height int) *XlibSurface {
	ret := &XlibSurface{wrapSurface(C.cairo_xlib_surface_create_for_bitmap((*C.Display)(dpy), C.Pixmap(bitmap), (*C.Screen)(screen), C.int(width), C.int(height)))}
	if err := ret.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_xlib_surface_set_size().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-surface-set-size
func (surface *XlibSurface) SetSize(width, height int) {
	C.cairo_xlib_surface_set_size(surface.Ptr, C.int(width), C.int(height))
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_xlib_surface_set_drawable().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-surface-set-drawable
func (surface *XlibSurface) SetDrawable(drawable uint64, width, height int) {
	C.cairo_xlib_surface_set_drawable(surface.Ptr, C.Drawable(drawable), C.int(width), C.int(height))
	if err := surface.status(); err != nil {
		panic(err)
	}
}

// See cairo_xlib_surface_get_display().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-surface-get-display
func (surface *XlibSurface) GetDisplay() unsafe.Pointer {
	ret := unsafe.Pointer(C.cairo_xlib_surface_get_display(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_xlib_surface_get_drawable().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-surface-get-drawable
func (surface *XlibSurface) GetDrawable() uint64 {
	ret := uint64(C.cairo_xlib_surface_get_drawable(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_xlib_surface_get_screen().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-surface-get-screen
func (surface *XlibSurface) GetScreen() unsafe.Pointer {
	ret := unsafe.Pointer(C.cairo_xlib_surface_get_screen(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_xlib_surface_get_visual().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-surface-get-visual
func (surface *XlibSurface) GetVisual() unsafe.Pointer {
	ret := unsafe.Pointer(C.cairo_xlib_surface_get_visual(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_xlib_surface_get_depth().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-surface-get-depth
func (surface *XlibSurface) GetDepth() int {
	ret := int(C.cairo_xlib_surface_get_depth(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_xlib_surface_get_width().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-surface-get-width
func (surface *XlibSurface) GetWidth() int {
	ret := int(C.cairo_xlib_surface_get_width(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_xlib_surface_get_height().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-surface-get-height
func (surface *XlibSurface) GetHeight() int {
	ret := int(C.cairo_xlib_surface_get_height(surface.Ptr))
	if err := surface.status(); err != nil {
		panic(err)
	}
	return ret
}

// See cairo_xlib_device_debug_cap_xrender_version().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-device-debug-cap-xrender-version
func (device *XlibDevice) DebugCapXrenderVersion(majorVersion, minorVersion int) {
	C.cairo_xlib_device_debug_cap_xrender_version(device.Ptr, C.int(majorVersion), C.int(minorVersion))
	if err := device.status(); err != nil {
		panic(err)
	}
}

// See cairo_xlib_device_debug_set_precision().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-device-debug-set-precision
func (device *XlibDevice) DebugSetPrecision(precision int) {
	C.cairo_xlib_device_debug_set_precision(device.Ptr, C.int(precision))
	if err := device.status(); err != nil {
		panic(err)
	}
}

// See cairo_xlib_device_debug_get_precision().
//
// C API documentation: http://cairographics.org/manual/cairo-XLib-Surfaces.html#cairo-xlib-device-debug-get-precision
func (device *XlibDevice) DebugGetPrecision() int {
	ret := int(C.cairo_xlib_device_debug_get_precision(device.Ptr))
	if err := device.status(); err != nil {
		panic(err)
	}
	return ret
}
