///
/// @file  Sieve_arrays.hpp
/// @brief Static lookup tables needed by Sieve.cpp.
///
/// Copyright (C) 2025 Kim Walisch, <kim.walisch@gmail.com>
///
/// This file is distributed under the BSD License. See the COPYING
/// file in the top level directory.
///

#ifndef SIEVE_ARRAYS_HPP
#define SIEVE_ARRAYS_HPP

#include <Sieve.hpp>
#include <Vector.hpp>

#include <stdint.h>
#include <initializer_list>

namespace {

struct WheelInit
{
  uint8_t factor;
  uint8_t index;
};

/// Categorize sieving primes according to their modulo 30
/// congruence class { 1, 7, 11, 13, 17, 19, 23, 29 }.
///
const primecount::Array<uint8_t, 30> wheel_offsets =
{
  0, 8 * 0, 0, 0, 0, 0,
  0, 8 * 1, 0, 0, 0, 8 * 2,
  0, 8 * 3, 0, 0, 0, 8 * 4,
  0, 8 * 5, 0, 0, 0, 8 * 6,
  0, 0,     0, 0, 0, 8 * 7
};

/// Used to calculate the first multiple > start of a
/// sieving prime that is coprime to 2, 3, 5.
///
const primecount::Array<WheelInit, 30> wheel_init
{{
  {1,  0}, {0,  0}, {5,  1}, {4,  1}, {3,  1},
  {2,  1}, {1,  1}, {0,  1}, {3,  2}, {2,  2},
  {1,  2}, {0,  2}, {1,  3}, {0,  3}, {3,  4},
  {2,  4}, {1,  4}, {0,  4}, {1,  5}, {0,  5},
  {3,  6}, {2,  6}, {1,  6}, {0,  6}, {5,  7},
  {4,  7}, {3,  7}, {2,  7}, {1,  7}, {0,  7}
}};

/// The 8 bits in each byte of the sieve array correspond
/// to the offsets { 1, 7, 11, 13, 17, 19, 23, 29 }.
///
constexpr int left_shift(int n)
{
  return   (n % 30 <= 1)  ? (n / 30 * 8) + 0
         : (n % 30 <= 7)  ? (n / 30 * 8) + 1
         : (n % 30 <= 11) ? (n / 30 * 8) + 2
         : (n % 30 <= 13) ? (n / 30 * 8) + 3
         : (n % 30 <= 17) ? (n / 30 * 8) + 4
         : (n % 30 <= 19) ? (n / 30 * 8) + 5
         : (n % 30 <= 23) ? (n / 30 * 8) + 6
         : (n / 30 * 8) + 7;
}

/// The 8 bits in each byte of the sieve array correspond
/// to the offsets { 1, 7, 11, 13, 17, 19, 23, 29 }.
///
constexpr int right_shift(int n)
{
  return   (n % 30 >= 29) ? 56 - (n / 30 * 8)
         : (n % 30 >= 23) ? 57 - (n / 30 * 8)
         : (n % 30 >= 19) ? 58 - (n / 30 * 8)
         : (n % 30 >= 17) ? 59 - (n / 30 * 8)
         : (n % 30 >= 13) ? 60 - (n / 30 * 8)
         : (n % 30 >= 11) ? 61 - (n / 30 * 8)
         : (n % 30 >= 7)  ? 62 - (n / 30 * 8)
         : (n % 30 >= 1)  ? 63 - (n / 30 * 8)
         : 64 - (n / 30 * 8);
}

#if defined(__BYTE_ORDER__) && \
    defined(__ORDER_BIG_ENDIAN__) && \
    __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__

/// Since the sieve array is a byte array but the
/// unset_smaller and unset_larger arrays are of
/// type uint64_t we need to reverse the byte order
/// of the unset_smaller and unset_larger arrays
/// on big endian CPU architectures.

/// Reverse byte order of 64-bit integer
constexpr uint64_t bswap64(uint64_t i, uint64_t j = 0, uint64_t n = 0)
{
  return (n == sizeof(uint64_t)) ? j :
    bswap64(i >> 8, (j << 8) | (i & 0xff), n + 1);
}

/// Bitmask to unset bits <= n
constexpr uint64_t unset_s(int n)
{
  return bswap64(~0ull << left_shift(n));
}

/// Bitmask to unset bits >= n
constexpr uint64_t unset_l(int n)
{
  return bswap64((n == 0) ? 0 : ~0ull >> right_shift(n));
}

#else

/// Bitmask to unset bits <= n
constexpr uint64_t unset_s(int n)
{
  return ~0ull << left_shift(n);
}

/// Bitmask to unset bits >= n
constexpr uint64_t unset_l(int n)
{
  return (n == 0) ? 0 : ~0ull >> right_shift(n);
}

#endif

} // namespace

namespace primecount {

/// Unset bits < start
const Array<uint64_t, 240> Sieve::unset_smaller =
{
  unset_s(0), unset_s(1), unset_s(2), unset_s(3), unset_s(4),
  unset_s(5), unset_s(6), unset_s(7), unset_s(8), unset_s(9),
  unset_s(10), unset_s(11), unset_s(12), unset_s(13), unset_s(14),
  unset_s(15), unset_s(16), unset_s(17), unset_s(18), unset_s(19),
  unset_s(20), unset_s(21), unset_s(22), unset_s(23), unset_s(24),
  unset_s(25), unset_s(26), unset_s(27), unset_s(28), unset_s(29),
  unset_s(30), unset_s(31), unset_s(32), unset_s(33), unset_s(34),
  unset_s(35), unset_s(36), unset_s(37), unset_s(38), unset_s(39),
  unset_s(40), unset_s(41), unset_s(42), unset_s(43), unset_s(44),
  unset_s(45), unset_s(46), unset_s(47), unset_s(48), unset_s(49),
  unset_s(50), unset_s(51), unset_s(52), unset_s(53), unset_s(54),
  unset_s(55), unset_s(56), unset_s(57), unset_s(58), unset_s(59),
  unset_s(60), unset_s(61), unset_s(62), unset_s(63), unset_s(64),
  unset_s(65), unset_s(66), unset_s(67), unset_s(68), unset_s(69),
  unset_s(70), unset_s(71), unset_s(72), unset_s(73), unset_s(74),
  unset_s(75), unset_s(76), unset_s(77), unset_s(78), unset_s(79),
  unset_s(80), unset_s(81), unset_s(82), unset_s(83), unset_s(84),
  unset_s(85), unset_s(86), unset_s(87), unset_s(88), unset_s(89),
  unset_s(90), unset_s(91), unset_s(92), unset_s(93), unset_s(94),
  unset_s(95), unset_s(96), unset_s(97), unset_s(98), unset_s(99),
  unset_s(100), unset_s(101), unset_s(102), unset_s(103), unset_s(104),
  unset_s(105), unset_s(106), unset_s(107), unset_s(108), unset_s(109),
  unset_s(110), unset_s(111), unset_s(112), unset_s(113), unset_s(114),
  unset_s(115), unset_s(116), unset_s(117), unset_s(118), unset_s(119),
  unset_s(120), unset_s(121), unset_s(122), unset_s(123), unset_s(124),
  unset_s(125), unset_s(126), unset_s(127), unset_s(128), unset_s(129),
  unset_s(130), unset_s(131), unset_s(132), unset_s(133), unset_s(134),
  unset_s(135), unset_s(136), unset_s(137), unset_s(138), unset_s(139),
  unset_s(140), unset_s(141), unset_s(142), unset_s(143), unset_s(144),
  unset_s(145), unset_s(146), unset_s(147), unset_s(148), unset_s(149),
  unset_s(150), unset_s(151), unset_s(152), unset_s(153), unset_s(154),
  unset_s(155), unset_s(156), unset_s(157), unset_s(158), unset_s(159),
  unset_s(160), unset_s(161), unset_s(162), unset_s(163), unset_s(164),
  unset_s(165), unset_s(166), unset_s(167), unset_s(168), unset_s(169),
  unset_s(170), unset_s(171), unset_s(172), unset_s(173), unset_s(174),
  unset_s(175), unset_s(176), unset_s(177), unset_s(178), unset_s(179),
  unset_s(180), unset_s(181), unset_s(182), unset_s(183), unset_s(184),
  unset_s(185), unset_s(186), unset_s(187), unset_s(188), unset_s(189),
  unset_s(190), unset_s(191), unset_s(192), unset_s(193), unset_s(194),
  unset_s(195), unset_s(196), unset_s(197), unset_s(198), unset_s(199),
  unset_s(200), unset_s(201), unset_s(202), unset_s(203), unset_s(204),
  unset_s(205), unset_s(206), unset_s(207), unset_s(208), unset_s(209),
  unset_s(210), unset_s(211), unset_s(212), unset_s(213), unset_s(214),
  unset_s(215), unset_s(216), unset_s(217), unset_s(218), unset_s(219),
  unset_s(220), unset_s(221), unset_s(222), unset_s(223), unset_s(224),
  unset_s(225), unset_s(226), unset_s(227), unset_s(228), unset_s(229),
  unset_s(230), unset_s(231), unset_s(232), unset_s(233), unset_s(234),
  unset_s(235), unset_s(236), unset_s(237), unset_s(238), unset_s(239)
};

/// Unset bits > stop
const Array<uint64_t, 240> Sieve::unset_larger =
{
  unset_l(0), unset_l(1), unset_l(2), unset_l(3), unset_l(4),
  unset_l(5), unset_l(6), unset_l(7), unset_l(8), unset_l(9),
  unset_l(10), unset_l(11), unset_l(12), unset_l(13), unset_l(14),
  unset_l(15), unset_l(16), unset_l(17), unset_l(18), unset_l(19),
  unset_l(20), unset_l(21), unset_l(22), unset_l(23), unset_l(24),
  unset_l(25), unset_l(26), unset_l(27), unset_l(28), unset_l(29),
  unset_l(30), unset_l(31), unset_l(32), unset_l(33), unset_l(34),
  unset_l(35), unset_l(36), unset_l(37), unset_l(38), unset_l(39),
  unset_l(40), unset_l(41), unset_l(42), unset_l(43), unset_l(44),
  unset_l(45), unset_l(46), unset_l(47), unset_l(48), unset_l(49),
  unset_l(50), unset_l(51), unset_l(52), unset_l(53), unset_l(54),
  unset_l(55), unset_l(56), unset_l(57), unset_l(58), unset_l(59),
  unset_l(60), unset_l(61), unset_l(62), unset_l(63), unset_l(64),
  unset_l(65), unset_l(66), unset_l(67), unset_l(68), unset_l(69),
  unset_l(70), unset_l(71), unset_l(72), unset_l(73), unset_l(74),
  unset_l(75), unset_l(76), unset_l(77), unset_l(78), unset_l(79),
  unset_l(80), unset_l(81), unset_l(82), unset_l(83), unset_l(84),
  unset_l(85), unset_l(86), unset_l(87), unset_l(88), unset_l(89),
  unset_l(90), unset_l(91), unset_l(92), unset_l(93), unset_l(94),
  unset_l(95), unset_l(96), unset_l(97), unset_l(98), unset_l(99),
  unset_l(100), unset_l(101), unset_l(102), unset_l(103), unset_l(104),
  unset_l(105), unset_l(106), unset_l(107), unset_l(108), unset_l(109),
  unset_l(110), unset_l(111), unset_l(112), unset_l(113), unset_l(114),
  unset_l(115), unset_l(116), unset_l(117), unset_l(118), unset_l(119),
  unset_l(120), unset_l(121), unset_l(122), unset_l(123), unset_l(124),
  unset_l(125), unset_l(126), unset_l(127), unset_l(128), unset_l(129),
  unset_l(130), unset_l(131), unset_l(132), unset_l(133), unset_l(134),
  unset_l(135), unset_l(136), unset_l(137), unset_l(138), unset_l(139),
  unset_l(140), unset_l(141), unset_l(142), unset_l(143), unset_l(144),
  unset_l(145), unset_l(146), unset_l(147), unset_l(148), unset_l(149),
  unset_l(150), unset_l(151), unset_l(152), unset_l(153), unset_l(154),
  unset_l(155), unset_l(156), unset_l(157), unset_l(158), unset_l(159),
  unset_l(160), unset_l(161), unset_l(162), unset_l(163), unset_l(164),
  unset_l(165), unset_l(166), unset_l(167), unset_l(168), unset_l(169),
  unset_l(170), unset_l(171), unset_l(172), unset_l(173), unset_l(174),
  unset_l(175), unset_l(176), unset_l(177), unset_l(178), unset_l(179),
  unset_l(180), unset_l(181), unset_l(182), unset_l(183), unset_l(184),
  unset_l(185), unset_l(186), unset_l(187), unset_l(188), unset_l(189),
  unset_l(190), unset_l(191), unset_l(192), unset_l(193), unset_l(194),
  unset_l(195), unset_l(196), unset_l(197), unset_l(198), unset_l(199),
  unset_l(200), unset_l(201), unset_l(202), unset_l(203), unset_l(204),
  unset_l(205), unset_l(206), unset_l(207), unset_l(208), unset_l(209),
  unset_l(210), unset_l(211), unset_l(212), unset_l(213), unset_l(214),
  unset_l(215), unset_l(216), unset_l(217), unset_l(218), unset_l(219),
  unset_l(220), unset_l(221), unset_l(222), unset_l(223), unset_l(224),
  unset_l(225), unset_l(226), unset_l(227), unset_l(228), unset_l(229),
  unset_l(230), unset_l(231), unset_l(232), unset_l(233), unset_l(234),
  unset_l(235), unset_l(236), unset_l(237), unset_l(238), unset_l(239)
};

} // namespace

namespace {

/// Static array from which the primes and multiples
/// of primes ≤ 13 have been removed.
///
const primecount::Array<uint8_t, 4004> pre_sieved_13 =
{
  0xf1, 0xdf, 0xef, 0x7e, 0xb6, 0xdb, 0x3d, 0xf9, 0xd5, 0x6f, 0x5e, 0xf3, 0xeb, 0xa7, 0xfd, 0x9e, 0xee, 0x3c, 0xd7, 0xf3,
  0xbb, 0xdd, 0x5b, 0xef, 0x6e, 0xe7, 0xb2, 0xbf, 0x7d, 0xde, 0xa7, 0x5e, 0xd7, 0xfb, 0xbd, 0x6d, 0xdf, 0xae, 0x6e, 0x75,
  0xfb, 0xb7, 0xfc, 0x9f, 0xcb, 0x7e, 0xe3, 0xf9, 0x3e, 0xfd, 0x5d, 0xef, 0x76, 0xf7, 0xdb, 0xba, 0xbd, 0xcf, 0xef, 0x3a,
  0xf5, 0x79, 0xbf, 0xf5, 0xcf, 0xc7, 0x7a, 0xf7, 0xea, 0xbf, 0x9c, 0xdf, 0x69, 0x7c, 0x77, 0xfb, 0x9f, 0xe9, 0xd7, 0xef,
  0x7e, 0xb6, 0xbb, 0x1d, 0xfd, 0xd3, 0xed, 0x5e, 0xf3, 0xfb, 0xaf, 0xf5, 0x9e, 0xef, 0x7c, 0xb7, 0xd3, 0xbf, 0xd9, 0xd9,
  0x6f, 0x6e, 0xf7, 0xaa, 0xb7, 0x7d, 0xdf, 0xe6, 0x3e, 0xd7, 0xfb, 0xbb, 0xed, 0x5f, 0xae, 0x7e, 0x65, 0xf3, 0xb7, 0xfd,
  0xde, 0xab, 0x5e, 0xe7, 0xfb, 0xbc, 0x7d, 0x5d, 0xef, 0x66, 0xf7, 0xdb, 0xbb, 0xfc, 0x8f, 0xcf, 0x3e, 0xf3, 0x79, 0x3f,
  0xf5, 0xdf, 0xcf, 0x72, 0xf7, 0xeb, 0xbe, 0xbc, 0xdf, 0x6d, 0x7a, 0xf5, 0x7b, 0x9f, 0xf9, 0xcf, 0xe7, 0x7e, 0xb6, 0xfa,
  0x3d, 0xdd, 0xd7, 0xeb, 0x5c, 0x73, 0xfb, 0xaf, 0xed, 0x96, 0xef, 0x7c, 0xf6, 0xb3, 0x9f, 0xdd, 0xdb, 0xed, 0x6e, 0xf7,
  0xba, 0xaf, 0x75, 0xdf, 0xe7, 0x7e, 0x97, 0xdb, 0xbf, 0xe9, 0xdd, 0x2e, 0x7e, 0x75, 0xeb, 0xb7, 0xfd, 0xdf, 0xea, 0x3e,
  0xc7, 0xfb, 0xba, 0xfd, 0x5d, 0xef, 0x76, 0xe7, 0xd3, 0xbb, 0xfd, 0xce, 0xaf, 0x1e, 0xf7, 0x79, 0xbd, 0x75, 0xdf, 0xcf,
  0x6a, 0xf7, 0xeb, 0xbf, 0xbc, 0x9f, 0x4d, 0x7e, 0xf3, 0xf9, 0x1f, 0xf9, 0xdf, 0xef, 0x76, 0xb6, 0xfb, 0x3c, 0xbd, 0xd7,
  0xef, 0x5a, 0xf1, 0x7b, 0xaf, 0xfd, 0x8e, 0xe7, 0x7c, 0xf7, 0xf2, 0xbf, 0xdd, 0xdb, 0xeb, 0x6c, 0x77, 0xba, 0xbf, 0x6d,
  0xd7, 0xe7, 0x7e, 0xd6, 0xbb, 0x9f, 0xed, 0xdb, 0xac, 0x7e, 0x75, 0xfb, 0xa7, 0xf5, 0xdf, 0xeb, 0x7e, 0xa7, 0xdb, 0xbe,
  0xf9, 0x5d, 0x6f, 0x76, 0xf7, 0xcb, 0xb3, 0xfd, 0xcf, 0xee, 0x3e, 0xd7, 0x79, 0xbb, 0xf5, 0x5f, 0xcf, 0x7a, 0xe7, 0xe3,
  0xbf, 0xbc, 0xde, 0x2d, 0x5e, 0xf7, 0xfb, 0x9d, 0x79, 0xdf, 0xef, 0x6e, 0xb6, 0xfb, 0x3d, 0xfc, 0x97, 0xcf, 0x5e, 0xf3,
  0xf9, 0x2f, 0xfd, 0x9e, 0xef, 0x74, 0xf7, 0xf3, 0xbe, 0x9d, 0xdb, 0xef, 0x6a, 0xf5, 0x3a, 0xbf, 0x7d, 0xcf, 0xe7, 0x7e,
  0xd7, 0xfa, 0xbf, 0xcd, 0xdf, 0xaa, 0x7c, 0x75, 0xfb, 0xb7, 0xed, 0xd7, 0xeb, 0x7e, 0xe6, 0xbb, 0x9e, 0xfd, 0x59, 0xed,
  0x76, 0xf7, 0xdb, 0xab, 0xf5, 0xcf, 0xef, 0x3e, 0xb7, 0x59, 0xbf, 0xf1, 0xdd, 0x4f, 0x7a, 0xf7, 0xeb, 0xb7, 0xbc, 0xdf,
  0x6c, 0x3e, 0xd7, 0xfb, 0x9b, 0xf9, 0x5f, 0xef, 0x7e, 0xa6, 0xf3, 0x3d, 0xfd, 0xd6, 0xaf, 0x5e, 0xf3, 0xfb, 0xad, 0x7d,
  0x9e, 0xef, 0x6c, 0xf7, 0xf3, 0xbf, 0xdc, 0x9b, 0xcf, 0x6e, 0xf3, 0xb8, 0x3f, 0x7d, 0xdf, 0xe7, 0x76, 0xd7, 0xfb, 0xbe,
  0xad, 0xdf, 0xae, 0x7a, 0x75, 0x7b, 0xb7, 0xfd, 0xcf, 0xe3, 0x7e, 0xe7, 0xfa, 0xbe, 0xdd, 0x5d, 0xeb, 0x74, 0x77, 0xdb,
  0xbb, 0xed, 0xc7, 0xef, 0x3e, 0xf6, 0x39, 0x9f, 0xf5, 0xdb, 0xcd, 0x7a, 0xf7, 0xeb, 0xaf, 0xb4, 0xdf, 0x6d, 0x7e, 0xb7,
  0xdb, 0x9f, 0xf9, 0xdd, 0x6f, 0x7e, 0xb6, 0xeb, 0x35, 0xfd, 0xd7, 0xee, 0x1e, 0xd3, 0xfb, 0xab, 0xfd, 0x1e, 0xef, 0x7c,
  0xe7, 0xf3, 0xbf, 0xdd, 0xda, 0xaf, 0x4e, 0xf7, 0xba, 0xbd, 0x7d, 0xdf, 0xe7, 0x6e, 0xd7, 0xfb, 0xbf, 0xec, 0x9f, 0x8e,
  0x7e, 0x71, 0xf9, 0x37, 0xfd, 0xdf, 0xeb, 0x76, 0xe7, 0xfb, 0xbe, 0xbd, 0x5d, 0xef, 0x72, 0xf5, 0x5b, 0xbb, 0xfd, 0xcf,
  0xe7, 0x3e, 0xf7, 0x78, 0xbf, 0xd5, 0xdf, 0xcb, 0x78, 0x77, 0xeb, 0xbf, 0xac, 0xd7, 0x6d, 0x7e, 0xf6, 0xbb, 0x9f, 0xf9,
  0xdb, 0xed, 0x7e, 0xb6, 0xfb, 0x2d, 0xf5, 0xd7, 0xef, 0x5e, 0xb3, 0xdb, 0xaf, 0xf9, 0x9c, 0x6f, 0x7c, 0xf7, 0xe3, 0xb7,
  0xdd, 0xdb, 0xee, 0x2e, 0xd7, 0xba, 0xbb, 0x7d, 0x5f, 0xe7, 0x7e, 0xc7, 0xf3, 0xbf, 0xed, 0xde, 0xae, 0x5e, 0x75, 0xfb,
  0xb5, 0x7d, 0xdf, 0xeb, 0x6e, 0xe7, 0xfb, 0xbe, 0xfc, 0x1d, 0xcf, 0x76, 0xf3, 0xd9, 0x3b, 0xfd, 0xcf, 0xef, 0x36, 0xf7,
  0x79, 0xbe, 0xb5, 0xdf, 0xcf, 0x7a, 0xf5, 0x6b, 0xbf, 0xbc, 0xcf, 0x65, 0x7e, 0xf7, 0xfa, 0x9f, 0xd9, 0xdf, 0xeb, 0x7c,
  0x36, 0xfb, 0x3d, 0xed, 0xd7, 0xef, 0x5e, 0xf2, 0xbb, 0x8f, 0xfd, 0x9a, 0xed, 0x7c, 0xf7, 0xf3, 0xaf, 0xd5, 0xdb, 0xef,
  0x6e, 0xb7, 0x9a, 0xbf, 0x79, 0xdd, 0x67, 0x7e, 0xd7, 0xeb, 0xb7, 0xed, 0xdf, 0xae, 0x3e, 0x55, 0xfb, 0xb3, 0xfd, 0x5f,
  0xeb, 0x7e, 0xe7, 0xf3, 0xbe, 0xfd, 0x5c, 0xaf, 0x56, 0xf7, 0xdb, 0xb9, 0x7d, 0xcf, 0xef, 0x2e, 0xf7, 0x79, 0xbf, 0xf4,
  0x9f, 0xcf, 0x7a, 0xf3, 0xe9, 0x3f, 0xbc, 0xdf, 0x6d, 0x76, 0xf7, 0xfb, 0x9e, 0xb9, 0xdf, 0xef, 0x7a, 0xb4, 0x7b, 0x3d,
  0xfd, 0xc7, 0xe7, 0x5e, 0xf3, 0xfa, 0xaf, 0xdd, 0x9e, 0xeb, 0x7c, 0x77, 0xf3, 0xbf, 0xcd, 0xd3, 0xef, 0x6e, 0xf6, 0xba,
  0x9f, 0x7d, 0xdb, 0xe5, 0x7e, 0xd7, 0xfb, 0xaf, 0xe5, 0xdf, 0xae, 0x7e, 0x35, 0xdb, 0xb7, 0xf9, 0xdd, 0x6b, 0x7e, 0xe7,
  0xeb, 0xb6, 0xfd, 0x5d, 0xee, 0x36, 0xd7, 0xdb, 0xbb, 0xfd, 0x4f, 0xef, 0x3e, 0xe7, 0x71, 0xbf, 0xf5, 0xde, 0x8f, 0x5a,
  0xf7, 0xeb, 0xbd, 0x3c, 0xdf, 0x6d, 0x6e, 0xf7, 0xfb, 0x9f, 0xf8, 0x9f, 0xcf, 0x7e, 0xb2, 0xf9, 0x3d, 0xfd, 0xd7, 0xef,
  0x56, 0xf3, 0xfb, 0xae, 0xbd, 0x9e, 0xef, 0x78, 0xf5, 0x73, 0xbf, 0xdd, 0xcb, 0xe7, 0x6e, 0xf7, 0xba, 0xbf, 0x5d, 0xdf,
  0xe3, 0x7c, 0x57, 0xfb, 0xbf, 0xed, 0xd7, 0xae, 0x7e, 0x74, 0xbb, 0x97, 0xfd, 0xdb, 0xe9, 0x7e, 0xe7, 0xfb, 0xae, 0xf5,
  0x5d, 0xef, 0x76, 0xb7, 0xdb, 0xbb, 0xf9, 0xcd, 0x6f, 0x3e, 0xf7, 0x69, 0xb7, 0xf5, 0xdf, 0xce, 0x3a, 0xd7, 0xeb, 0xbb,
  0xbc, 0x5f, 0x6d, 0x7e, 0xe7, 0xf3, 0x9f, 0xf9, 0xde, 0xaf, 0x5e, 0xb6, 0xfb, 0x3d, 0x7d, 0xd7, 0xef, 0x4e, 0xf3, 0xfb,
  0xaf, 0xfc, 0x9e, 0xcf, 0x7c, 0xf3, 0xf1, 0x3f, 0xdd, 0xdb, 0xef, 0x66, 0xf7, 0xba, 0xbe, 0x3d, 0xdf, 0xe7, 0x7a, 0xd5,
  0x7b, 0xbf, 0xed, 0xcf, 0xa6, 0x7e, 0x75, 0xfa, 0xb7, 0xdd, 0xdf, 0xeb, 0x7c, 0x67, 0xfb, 0xbe, 0xed, 0x55, 0xef, 0x76,
  0xf6, 0x9b, 0x9b, 0xfd, 0xcb, 0xed, 0x3e, 0xf7, 0x79, 0xaf, 0xf5, 0xdf, 0xcf, 0x7a, 0xb7, 0xcb, 0xbf, 0xb8, 0xdd, 0x6d,
  0x7e, 0xf7, 0xeb, 0x97, 0xf9, 0xdf, 0xee, 0x3e, 0x96, 0xfb, 0x39, 0xfd, 0x57, 0xef, 0x5e, 0xe3, 0xf3, 0xaf, 0xfd, 0x9e,
  0xaf, 0x5c, 0xf7, 0xf3, 0xbd, 0x5d, 0xdb, 0xef, 0x6e, 0xf7, 0xba, 0xbf, 0x7c, 0x9f, 0xc7, 0x7e, 0xd3, 0xf9, 0x3f, 0xed,
  0xdf, 0xae, 0x76, 0x75, 0xfb, 0xb6, 0xbd, 0xdf, 0xeb, 0x7a, 0xe5, 0x7b, 0xbe, 0xfd, 0x4d, 0xe7, 0x76, 0xf7, 0xda, 0xbb,
  0xdd, 0xcf, 0xeb, 0x3c, 0x77, 0x79, 0xbf, 0xe5, 0xd7, 0xcf, 0x7a, 0xf6, 0xab, 0x9f, 0xbc, 0xdb, 0x6d, 0x7e, 0xf7, 0xfb,
  0x8f, 0xf1, 0xdf, 0xef, 0x7e, 0xb6, 0xdb, 0x3d, 0xf9, 0xd5, 0x6f, 0x5e, 0xf3, 0xeb, 0xa7, 0xfd, 0x9e, 0xee, 0x3c, 0xd7,
  0xf3, 0xbb, 0xdd, 0x5b, 0xef, 0x6e, 0xe7, 0xb2, 0xbf, 0x7d, 0xde, 0xa7, 0x5e, 0xd7, 0xfb, 0xbd, 0x6d, 0xdf, 0xae, 0x6e,
  0x75, 0xfb, 0xb7, 0xfc, 0x9f, 0xcb, 0x7e, 0xe3, 0xf9, 0x3e, 0xfd, 0x5d, 0xef, 0x76, 0xf7, 0xdb, 0xba, 0xbd, 0xcf, 0xef,
  0x3a, 0xf5, 0x79, 0xbf, 0xf5, 0xcf, 0xc7, 0x7a, 0xf7, 0xea, 0xbf, 0x9c, 0xdf, 0x69, 0x7c, 0x77, 0xfb, 0x9f, 0xe9, 0xd7,
  0xef, 0x7e, 0xb6, 0xbb, 0x1d, 0xfd, 0xd3, 0xed, 0x5e, 0xf3, 0xfb, 0xaf, 0xf5, 0x9e, 0xef, 0x7c, 0xb7, 0xd3, 0xbf, 0xd9,
  0xd9, 0x6f, 0x6e, 0xf7, 0xaa, 0xb7, 0x7d, 0xdf, 0xe6, 0x3e, 0xd7, 0xfb, 0xbb, 0xed, 0x5f, 0xae, 0x7e, 0x65, 0xf3, 0xb7,
  0xfd, 0xde, 0xab, 0x5e, 0xe7, 0xfb, 0xbc, 0x7d, 0x5d, 0xef, 0x66, 0xf7, 0xdb, 0xbb, 0xfc, 0x8f, 0xcf, 0x3e, 0xf3, 0x79,
  0x3f, 0xf5, 0xdf, 0xcf, 0x72, 0xf7, 0xeb, 0xbe, 0xbc, 0xdf, 0x6d, 0x7a, 0xf5, 0x7b, 0x9f, 0xf9, 0xcf, 0xe7, 0x7e, 0xb6,
  0xfa, 0x3d, 0xdd, 0xd7, 0xeb, 0x5c, 0x73, 0xfb, 0xaf, 0xed, 0x96, 0xef, 0x7c, 0xf6, 0xb3, 0x9f, 0xdd, 0xdb, 0xed, 0x6e,
  0xf7, 0xba, 0xaf, 0x75, 0xdf, 0xe7, 0x7e, 0x97, 0xdb, 0xbf, 0xe9, 0xdd, 0x2e, 0x7e, 0x75, 0xeb, 0xb7, 0xfd, 0xdf, 0xea,
  0x3e, 0xc7, 0xfb, 0xba, 0xfd, 0x5d, 0xef, 0x76, 0xe7, 0xd3, 0xbb, 0xfd, 0xce, 0xaf, 0x1e, 0xf7, 0x79, 0xbd, 0x75, 0xdf,
  0xcf, 0x6a, 0xf7, 0xeb, 0xbf, 0xbc, 0x9f, 0x4d, 0x7e, 0xf3, 0xf9, 0x1f, 0xf9, 0xdf, 0xef, 0x76, 0xb6, 0xfb, 0x3c, 0xbd,
  0xd7, 0xef, 0x5a, 0xf1, 0x7b, 0xaf, 0xfd, 0x8e, 0xe7, 0x7c, 0xf7, 0xf2, 0xbf, 0xdd, 0xdb, 0xeb, 0x6c, 0x77, 0xba, 0xbf,
  0x6d, 0xd7, 0xe7, 0x7e, 0xd6, 0xbb, 0x9f, 0xed, 0xdb, 0xac, 0x7e, 0x75, 0xfb, 0xa7, 0xf5, 0xdf, 0xeb, 0x7e, 0xa7, 0xdb,
  0xbe, 0xf9, 0x5d, 0x6f, 0x76, 0xf7, 0xcb, 0xb3, 0xfd, 0xcf, 0xee, 0x3e, 0xd7, 0x79, 0xbb, 0xf5, 0x5f, 0xcf, 0x7a, 0xe7,
  0xe3, 0xbf, 0xbc, 0xde, 0x2d, 0x5e, 0xf7, 0xfb, 0x9d, 0x79, 0xdf, 0xef, 0x6e, 0xb6, 0xfb, 0x3d, 0xfc, 0x97, 0xcf, 0x5e,
  0xf3, 0xf9, 0x2f, 0xfd, 0x9e, 0xef, 0x74, 0xf7, 0xf3, 0xbe, 0x9d, 0xdb, 0xef, 0x6a, 0xf5, 0x3a, 0xbf, 0x7d, 0xcf, 0xe7,
  0x7e, 0xd7, 0xfa, 0xbf, 0xcd, 0xdf, 0xaa, 0x7c, 0x75, 0xfb, 0xb7, 0xed, 0xd7, 0xeb, 0x7e, 0xe6, 0xbb, 0x9e, 0xfd, 0x59,
  0xed, 0x76, 0xf7, 0xdb, 0xab, 0xf5, 0xcf, 0xef, 0x3e, 0xb7, 0x59, 0xbf, 0xf1, 0xdd, 0x4f, 0x7a, 0xf7, 0xeb, 0xb7, 0xbc,
  0xdf, 0x6c, 0x3e, 0xd7, 0xfb, 0x9b, 0xf9, 0x5f, 0xef, 0x7e, 0xa6, 0xf3, 0x3d, 0xfd, 0xd6, 0xaf, 0x5e, 0xf3, 0xfb, 0xad,
  0x7d, 0x9e, 0xef, 0x6c, 0xf7, 0xf3, 0xbf, 0xdc, 0x9b, 0xcf, 0x6e, 0xf3, 0xb8, 0x3f, 0x7d, 0xdf, 0xe7, 0x76, 0xd7, 0xfb,
  0xbe, 0xad, 0xdf, 0xae, 0x7a, 0x75, 0x7b, 0xb7, 0xfd, 0xcf, 0xe3, 0x7e, 0xe7, 0xfa, 0xbe, 0xdd, 0x5d, 0xeb, 0x74, 0x77,
  0xdb, 0xbb, 0xed, 0xc7, 0xef, 0x3e, 0xf6, 0x39, 0x9f, 0xf5, 0xdb, 0xcd, 0x7a, 0xf7, 0xeb, 0xaf, 0xb4, 0xdf, 0x6d, 0x7e,
  0xb7, 0xdb, 0x9f, 0xf9, 0xdd, 0x6f, 0x7e, 0xb6, 0xeb, 0x35, 0xfd, 0xd7, 0xee, 0x1e, 0xd3, 0xfb, 0xab, 0xfd, 0x1e, 0xef,
  0x7c, 0xe7, 0xf3, 0xbf, 0xdd, 0xda, 0xaf, 0x4e, 0xf7, 0xba, 0xbd, 0x7d, 0xdf, 0xe7, 0x6e, 0xd7, 0xfb, 0xbf, 0xec, 0x9f,
  0x8e, 0x7e, 0x71, 0xf9, 0x37, 0xfd, 0xdf, 0xeb, 0x76, 0xe7, 0xfb, 0xbe, 0xbd, 0x5d, 0xef, 0x72, 0xf5, 0x5b, 0xbb, 0xfd,
  0xcf, 0xe7, 0x3e, 0xf7, 0x78, 0xbf, 0xd5, 0xdf, 0xcb, 0x78, 0x77, 0xeb, 0xbf, 0xac, 0xd7, 0x6d, 0x7e, 0xf6, 0xbb, 0x9f,
  0xf9, 0xdb, 0xed, 0x7e, 0xb6, 0xfb, 0x2d, 0xf5, 0xd7, 0xef, 0x5e, 0xb3, 0xdb, 0xaf, 0xf9, 0x9c, 0x6f, 0x7c, 0xf7, 0xe3,
  0xb7, 0xdd, 0xdb, 0xee, 0x2e, 0xd7, 0xba, 0xbb, 0x7d, 0x5f, 0xe7, 0x7e, 0xc7, 0xf3, 0xbf, 0xed, 0xde, 0xae, 0x5e, 0x75,
  0xfb, 0xb5, 0x7d, 0xdf, 0xeb, 0x6e, 0xe7, 0xfb, 0xbe, 0xfc, 0x1d, 0xcf, 0x76, 0xf3, 0xd9, 0x3b, 0xfd, 0xcf, 0xef, 0x36,
  0xf7, 0x79, 0xbe, 0xb5, 0xdf, 0xcf, 0x7a, 0xf5, 0x6b, 0xbf, 0xbc, 0xcf, 0x65, 0x7e, 0xf7, 0xfa, 0x9f, 0xd9, 0xdf, 0xeb,
  0x7c, 0x36, 0xfb, 0x3d, 0xed, 0xd7, 0xef, 0x5e, 0xf2, 0xbb, 0x8f, 0xfd, 0x9a, 0xed, 0x7c, 0xf7, 0xf3, 0xaf, 0xd5, 0xdb,
  0xef, 0x6e, 0xb7, 0x9a, 0xbf, 0x79, 0xdd, 0x67, 0x7e, 0xd7, 0xeb, 0xb7, 0xed, 0xdf, 0xae, 0x3e, 0x55, 0xfb, 0xb3, 0xfd,
  0x5f, 0xeb, 0x7e, 0xe7, 0xf3, 0xbe, 0xfd, 0x5c, 0xaf, 0x56, 0xf7, 0xdb, 0xb9, 0x7d, 0xcf, 0xef, 0x2e, 0xf7, 0x79, 0xbf,
  0xf4, 0x9f, 0xcf, 0x7a, 0xf3, 0xe9, 0x3f, 0xbc, 0xdf, 0x6d, 0x76, 0xf7, 0xfb, 0x9e, 0xb9, 0xdf, 0xef, 0x7a, 0xb4, 0x7b,
  0x3d, 0xfd, 0xc7, 0xe7, 0x5e, 0xf3, 0xfa, 0xaf, 0xdd, 0x9e, 0xeb, 0x7c, 0x77, 0xf3, 0xbf, 0xcd, 0xd3, 0xef, 0x6e, 0xf6,
  0xba, 0x9f, 0x7d, 0xdb, 0xe5, 0x7e, 0xd7, 0xfb, 0xaf, 0xe5, 0xdf, 0xae, 0x7e, 0x35, 0xdb, 0xb7, 0xf9, 0xdd, 0x6b, 0x7e,
  0xe7, 0xeb, 0xb6, 0xfd, 0x5d, 0xee, 0x36, 0xd7, 0xdb, 0xbb, 0xfd, 0x4f, 0xef, 0x3e, 0xe7, 0x71, 0xbf, 0xf5, 0xde, 0x8f,
  0x5a, 0xf7, 0xeb, 0xbd, 0x3c, 0xdf, 0x6d, 0x6e, 0xf7, 0xfb, 0x9f, 0xf8, 0x9f, 0xcf, 0x7e, 0xb2, 0xf9, 0x3d, 0xfd, 0xd7,
  0xef, 0x56, 0xf3, 0xfb, 0xae, 0xbd, 0x9e, 0xef, 0x78, 0xf5, 0x73, 0xbf, 0xdd, 0xcb, 0xe7, 0x6e, 0xf7, 0xba, 0xbf, 0x5d,
  0xdf, 0xe3, 0x7c, 0x57, 0xfb, 0xbf, 0xed, 0xd7, 0xae, 0x7e, 0x74, 0xbb, 0x97, 0xfd, 0xdb, 0xe9, 0x7e, 0xe7, 0xfb, 0xae,
  0xf5, 0x5d, 0xef, 0x76, 0xb7, 0xdb, 0xbb, 0xf9, 0xcd, 0x6f, 0x3e, 0xf7, 0x69, 0xb7, 0xf5, 0xdf, 0xce, 0x3a, 0xd7, 0xeb,
  0xbb, 0xbc, 0x5f, 0x6d, 0x7e, 0xe7, 0xf3, 0x9f, 0xf9, 0xde, 0xaf, 0x5e, 0xb6, 0xfb, 0x3d, 0x7d, 0xd7, 0xef, 0x4e, 0xf3,
  0xfb, 0xaf, 0xfc, 0x9e, 0xcf, 0x7c, 0xf3, 0xf1, 0x3f, 0xdd, 0xdb, 0xef, 0x66, 0xf7, 0xba, 0xbe, 0x3d, 0xdf, 0xe7, 0x7a,
  0xd5, 0x7b, 0xbf, 0xed, 0xcf, 0xa6, 0x7e, 0x75, 0xfa, 0xb7, 0xdd, 0xdf, 0xeb, 0x7c, 0x67, 0xfb, 0xbe, 0xed, 0x55, 0xef,
  0x76, 0xf6, 0x9b, 0x9b, 0xfd, 0xcb, 0xed, 0x3e, 0xf7, 0x79, 0xaf, 0xf5, 0xdf, 0xcf, 0x7a, 0xb7, 0xcb, 0xbf, 0xb8, 0xdd,
  0x6d, 0x7e, 0xf7, 0xeb, 0x97, 0xf9, 0xdf, 0xee, 0x3e, 0x96, 0xfb, 0x39, 0xfd, 0x57, 0xef, 0x5e, 0xe3, 0xf3, 0xaf, 0xfd,
  0x9e, 0xaf, 0x5c, 0xf7, 0xf3, 0xbd, 0x5d, 0xdb, 0xef, 0x6e, 0xf7, 0xba, 0xbf, 0x7c, 0x9f, 0xc7, 0x7e, 0xd3, 0xf9, 0x3f,
  0xed, 0xdf, 0xae, 0x76, 0x75, 0xfb, 0xb6, 0xbd, 0xdf, 0xeb, 0x7a, 0xe5, 0x7b, 0xbe, 0xfd, 0x4d, 0xe7, 0x76, 0xf7, 0xda,
  0xbb, 0xdd, 0xcf, 0xeb, 0x3c, 0x77, 0x79, 0xbf, 0xe5, 0xd7, 0xcf, 0x7a, 0xf6, 0xab, 0x9f, 0xbc, 0xdb, 0x6d, 0x7e, 0xf7,
  0xfb, 0x8f, 0xf1, 0xdf, 0xef, 0x7e, 0xb6, 0xdb, 0x3d, 0xf9, 0xd5, 0x6f, 0x5e, 0xf3, 0xeb, 0xa7, 0xfd, 0x9e, 0xee, 0x3c,
  0xd7, 0xf3, 0xbb, 0xdd, 0x5b, 0xef, 0x6e, 0xe7, 0xb2, 0xbf, 0x7d, 0xde, 0xa7, 0x5e, 0xd7, 0xfb, 0xbd, 0x6d, 0xdf, 0xae,
  0x6e, 0x75, 0xfb, 0xb7, 0xfc, 0x9f, 0xcb, 0x7e, 0xe3, 0xf9, 0x3e, 0xfd, 0x5d, 0xef, 0x76, 0xf7, 0xdb, 0xba, 0xbd, 0xcf,
  0xef, 0x3a, 0xf5, 0x79, 0xbf, 0xf5, 0xcf, 0xc7, 0x7a, 0xf7, 0xea, 0xbf, 0x9c, 0xdf, 0x69, 0x7c, 0x77, 0xfb, 0x9f, 0xe9,
  0xd7, 0xef, 0x7e, 0xb6, 0xbb, 0x1d, 0xfd, 0xd3, 0xed, 0x5e, 0xf3, 0xfb, 0xaf, 0xf5, 0x9e, 0xef, 0x7c, 0xb7, 0xd3, 0xbf,
  0xd9, 0xd9, 0x6f, 0x6e, 0xf7, 0xaa, 0xb7, 0x7d, 0xdf, 0xe6, 0x3e, 0xd7, 0xfb, 0xbb, 0xed, 0x5f, 0xae, 0x7e, 0x65, 0xf3,
  0xb7, 0xfd, 0xde, 0xab, 0x5e, 0xe7, 0xfb, 0xbc, 0x7d, 0x5d, 0xef, 0x66, 0xf7, 0xdb, 0xbb, 0xfc, 0x8f, 0xcf, 0x3e, 0xf3,
  0x79, 0x3f, 0xf5, 0xdf, 0xcf, 0x72, 0xf7, 0xeb, 0xbe, 0xbc, 0xdf, 0x6d, 0x7a, 0xf5, 0x7b, 0x9f, 0xf9, 0xcf, 0xe7, 0x7e,
  0xb6, 0xfa, 0x3d, 0xdd, 0xd7, 0xeb, 0x5c, 0x73, 0xfb, 0xaf, 0xed, 0x96, 0xef, 0x7c, 0xf6, 0xb3, 0x9f, 0xdd, 0xdb, 0xed,
  0x6e, 0xf7, 0xba, 0xaf, 0x75, 0xdf, 0xe7, 0x7e, 0x97, 0xdb, 0xbf, 0xe9, 0xdd, 0x2e, 0x7e, 0x75, 0xeb, 0xb7, 0xfd, 0xdf,
  0xea, 0x3e, 0xc7, 0xfb, 0xba, 0xfd, 0x5d, 0xef, 0x76, 0xe7, 0xd3, 0xbb, 0xfd, 0xce, 0xaf, 0x1e, 0xf7, 0x79, 0xbd, 0x75,
  0xdf, 0xcf, 0x6a, 0xf7, 0xeb, 0xbf, 0xbc, 0x9f, 0x4d, 0x7e, 0xf3, 0xf9, 0x1f, 0xf9, 0xdf, 0xef, 0x76, 0xb6, 0xfb, 0x3c,
  0xbd, 0xd7, 0xef, 0x5a, 0xf1, 0x7b, 0xaf, 0xfd, 0x8e, 0xe7, 0x7c, 0xf7, 0xf2, 0xbf, 0xdd, 0xdb, 0xeb, 0x6c, 0x77, 0xba,
  0xbf, 0x6d, 0xd7, 0xe7, 0x7e, 0xd6, 0xbb, 0x9f, 0xed, 0xdb, 0xac, 0x7e, 0x75, 0xfb, 0xa7, 0xf5, 0xdf, 0xeb, 0x7e, 0xa7,
  0xdb, 0xbe, 0xf9, 0x5d, 0x6f, 0x76, 0xf7, 0xcb, 0xb3, 0xfd, 0xcf, 0xee, 0x3e, 0xd7, 0x79, 0xbb, 0xf5, 0x5f, 0xcf, 0x7a,
  0xe7, 0xe3, 0xbf, 0xbc, 0xde, 0x2d, 0x5e, 0xf7, 0xfb, 0x9d, 0x79, 0xdf, 0xef, 0x6e, 0xb6, 0xfb, 0x3d, 0xfc, 0x97, 0xcf,
  0x5e, 0xf3, 0xf9, 0x2f, 0xfd, 0x9e, 0xef, 0x74, 0xf7, 0xf3, 0xbe, 0x9d, 0xdb, 0xef, 0x6a, 0xf5, 0x3a, 0xbf, 0x7d, 0xcf,
  0xe7, 0x7e, 0xd7, 0xfa, 0xbf, 0xcd, 0xdf, 0xaa, 0x7c, 0x75, 0xfb, 0xb7, 0xed, 0xd7, 0xeb, 0x7e, 0xe6, 0xbb, 0x9e, 0xfd,
  0x59, 0xed, 0x76, 0xf7, 0xdb, 0xab, 0xf5, 0xcf, 0xef, 0x3e, 0xb7, 0x59, 0xbf, 0xf1, 0xdd, 0x4f, 0x7a, 0xf7, 0xeb, 0xb7,
  0xbc, 0xdf, 0x6c, 0x3e, 0xd7, 0xfb, 0x9b, 0xf9, 0x5f, 0xef, 0x7e, 0xa6, 0xf3, 0x3d, 0xfd, 0xd6, 0xaf, 0x5e, 0xf3, 0xfb,
  0xad, 0x7d, 0x9e, 0xef, 0x6c, 0xf7, 0xf3, 0xbf, 0xdc, 0x9b, 0xcf, 0x6e, 0xf3, 0xb8, 0x3f, 0x7d, 0xdf, 0xe7, 0x76, 0xd7,
  0xfb, 0xbe, 0xad, 0xdf, 0xae, 0x7a, 0x75, 0x7b, 0xb7, 0xfd, 0xcf, 0xe3, 0x7e, 0xe7, 0xfa, 0xbe, 0xdd, 0x5d, 0xeb, 0x74,
  0x77, 0xdb, 0xbb, 0xed, 0xc7, 0xef, 0x3e, 0xf6, 0x39, 0x9f, 0xf5, 0xdb, 0xcd, 0x7a, 0xf7, 0xeb, 0xaf, 0xb4, 0xdf, 0x6d,
  0x7e, 0xb7, 0xdb, 0x9f, 0xf9, 0xdd, 0x6f, 0x7e, 0xb6, 0xeb, 0x35, 0xfd, 0xd7, 0xee, 0x1e, 0xd3, 0xfb, 0xab, 0xfd, 0x1e,
  0xef, 0x7c, 0xe7, 0xf3, 0xbf, 0xdd, 0xda, 0xaf, 0x4e, 0xf7, 0xba, 0xbd, 0x7d, 0xdf, 0xe7, 0x6e, 0xd7, 0xfb, 0xbf, 0xec,
  0x9f, 0x8e, 0x7e, 0x71, 0xf9, 0x37, 0xfd, 0xdf, 0xeb, 0x76, 0xe7, 0xfb, 0xbe, 0xbd, 0x5d, 0xef, 0x72, 0xf5, 0x5b, 0xbb,
  0xfd, 0xcf, 0xe7, 0x3e, 0xf7, 0x78, 0xbf, 0xd5, 0xdf, 0xcb, 0x78, 0x77, 0xeb, 0xbf, 0xac, 0xd7, 0x6d, 0x7e, 0xf6, 0xbb,
  0x9f, 0xf9, 0xdb, 0xed, 0x7e, 0xb6, 0xfb, 0x2d, 0xf5, 0xd7, 0xef, 0x5e, 0xb3, 0xdb, 0xaf, 0xf9, 0x9c, 0x6f, 0x7c, 0xf7,
  0xe3, 0xb7, 0xdd, 0xdb, 0xee, 0x2e, 0xd7, 0xba, 0xbb, 0x7d, 0x5f, 0xe7, 0x7e, 0xc7, 0xf3, 0xbf, 0xed, 0xde, 0xae, 0x5e,
  0x75, 0xfb, 0xb5, 0x7d, 0xdf, 0xeb, 0x6e, 0xe7, 0xfb, 0xbe, 0xfc, 0x1d, 0xcf, 0x76, 0xf3, 0xd9, 0x3b, 0xfd, 0xcf, 0xef,
  0x36, 0xf7, 0x79, 0xbe, 0xb5, 0xdf, 0xcf, 0x7a, 0xf5, 0x6b, 0xbf, 0xbc, 0xcf, 0x65, 0x7e, 0xf7, 0xfa, 0x9f, 0xd9, 0xdf,
  0xeb, 0x7c, 0x36, 0xfb, 0x3d, 0xed, 0xd7, 0xef, 0x5e, 0xf2, 0xbb, 0x8f, 0xfd, 0x9a, 0xed, 0x7c, 0xf7, 0xf3, 0xaf, 0xd5,
  0xdb, 0xef, 0x6e, 0xb7, 0x9a, 0xbf, 0x79, 0xdd, 0x67, 0x7e, 0xd7, 0xeb, 0xb7, 0xed, 0xdf, 0xae, 0x3e, 0x55, 0xfb, 0xb3,
  0xfd, 0x5f, 0xeb, 0x7e, 0xe7, 0xf3, 0xbe, 0xfd, 0x5c, 0xaf, 0x56, 0xf7, 0xdb, 0xb9, 0x7d, 0xcf, 0xef, 0x2e, 0xf7, 0x79,
  0xbf, 0xf4, 0x9f, 0xcf, 0x7a, 0xf3, 0xe9, 0x3f, 0xbc, 0xdf, 0x6d, 0x76, 0xf7, 0xfb, 0x9e, 0xb9, 0xdf, 0xef, 0x7a, 0xb4,
  0x7b, 0x3d, 0xfd, 0xc7, 0xe7, 0x5e, 0xf3, 0xfa, 0xaf, 0xdd, 0x9e, 0xeb, 0x7c, 0x77, 0xf3, 0xbf, 0xcd, 0xd3, 0xef, 0x6e,
  0xf6, 0xba, 0x9f, 0x7d, 0xdb, 0xe5, 0x7e, 0xd7, 0xfb, 0xaf, 0xe5, 0xdf, 0xae, 0x7e, 0x35, 0xdb, 0xb7, 0xf9, 0xdd, 0x6b,
  0x7e, 0xe7, 0xeb, 0xb6, 0xfd, 0x5d, 0xee, 0x36, 0xd7, 0xdb, 0xbb, 0xfd, 0x4f, 0xef, 0x3e, 0xe7, 0x71, 0xbf, 0xf5, 0xde,
  0x8f, 0x5a, 0xf7, 0xeb, 0xbd, 0x3c, 0xdf, 0x6d, 0x6e, 0xf7, 0xfb, 0x9f, 0xf8, 0x9f, 0xcf, 0x7e, 0xb2, 0xf9, 0x3d, 0xfd,
  0xd7, 0xef, 0x56, 0xf3, 0xfb, 0xae, 0xbd, 0x9e, 0xef, 0x78, 0xf5, 0x73, 0xbf, 0xdd, 0xcb, 0xe7, 0x6e, 0xf7, 0xba, 0xbf,
  0x5d, 0xdf, 0xe3, 0x7c, 0x57, 0xfb, 0xbf, 0xed, 0xd7, 0xae, 0x7e, 0x74, 0xbb, 0x97, 0xfd, 0xdb, 0xe9, 0x7e, 0xe7, 0xfb,
  0xae, 0xf5, 0x5d, 0xef, 0x76, 0xb7, 0xdb, 0xbb, 0xf9, 0xcd, 0x6f, 0x3e, 0xf7, 0x69, 0xb7, 0xf5, 0xdf, 0xce, 0x3a, 0xd7,
  0xeb, 0xbb, 0xbc, 0x5f, 0x6d, 0x7e, 0xe7, 0xf3, 0x9f, 0xf9, 0xde, 0xaf, 0x5e, 0xb6, 0xfb, 0x3d, 0x7d, 0xd7, 0xef, 0x4e,
  0xf3, 0xfb, 0xaf, 0xfc, 0x9e, 0xcf, 0x7c, 0xf3, 0xf1, 0x3f, 0xdd, 0xdb, 0xef, 0x66, 0xf7, 0xba, 0xbe, 0x3d, 0xdf, 0xe7,
  0x7a, 0xd5, 0x7b, 0xbf, 0xed, 0xcf, 0xa6, 0x7e, 0x75, 0xfa, 0xb7, 0xdd, 0xdf, 0xeb, 0x7c, 0x67, 0xfb, 0xbe, 0xed, 0x55,
  0xef, 0x76, 0xf6, 0x9b, 0x9b, 0xfd, 0xcb, 0xed, 0x3e, 0xf7, 0x79, 0xaf, 0xf5, 0xdf, 0xcf, 0x7a, 0xb7, 0xcb, 0xbf, 0xb8,
  0xdd, 0x6d, 0x7e, 0xf7, 0xeb, 0x97, 0xf9, 0xdf, 0xee, 0x3e, 0x96, 0xfb, 0x39, 0xfd, 0x57, 0xef, 0x5e, 0xe3, 0xf3, 0xaf,
  0xfd, 0x9e, 0xaf, 0x5c, 0xf7, 0xf3, 0xbd, 0x5d, 0xdb, 0xef, 0x6e, 0xf7, 0xba, 0xbf, 0x7c, 0x9f, 0xc7, 0x7e, 0xd3, 0xf9,
  0x3f, 0xed, 0xdf, 0xae, 0x76, 0x75, 0xfb, 0xb6, 0xbd, 0xdf, 0xeb, 0x7a, 0xe5, 0x7b, 0xbe, 0xfd, 0x4d, 0xe7, 0x76, 0xf7,
  0xda, 0xbb, 0xdd, 0xcf, 0xeb, 0x3c, 0x77, 0x79, 0xbf, 0xe5, 0xd7, 0xcf, 0x7a, 0xf6, 0xab, 0x9f, 0xbc, 0xdb, 0x6d, 0x7e,
  0xf7, 0xfb, 0x8f, 0xf1, 0xdf, 0xef, 0x7e, 0xb6, 0xdb, 0x3d, 0xf9, 0xd5, 0x6f, 0x5e, 0xf3, 0xeb, 0xa7, 0xfd, 0x9e, 0xee,
  0x3c, 0xd7, 0xf3, 0xbb, 0xdd, 0x5b, 0xef, 0x6e, 0xe7, 0xb2, 0xbf, 0x7d, 0xde, 0xa7, 0x5e, 0xd7, 0xfb, 0xbd, 0x6d, 0xdf,
  0xae, 0x6e, 0x75, 0xfb, 0xb7, 0xfc, 0x9f, 0xcb, 0x7e, 0xe3, 0xf9, 0x3e, 0xfd, 0x5d, 0xef, 0x76, 0xf7, 0xdb, 0xba, 0xbd,
  0xcf, 0xef, 0x3a, 0xf5, 0x79, 0xbf, 0xf5, 0xcf, 0xc7, 0x7a, 0xf7, 0xea, 0xbf, 0x9c, 0xdf, 0x69, 0x7c, 0x77, 0xfb, 0x9f,
  0xe9, 0xd7, 0xef, 0x7e, 0xb6, 0xbb, 0x1d, 0xfd, 0xd3, 0xed, 0x5e, 0xf3, 0xfb, 0xaf, 0xf5, 0x9e, 0xef, 0x7c, 0xb7, 0xd3,
  0xbf, 0xd9, 0xd9, 0x6f, 0x6e, 0xf7, 0xaa, 0xb7, 0x7d, 0xdf, 0xe6, 0x3e, 0xd7, 0xfb, 0xbb, 0xed, 0x5f, 0xae, 0x7e, 0x65,
  0xf3, 0xb7, 0xfd, 0xde, 0xab, 0x5e, 0xe7, 0xfb, 0xbc, 0x7d, 0x5d, 0xef, 0x66, 0xf7, 0xdb, 0xbb, 0xfc, 0x8f, 0xcf, 0x3e,
  0xf3, 0x79, 0x3f, 0xf5, 0xdf, 0xcf, 0x72, 0xf7, 0xeb, 0xbe, 0xbc, 0xdf, 0x6d, 0x7a, 0xf5, 0x7b, 0x9f, 0xf9, 0xcf, 0xe7,
  0x7e, 0xb6, 0xfa, 0x3d, 0xdd, 0xd7, 0xeb, 0x5c, 0x73, 0xfb, 0xaf, 0xed, 0x96, 0xef, 0x7c, 0xf6, 0xb3, 0x9f, 0xdd, 0xdb,
  0xed, 0x6e, 0xf7, 0xba, 0xaf, 0x75, 0xdf, 0xe7, 0x7e, 0x97, 0xdb, 0xbf, 0xe9, 0xdd, 0x2e, 0x7e, 0x75, 0xeb, 0xb7, 0xfd,
  0xdf, 0xea, 0x3e, 0xc7, 0xfb, 0xba, 0xfd, 0x5d, 0xef, 0x76, 0xe7, 0xd3, 0xbb, 0xfd, 0xce, 0xaf, 0x1e, 0xf7, 0x79, 0xbd,
  0x75, 0xdf, 0xcf, 0x6a, 0xf7, 0xeb, 0xbf, 0xbc, 0x9f, 0x4d, 0x7e, 0xf3, 0xf9, 0x1f, 0xf9, 0xdf, 0xef, 0x76, 0xb6, 0xfb,
  0x3c, 0xbd, 0xd7, 0xef, 0x5a, 0xf1, 0x7b, 0xaf, 0xfd, 0x8e, 0xe7, 0x7c, 0xf7, 0xf2, 0xbf, 0xdd, 0xdb, 0xeb, 0x6c, 0x77,
  0xba, 0xbf, 0x6d, 0xd7, 0xe7, 0x7e, 0xd6, 0xbb, 0x9f, 0xed, 0xdb, 0xac, 0x7e, 0x75, 0xfb, 0xa7, 0xf5, 0xdf, 0xeb, 0x7e,
  0xa7, 0xdb, 0xbe, 0xf9, 0x5d, 0x6f, 0x76, 0xf7, 0xcb, 0xb3, 0xfd, 0xcf, 0xee, 0x3e, 0xd7, 0x79, 0xbb, 0xf5, 0x5f, 0xcf,
  0x7a, 0xe7, 0xe3, 0xbf, 0xbc, 0xde, 0x2d, 0x5e, 0xf7, 0xfb, 0x9d, 0x79, 0xdf, 0xef, 0x6e, 0xb6, 0xfb, 0x3d, 0xfc, 0x97,
  0xcf, 0x5e, 0xf3, 0xf9, 0x2f, 0xfd, 0x9e, 0xef, 0x74, 0xf7, 0xf3, 0xbe, 0x9d, 0xdb, 0xef, 0x6a, 0xf5, 0x3a, 0xbf, 0x7d,
  0xcf, 0xe7, 0x7e, 0xd7, 0xfa, 0xbf, 0xcd, 0xdf, 0xaa, 0x7c, 0x75, 0xfb, 0xb7, 0xed, 0xd7, 0xeb, 0x7e, 0xe6, 0xbb, 0x9e,
  0xfd, 0x59, 0xed, 0x76, 0xf7, 0xdb, 0xab, 0xf5, 0xcf, 0xef, 0x3e, 0xb7, 0x59, 0xbf, 0xf1, 0xdd, 0x4f, 0x7a, 0xf7, 0xeb,
  0xb7, 0xbc, 0xdf, 0x6c, 0x3e, 0xd7, 0xfb, 0x9b, 0xf9, 0x5f, 0xef, 0x7e, 0xa6, 0xf3, 0x3d, 0xfd, 0xd6, 0xaf, 0x5e, 0xf3,
  0xfb, 0xad, 0x7d, 0x9e, 0xef, 0x6c, 0xf7, 0xf3, 0xbf, 0xdc, 0x9b, 0xcf, 0x6e, 0xf3, 0xb8, 0x3f, 0x7d, 0xdf, 0xe7, 0x76,
  0xd7, 0xfb, 0xbe, 0xad, 0xdf, 0xae, 0x7a, 0x75, 0x7b, 0xb7, 0xfd, 0xcf, 0xe3, 0x7e, 0xe7, 0xfa, 0xbe, 0xdd, 0x5d, 0xeb,
  0x74, 0x77, 0xdb, 0xbb, 0xed, 0xc7, 0xef, 0x3e, 0xf6, 0x39, 0x9f, 0xf5, 0xdb, 0xcd, 0x7a, 0xf7, 0xeb, 0xaf, 0xb4, 0xdf,
  0x6d, 0x7e, 0xb7, 0xdb, 0x9f, 0xf9, 0xdd, 0x6f, 0x7e, 0xb6, 0xeb, 0x35, 0xfd, 0xd7, 0xee, 0x1e, 0xd3, 0xfb, 0xab, 0xfd,
  0x1e, 0xef, 0x7c, 0xe7, 0xf3, 0xbf, 0xdd, 0xda, 0xaf, 0x4e, 0xf7, 0xba, 0xbd, 0x7d, 0xdf, 0xe7, 0x6e, 0xd7, 0xfb, 0xbf,
  0xec, 0x9f, 0x8e, 0x7e, 0x71, 0xf9, 0x37, 0xfd, 0xdf, 0xeb, 0x76, 0xe7, 0xfb, 0xbe, 0xbd, 0x5d, 0xef, 0x72, 0xf5, 0x5b,
  0xbb, 0xfd, 0xcf, 0xe7, 0x3e, 0xf7, 0x78, 0xbf, 0xd5, 0xdf, 0xcb, 0x78, 0x77, 0xeb, 0xbf, 0xac, 0xd7, 0x6d, 0x7e, 0xf6,
  0xbb, 0x9f, 0xf9, 0xdb, 0xed, 0x7e, 0xb6, 0xfb, 0x2d, 0xf5, 0xd7, 0xef, 0x5e, 0xb3, 0xdb, 0xaf, 0xf9, 0x9c, 0x6f, 0x7c,
  0xf7, 0xe3, 0xb7, 0xdd, 0xdb, 0xee, 0x2e, 0xd7, 0xba, 0xbb, 0x7d, 0x5f, 0xe7, 0x7e, 0xc7, 0xf3, 0xbf, 0xed, 0xde, 0xae,
  0x5e, 0x75, 0xfb, 0xb5, 0x7d, 0xdf, 0xeb, 0x6e, 0xe7, 0xfb, 0xbe, 0xfc, 0x1d, 0xcf, 0x76, 0xf3, 0xd9, 0x3b, 0xfd, 0xcf,
  0xef, 0x36, 0xf7, 0x79, 0xbe, 0xb5, 0xdf, 0xcf, 0x7a, 0xf5, 0x6b, 0xbf, 0xbc, 0xcf, 0x65, 0x7e, 0xf7, 0xfa, 0x9f, 0xd9,
  0xdf, 0xeb, 0x7c, 0x36, 0xfb, 0x3d, 0xed, 0xd7, 0xef, 0x5e, 0xf2, 0xbb, 0x8f, 0xfd, 0x9a, 0xed, 0x7c, 0xf7, 0xf3, 0xaf,
  0xd5, 0xdb, 0xef, 0x6e, 0xb7, 0x9a, 0xbf, 0x79, 0xdd, 0x67, 0x7e, 0xd7, 0xeb, 0xb7, 0xed, 0xdf, 0xae, 0x3e, 0x55, 0xfb,
  0xb3, 0xfd, 0x5f, 0xeb, 0x7e, 0xe7, 0xf3, 0xbe, 0xfd, 0x5c, 0xaf, 0x56, 0xf7, 0xdb, 0xb9, 0x7d, 0xcf, 0xef, 0x2e, 0xf7,
  0x79, 0xbf, 0xf4, 0x9f, 0xcf, 0x7a, 0xf3, 0xe9, 0x3f, 0xbc, 0xdf, 0x6d, 0x76, 0xf7, 0xfb, 0x9e, 0xb9, 0xdf, 0xef, 0x7a,
  0xb4, 0x7b, 0x3d, 0xfd, 0xc7, 0xe7, 0x5e, 0xf3, 0xfa, 0xaf, 0xdd, 0x9e, 0xeb, 0x7c, 0x77, 0xf3, 0xbf, 0xcd, 0xd3, 0xef,
  0x6e, 0xf6, 0xba, 0x9f, 0x7d, 0xdb, 0xe5, 0x7e, 0xd7, 0xfb, 0xaf, 0xe5, 0xdf, 0xae, 0x7e, 0x35, 0xdb, 0xb7, 0xf9, 0xdd,
  0x6b, 0x7e, 0xe7, 0xeb, 0xb6, 0xfd, 0x5d, 0xee, 0x36, 0xd7, 0xdb, 0xbb, 0xfd, 0x4f, 0xef, 0x3e, 0xe7, 0x71, 0xbf, 0xf5,
  0xde, 0x8f, 0x5a, 0xf7, 0xeb, 0xbd, 0x3c, 0xdf, 0x6d, 0x6e, 0xf7, 0xfb, 0x9f, 0xf8, 0x9f, 0xcf, 0x7e, 0xb2, 0xf9, 0x3d,
  0xfd, 0xd7, 0xef, 0x56, 0xf3, 0xfb, 0xae, 0xbd, 0x9e, 0xef, 0x78, 0xf5, 0x73, 0xbf, 0xdd, 0xcb, 0xe7, 0x6e, 0xf7, 0xba,
  0xbf, 0x5d, 0xdf, 0xe3, 0x7c, 0x57, 0xfb, 0xbf, 0xed, 0xd7, 0xae, 0x7e, 0x74, 0xbb, 0x97, 0xfd, 0xdb, 0xe9, 0x7e, 0xe7,
  0xfb, 0xae, 0xf5, 0x5d, 0xef, 0x76, 0xb7, 0xdb, 0xbb, 0xf9, 0xcd, 0x6f, 0x3e, 0xf7, 0x69, 0xb7, 0xf5, 0xdf, 0xce, 0x3a,
  0xd7, 0xeb, 0xbb, 0xbc, 0x5f, 0x6d, 0x7e, 0xe7, 0xf3, 0x9f, 0xf9, 0xde, 0xaf, 0x5e, 0xb6, 0xfb, 0x3d, 0x7d, 0xd7, 0xef,
  0x4e, 0xf3, 0xfb, 0xaf, 0xfc, 0x9e, 0xcf, 0x7c, 0xf3, 0xf1, 0x3f, 0xdd, 0xdb, 0xef, 0x66, 0xf7, 0xba, 0xbe, 0x3d, 0xdf,
  0xe7, 0x7a, 0xd5, 0x7b, 0xbf, 0xed, 0xcf, 0xa6, 0x7e, 0x75, 0xfa, 0xb7, 0xdd, 0xdf, 0xeb, 0x7c, 0x67, 0xfb, 0xbe, 0xed,
  0x55, 0xef, 0x76, 0xf6, 0x9b, 0x9b, 0xfd, 0xcb, 0xed, 0x3e, 0xf7, 0x79, 0xaf, 0xf5, 0xdf, 0xcf, 0x7a, 0xb7, 0xcb, 0xbf,
  0xb8, 0xdd, 0x6d, 0x7e, 0xf7, 0xeb, 0x97, 0xf9, 0xdf, 0xee, 0x3e, 0x96, 0xfb, 0x39, 0xfd, 0x57, 0xef, 0x5e, 0xe3, 0xf3,
  0xaf, 0xfd, 0x9e, 0xaf, 0x5c, 0xf7, 0xf3, 0xbd, 0x5d, 0xdb, 0xef, 0x6e, 0xf7, 0xba, 0xbf, 0x7c, 0x9f, 0xc7, 0x7e, 0xd3,
  0xf9, 0x3f, 0xed, 0xdf, 0xae, 0x76, 0x75, 0xfb, 0xb6, 0xbd, 0xdf, 0xeb, 0x7a, 0xe5, 0x7b, 0xbe, 0xfd, 0x4d, 0xe7, 0x76,
  0xf7, 0xda, 0xbb, 0xdd, 0xcf, 0xeb, 0x3c, 0x77, 0x79, 0xbf, 0xe5, 0xd7, 0xcf, 0x7a, 0xf6, 0xab, 0x9f, 0xbc, 0xdb, 0x6d,
  0x7e, 0xf7, 0xfb, 0x8f
};

#if 0

/// C++ program that generates the pre_sieved arrays
/// used in Sieve_pre_sieve.hpp.
///

#include <Sieve.hpp>
#include <generate_primes.hpp>

#include <iostream>
#include <iomanip>
#include <vector>

int main()
{
  const std::vector<std::vector<long>> vect =
  {{
    { 17, 19 },
    { 23, 29 },
    { 31, 37 },
    { 41, 43 },
    { 47, 53 },
    { 59, 61 },
    { 67, 71 }
  }};

  std::cout << "const primecount::Array<std::initializer_list<uint8_t>, "
            << vect.size() << "> pre_sieved_arrays =" << std::endl;
  std::cout << "{{" << std::endl;

  for (const auto& v : vect)
  {
    long min_size = 3000; // bytes
    long prime_product = 1;
    for (long prime : v) prime_product *= prime;
    long factor = 2;
    while (prime_product * factor < min_size) factor++;
    long bytes = prime_product * factor;

    using namespace primecount;
    auto primes = generate_primes<int>(vect.back().back());
    Sieve sieve(prime_product * 30, prime_product * 30 * factor, primes.size());
    std::fill_n(sieve.sieve_.data(), sieve.sieve_.size(), 0xff);

    for (unsigned i = 0; i < v.size(); i++)
      sieve.cross_off(v[i], i + 4);

    std::cout << "  { ";

    for (long j = 0; j < bytes; j++)
    {
      if (j != 0 && j % 20 == 0)
        std::cout << std::endl << "    ";
      std::cout << "0x" << std::hex << std::setw(2) << std::setfill('0') << (int) sieve.sieve_[j];
      if (j + 1 < bytes)
        std::cout << ((j+1) % 20 == 0 ? "," : ", ");
      else
        std::cout << (v.back() < vect.back().back() ? " }," : " }") << std::endl;
    }
  }

  std::cout << "}};" << std::endl;

  return 0;
}

#endif

/// Static arrays from which the primes and multiples of primes have
/// been removed. Our Sieve::pre_sieve() method requires that each
/// pre_sieved lookup table corresponds to exactly 2 sieving primes.
///
/// pre_sieved_arrays[0] = { 17, 19 }
/// pre_sieved_arrays[1] = { 23, 29 }
/// pre_sieved_arrays[2] = { 31, 37 }
/// pre_sieved_arrays[3] = { 41, 43 }
/// pre_sieved_arrays[4] = { 47, 53 }
/// pre_sieved_arrays[5] = { 59, 61 }
/// pre_sieved_arrays[6] = { 67, 71 }
///
const primecount::Array<std::initializer_list<uint8_t>, 7> pre_sieved_arrays =
{{
  { 0xcf, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0x7d, 0xfb, 0xfd, 0xdf, 0xbf, 0xff, 0xfe, 0xfe, 0xef, 0xff, 0xf7, 0xef, 0xfb, 0xdf,
    0x7f, 0xff, 0xff, 0xf5, 0xfb, 0x7f, 0xdf, 0xbd, 0xff, 0xbf, 0xfe, 0xfe, 0xff, 0xe7, 0xef, 0xff, 0xff, 0x7b, 0xdf, 0xff,
    0xfd, 0xfb, 0xf7, 0xdf, 0x3f, 0xff, 0xfd, 0xfe, 0xbf, 0xff, 0xf6, 0xef, 0xef, 0xff, 0x7f, 0xff, 0xfb, 0xdd, 0xfb, 0xff,
    0xdf, 0xb7, 0xff, 0x7f, 0xfe, 0xfd, 0xff, 0xb7, 0xef, 0xfe, 0xff, 0x6f, 0xff, 0xff, 0xfd, 0xfb, 0xdf, 0xdf, 0xbf, 0xff,
    0xf7, 0xfe, 0x7f, 0xff, 0xf5, 0xef, 0xbf, 0xff, 0x7e, 0xff, 0xef, 0xfd, 0xfb, 0xff, 0xdb, 0x9f, 0xff, 0xff, 0xfe, 0xf7,
    0xff, 0x77, 0xef, 0xfd, 0xff, 0x3f, 0xff, 0xfe, 0xfd, 0xeb, 0xff, 0xdf, 0xbf, 0xfb, 0xdf, 0xfe, 0xff, 0xff, 0xf7, 0xef,
    0x7f, 0xff, 0x7d, 0xff, 0xbf, 0xfd, 0xfa, 0xff, 0xcf, 0xbf, 0xff, 0xff, 0xfa, 0xdf, 0xff, 0xf7, 0xef, 0xf7, 0xff, 0x7f,
    0xff, 0xfd, 0xfd, 0xbb, 0xff, 0xde, 0xbf, 0xef, 0xff, 0xfe, 0xff, 0xfb, 0xd7, 0xef, 0xff, 0xff, 0x77, 0xff, 0x7f, 0xfd,
    0xf9, 0xff, 0x9f, 0xbf, 0xfe, 0xff, 0xee, 0xff, 0xff, 0xf7, 0xeb, 0xdf, 0xff, 0x7f, 0xff, 0xf7, 0xfd, 0x7b, 0xff, 0xdd,
    0xbf, 0xbf, 0xff, 0xfe, 0xff, 0xef, 0xf7, 0xef, 0xff, 0xfb, 0x5f, 0xff, 0xff, 0xfd, 0xf3, 0xff, 0x5f, 0xbf, 0xfd, 0xff,
    0xbe, 0xff, 0xfe, 0xf7, 0xef, 0xff, 0xff, 0x7f, 0xfb, 0xdf, 0xfd, 0xfb, 0xff, 0xd7, 0xbf, 0x7f, 0xff, 0xfc, 0xff, 0xbf,
    0xf7, 0xee, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xf9, 0xdb, 0xff, 0xdf, 0xbf, 0xf7, 0xff, 0x7e, 0xff, 0xfd, 0xf7, 0xaf, 0xff,
    0xfe, 0x7f, 0xef, 0xff, 0xfd, 0xfb, 0xfb, 0xdf, 0xbf, 0xff, 0xff, 0xf6, 0xff, 0x7f, 0xf7, 0xed, 0xff, 0xbf, 0x7f, 0xfe,
    0xff, 0xed, 0xfb, 0xff, 0xdf, 0xbb, 0xdf, 0xff, 0xfe, 0xff, 0xf7, 0xf7, 0x6f, 0xff, 0xfd, 0x7f, 0xbf, 0xff, 0xfc, 0xfb,
    0xef, 0xdf, 0xbf, 0xff, 0xfb, 0xde, 0xff, 0xff, 0xf7, 0xe7, 0xff, 0x7f, 0x7f, 0xfd, 0xff, 0xbd, 0xfb, 0xfe, 0xdf, 0xaf,
    0xff, 0xff, 0xfe, 0xfb, 0xdf, 0xf7, 0xef, 0xff, 0xf7, 0x7f, 0x7f, 0xff, 0xfd, 0xfb, 0xbf, 0xdf, 0xbe, 0xff, 0xef, 0xfe,
    0xff, 0xff, 0xf3, 0xcf, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0x7d, 0xfb, 0xfd, 0xdf, 0xbf, 0xff, 0xfe, 0xfe, 0xef, 0xff, 0xf7,
    0xef, 0xfb, 0xdf, 0x7f, 0xff, 0xff, 0xf5, 0xfb, 0x7f, 0xdf, 0xbd, 0xff, 0xbf, 0xfe, 0xfe, 0xff, 0xe7, 0xef, 0xff, 0xff,
    0x7b, 0xdf, 0xff, 0xfd, 0xfb, 0xf7, 0xdf, 0x3f, 0xff, 0xfd, 0xfe, 0xbf, 0xff, 0xf6, 0xef, 0xef, 0xff, 0x7f, 0xff, 0xfb,
    0xdd, 0xfb, 0xff, 0xdf, 0xb7, 0xff, 0x7f, 0xfe, 0xfd, 0xff, 0xb7, 0xef, 0xfe, 0xff, 0x6f, 0xff, 0xff, 0xfd, 0xfb, 0xdf,
    0xdf, 0xbf, 0xff, 0xf7, 0xfe, 0x7f, 0xff, 0xf5, 0xef, 0xbf, 0xff, 0x7e, 0xff, 0xef, 0xfd, 0xfb, 0xff, 0xdb, 0x9f, 0xff,
    0xff, 0xfe, 0xf7, 0xff, 0x77, 0xef, 0xfd, 0xff, 0x3f, 0xff, 0xfe, 0xfd, 0xeb, 0xff, 0xdf, 0xbf, 0xfb, 0xdf, 0xfe, 0xff,
    0xff, 0xf7, 0xef, 0x7f, 0xff, 0x7d, 0xff, 0xbf, 0xfd, 0xfa, 0xff, 0xcf, 0xbf, 0xff, 0xff, 0xfa, 0xdf, 0xff, 0xf7, 0xef,
    0xf7, 0xff, 0x7f, 0xff, 0xfd, 0xfd, 0xbb, 0xff, 0xde, 0xbf, 0xef, 0xff, 0xfe, 0xff, 0xfb, 0xd7, 0xef, 0xff, 0xff, 0x77,
    0xff, 0x7f, 0xfd, 0xf9, 0xff, 0x9f, 0xbf, 0xfe, 0xff, 0xee, 0xff, 0xff, 0xf7, 0xeb, 0xdf, 0xff, 0x7f, 0xff, 0xf7, 0xfd,
    0x7b, 0xff, 0xdd, 0xbf, 0xbf, 0xff, 0xfe, 0xff, 0xef, 0xf7, 0xef, 0xff, 0xfb, 0x5f, 0xff, 0xff, 0xfd, 0xf3, 0xff, 0x5f,
    0xbf, 0xfd, 0xff, 0xbe, 0xff, 0xfe, 0xf7, 0xef, 0xff, 0xff, 0x7f, 0xfb, 0xdf, 0xfd, 0xfb, 0xff, 0xd7, 0xbf, 0x7f, 0xff,
    0xfc, 0xff, 0xbf, 0xf7, 0xee, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xf9, 0xdb, 0xff, 0xdf, 0xbf, 0xf7, 0xff, 0x7e, 0xff, 0xfd,
    0xf7, 0xaf, 0xff, 0xfe, 0x7f, 0xef, 0xff, 0xfd, 0xfb, 0xfb, 0xdf, 0xbf, 0xff, 0xff, 0xf6, 0xff, 0x7f, 0xf7, 0xed, 0xff,
    0xbf, 0x7f, 0xfe, 0xff, 0xed, 0xfb, 0xff, 0xdf, 0xbb, 0xdf, 0xff, 0xfe, 0xff, 0xf7, 0xf7, 0x6f, 0xff, 0xfd, 0x7f, 0xbf,
    0xff, 0xfc, 0xfb, 0xef, 0xdf, 0xbf, 0xff, 0xfb, 0xde, 0xff, 0xff, 0xf7, 0xe7, 0xff, 0x7f, 0x7f, 0xfd, 0xff, 0xbd, 0xfb,
    0xfe, 0xdf, 0xaf, 0xff, 0xff, 0xfe, 0xfb, 0xdf, 0xf7, 0xef, 0xff, 0xf7, 0x7f, 0x7f, 0xff, 0xfd, 0xfb, 0xbf, 0xdf, 0xbe,
    0xff, 0xef, 0xfe, 0xff, 0xff, 0xf3, 0xcf, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0x7d, 0xfb, 0xfd, 0xdf, 0xbf, 0xff, 0xfe, 0xfe,
    0xef, 0xff, 0xf7, 0xef, 0xfb, 0xdf, 0x7f, 0xff, 0xff, 0xf5, 0xfb, 0x7f, 0xdf, 0xbd, 0xff, 0xbf, 0xfe, 0xfe, 0xff, 0xe7,
    0xef, 0xff, 0xff, 0x7b, 0xdf, 0xff, 0xfd, 0xfb, 0xf7, 0xdf, 0x3f, 0xff, 0xfd, 0xfe, 0xbf, 0xff, 0xf6, 0xef, 0xef, 0xff,
    0x7f, 0xff, 0xfb, 0xdd, 0xfb, 0xff, 0xdf, 0xb7, 0xff, 0x7f, 0xfe, 0xfd, 0xff, 0xb7, 0xef, 0xfe, 0xff, 0x6f, 0xff, 0xff,
    0xfd, 0xfb, 0xdf, 0xdf, 0xbf, 0xff, 0xf7, 0xfe, 0x7f, 0xff, 0xf5, 0xef, 0xbf, 0xff, 0x7e, 0xff, 0xef, 0xfd, 0xfb, 0xff,
    0xdb, 0x9f, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0x77, 0xef, 0xfd, 0xff, 0x3f, 0xff, 0xfe, 0xfd, 0xeb, 0xff, 0xdf, 0xbf, 0xfb,
    0xdf, 0xfe, 0xff, 0xff, 0xf7, 0xef, 0x7f, 0xff, 0x7d, 0xff, 0xbf, 0xfd, 0xfa, 0xff, 0xcf, 0xbf, 0xff, 0xff, 0xfa, 0xdf,
    0xff, 0xf7, 0xef, 0xf7, 0xff, 0x7f, 0xff, 0xfd, 0xfd, 0xbb, 0xff, 0xde, 0xbf, 0xef, 0xff, 0xfe, 0xff, 0xfb, 0xd7, 0xef,
    0xff, 0xff, 0x77, 0xff, 0x7f, 0xfd, 0xf9, 0xff, 0x9f, 0xbf, 0xfe, 0xff, 0xee, 0xff, 0xff, 0xf7, 0xeb, 0xdf, 0xff, 0x7f,
    0xff, 0xf7, 0xfd, 0x7b, 0xff, 0xdd, 0xbf, 0xbf, 0xff, 0xfe, 0xff, 0xef, 0xf7, 0xef, 0xff, 0xfb, 0x5f, 0xff, 0xff, 0xfd,
    0xf3, 0xff, 0x5f, 0xbf, 0xfd, 0xff, 0xbe, 0xff, 0xfe, 0xf7, 0xef, 0xff, 0xff, 0x7f, 0xfb, 0xdf, 0xfd, 0xfb, 0xff, 0xd7,
    0xbf, 0x7f, 0xff, 0xfc, 0xff, 0xbf, 0xf7, 0xee, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xf9, 0xdb, 0xff, 0xdf, 0xbf, 0xf7, 0xff,
    0x7e, 0xff, 0xfd, 0xf7, 0xaf, 0xff, 0xfe, 0x7f, 0xef, 0xff, 0xfd, 0xfb, 0xfb, 0xdf, 0xbf, 0xff, 0xff, 0xf6, 0xff, 0x7f,
    0xf7, 0xed, 0xff, 0xbf, 0x7f, 0xfe, 0xff, 0xed, 0xfb, 0xff, 0xdf, 0xbb, 0xdf, 0xff, 0xfe, 0xff, 0xf7, 0xf7, 0x6f, 0xff,
    0xfd, 0x7f, 0xbf, 0xff, 0xfc, 0xfb, 0xef, 0xdf, 0xbf, 0xff, 0xfb, 0xde, 0xff, 0xff, 0xf7, 0xe7, 0xff, 0x7f, 0x7f, 0xfd,
    0xff, 0xbd, 0xfb, 0xfe, 0xdf, 0xaf, 0xff, 0xff, 0xfe, 0xfb, 0xdf, 0xf7, 0xef, 0xff, 0xf7, 0x7f, 0x7f, 0xff, 0xfd, 0xfb,
    0xbf, 0xdf, 0xbe, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xf3, 0xcf, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0x7d, 0xfb, 0xfd, 0xdf, 0xbf,
    0xff, 0xfe, 0xfe, 0xef, 0xff, 0xf7, 0xef, 0xfb, 0xdf, 0x7f, 0xff, 0xff, 0xf5, 0xfb, 0x7f, 0xdf, 0xbd, 0xff, 0xbf, 0xfe,
    0xfe, 0xff, 0xe7, 0xef, 0xff, 0xff, 0x7b, 0xdf, 0xff, 0xfd, 0xfb, 0xf7, 0xdf, 0x3f, 0xff, 0xfd, 0xfe, 0xbf, 0xff, 0xf6,
    0xef, 0xef, 0xff, 0x7f, 0xff, 0xfb, 0xdd, 0xfb, 0xff, 0xdf, 0xb7, 0xff, 0x7f, 0xfe, 0xfd, 0xff, 0xb7, 0xef, 0xfe, 0xff,
    0x6f, 0xff, 0xff, 0xfd, 0xfb, 0xdf, 0xdf, 0xbf, 0xff, 0xf7, 0xfe, 0x7f, 0xff, 0xf5, 0xef, 0xbf, 0xff, 0x7e, 0xff, 0xef,
    0xfd, 0xfb, 0xff, 0xdb, 0x9f, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0x77, 0xef, 0xfd, 0xff, 0x3f, 0xff, 0xfe, 0xfd, 0xeb, 0xff,
    0xdf, 0xbf, 0xfb, 0xdf, 0xfe, 0xff, 0xff, 0xf7, 0xef, 0x7f, 0xff, 0x7d, 0xff, 0xbf, 0xfd, 0xfa, 0xff, 0xcf, 0xbf, 0xff,
    0xff, 0xfa, 0xdf, 0xff, 0xf7, 0xef, 0xf7, 0xff, 0x7f, 0xff, 0xfd, 0xfd, 0xbb, 0xff, 0xde, 0xbf, 0xef, 0xff, 0xfe, 0xff,
    0xfb, 0xd7, 0xef, 0xff, 0xff, 0x77, 0xff, 0x7f, 0xfd, 0xf9, 0xff, 0x9f, 0xbf, 0xfe, 0xff, 0xee, 0xff, 0xff, 0xf7, 0xeb,
    0xdf, 0xff, 0x7f, 0xff, 0xf7, 0xfd, 0x7b, 0xff, 0xdd, 0xbf, 0xbf, 0xff, 0xfe, 0xff, 0xef, 0xf7, 0xef, 0xff, 0xfb, 0x5f,
    0xff, 0xff, 0xfd, 0xf3, 0xff, 0x5f, 0xbf, 0xfd, 0xff, 0xbe, 0xff, 0xfe, 0xf7, 0xef, 0xff, 0xff, 0x7f, 0xfb, 0xdf, 0xfd,
    0xfb, 0xff, 0xd7, 0xbf, 0x7f, 0xff, 0xfc, 0xff, 0xbf, 0xf7, 0xee, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xf9, 0xdb, 0xff, 0xdf,
    0xbf, 0xf7, 0xff, 0x7e, 0xff, 0xfd, 0xf7, 0xaf, 0xff, 0xfe, 0x7f, 0xef, 0xff, 0xfd, 0xfb, 0xfb, 0xdf, 0xbf, 0xff, 0xff,
    0xf6, 0xff, 0x7f, 0xf7, 0xed, 0xff, 0xbf, 0x7f, 0xfe, 0xff, 0xed, 0xfb, 0xff, 0xdf, 0xbb, 0xdf, 0xff, 0xfe, 0xff, 0xf7,
    0xf7, 0x6f, 0xff, 0xfd, 0x7f, 0xbf, 0xff, 0xfc, 0xfb, 0xef, 0xdf, 0xbf, 0xff, 0xfb, 0xde, 0xff, 0xff, 0xf7, 0xe7, 0xff,
    0x7f, 0x7f, 0xfd, 0xff, 0xbd, 0xfb, 0xfe, 0xdf, 0xaf, 0xff, 0xff, 0xfe, 0xfb, 0xdf, 0xf7, 0xef, 0xff, 0xf7, 0x7f, 0x7f,
    0xff, 0xfd, 0xfb, 0xbf, 0xdf, 0xbe, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xf3, 0xcf, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0x7d, 0xfb,
    0xfd, 0xdf, 0xbf, 0xff, 0xfe, 0xfe, 0xef, 0xff, 0xf7, 0xef, 0xfb, 0xdf, 0x7f, 0xff, 0xff, 0xf5, 0xfb, 0x7f, 0xdf, 0xbd,
    0xff, 0xbf, 0xfe, 0xfe, 0xff, 0xe7, 0xef, 0xff, 0xff, 0x7b, 0xdf, 0xff, 0xfd, 0xfb, 0xf7, 0xdf, 0x3f, 0xff, 0xfd, 0xfe,
    0xbf, 0xff, 0xf6, 0xef, 0xef, 0xff, 0x7f, 0xff, 0xfb, 0xdd, 0xfb, 0xff, 0xdf, 0xb7, 0xff, 0x7f, 0xfe, 0xfd, 0xff, 0xb7,
    0xef, 0xfe, 0xff, 0x6f, 0xff, 0xff, 0xfd, 0xfb, 0xdf, 0xdf, 0xbf, 0xff, 0xf7, 0xfe, 0x7f, 0xff, 0xf5, 0xef, 0xbf, 0xff,
    0x7e, 0xff, 0xef, 0xfd, 0xfb, 0xff, 0xdb, 0x9f, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0x77, 0xef, 0xfd, 0xff, 0x3f, 0xff, 0xfe,
    0xfd, 0xeb, 0xff, 0xdf, 0xbf, 0xfb, 0xdf, 0xfe, 0xff, 0xff, 0xf7, 0xef, 0x7f, 0xff, 0x7d, 0xff, 0xbf, 0xfd, 0xfa, 0xff,
    0xcf, 0xbf, 0xff, 0xff, 0xfa, 0xdf, 0xff, 0xf7, 0xef, 0xf7, 0xff, 0x7f, 0xff, 0xfd, 0xfd, 0xbb, 0xff, 0xde, 0xbf, 0xef,
    0xff, 0xfe, 0xff, 0xfb, 0xd7, 0xef, 0xff, 0xff, 0x77, 0xff, 0x7f, 0xfd, 0xf9, 0xff, 0x9f, 0xbf, 0xfe, 0xff, 0xee, 0xff,
    0xff, 0xf7, 0xeb, 0xdf, 0xff, 0x7f, 0xff, 0xf7, 0xfd, 0x7b, 0xff, 0xdd, 0xbf, 0xbf, 0xff, 0xfe, 0xff, 0xef, 0xf7, 0xef,
    0xff, 0xfb, 0x5f, 0xff, 0xff, 0xfd, 0xf3, 0xff, 0x5f, 0xbf, 0xfd, 0xff, 0xbe, 0xff, 0xfe, 0xf7, 0xef, 0xff, 0xff, 0x7f,
    0xfb, 0xdf, 0xfd, 0xfb, 0xff, 0xd7, 0xbf, 0x7f, 0xff, 0xfc, 0xff, 0xbf, 0xf7, 0xee, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xf9,
    0xdb, 0xff, 0xdf, 0xbf, 0xf7, 0xff, 0x7e, 0xff, 0xfd, 0xf7, 0xaf, 0xff, 0xfe, 0x7f, 0xef, 0xff, 0xfd, 0xfb, 0xfb, 0xdf,
    0xbf, 0xff, 0xff, 0xf6, 0xff, 0x7f, 0xf7, 0xed, 0xff, 0xbf, 0x7f, 0xfe, 0xff, 0xed, 0xfb, 0xff, 0xdf, 0xbb, 0xdf, 0xff,
    0xfe, 0xff, 0xf7, 0xf7, 0x6f, 0xff, 0xfd, 0x7f, 0xbf, 0xff, 0xfc, 0xfb, 0xef, 0xdf, 0xbf, 0xff, 0xfb, 0xde, 0xff, 0xff,
    0xf7, 0xe7, 0xff, 0x7f, 0x7f, 0xfd, 0xff, 0xbd, 0xfb, 0xfe, 0xdf, 0xaf, 0xff, 0xff, 0xfe, 0xfb, 0xdf, 0xf7, 0xef, 0xff,
    0xf7, 0x7f, 0x7f, 0xff, 0xfd, 0xfb, 0xbf, 0xdf, 0xbe, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xf3, 0xcf, 0xff, 0xff, 0x7f, 0xf7,
    0xff, 0x7d, 0xfb, 0xfd, 0xdf, 0xbf, 0xff, 0xfe, 0xfe, 0xef, 0xff, 0xf7, 0xef, 0xfb, 0xdf, 0x7f, 0xff, 0xff, 0xf5, 0xfb,
    0x7f, 0xdf, 0xbd, 0xff, 0xbf, 0xfe, 0xfe, 0xff, 0xe7, 0xef, 0xff, 0xff, 0x7b, 0xdf, 0xff, 0xfd, 0xfb, 0xf7, 0xdf, 0x3f,
    0xff, 0xfd, 0xfe, 0xbf, 0xff, 0xf6, 0xef, 0xef, 0xff, 0x7f, 0xff, 0xfb, 0xdd, 0xfb, 0xff, 0xdf, 0xb7, 0xff, 0x7f, 0xfe,
    0xfd, 0xff, 0xb7, 0xef, 0xfe, 0xff, 0x6f, 0xff, 0xff, 0xfd, 0xfb, 0xdf, 0xdf, 0xbf, 0xff, 0xf7, 0xfe, 0x7f, 0xff, 0xf5,
    0xef, 0xbf, 0xff, 0x7e, 0xff, 0xef, 0xfd, 0xfb, 0xff, 0xdb, 0x9f, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0x77, 0xef, 0xfd, 0xff,
    0x3f, 0xff, 0xfe, 0xfd, 0xeb, 0xff, 0xdf, 0xbf, 0xfb, 0xdf, 0xfe, 0xff, 0xff, 0xf7, 0xef, 0x7f, 0xff, 0x7d, 0xff, 0xbf,
    0xfd, 0xfa, 0xff, 0xcf, 0xbf, 0xff, 0xff, 0xfa, 0xdf, 0xff, 0xf7, 0xef, 0xf7, 0xff, 0x7f, 0xff, 0xfd, 0xfd, 0xbb, 0xff,
    0xde, 0xbf, 0xef, 0xff, 0xfe, 0xff, 0xfb, 0xd7, 0xef, 0xff, 0xff, 0x77, 0xff, 0x7f, 0xfd, 0xf9, 0xff, 0x9f, 0xbf, 0xfe,
    0xff, 0xee, 0xff, 0xff, 0xf7, 0xeb, 0xdf, 0xff, 0x7f, 0xff, 0xf7, 0xfd, 0x7b, 0xff, 0xdd, 0xbf, 0xbf, 0xff, 0xfe, 0xff,
    0xef, 0xf7, 0xef, 0xff, 0xfb, 0x5f, 0xff, 0xff, 0xfd, 0xf3, 0xff, 0x5f, 0xbf, 0xfd, 0xff, 0xbe, 0xff, 0xfe, 0xf7, 0xef,
    0xff, 0xff, 0x7f, 0xfb, 0xdf, 0xfd, 0xfb, 0xff, 0xd7, 0xbf, 0x7f, 0xff, 0xfc, 0xff, 0xbf, 0xf7, 0xee, 0xff, 0xef, 0x7f,
    0xff, 0xff, 0xf9, 0xdb, 0xff, 0xdf, 0xbf, 0xf7, 0xff, 0x7e, 0xff, 0xfd, 0xf7, 0xaf, 0xff, 0xfe, 0x7f, 0xef, 0xff, 0xfd,
    0xfb, 0xfb, 0xdf, 0xbf, 0xff, 0xff, 0xf6, 0xff, 0x7f, 0xf7, 0xed, 0xff, 0xbf, 0x7f, 0xfe, 0xff, 0xed, 0xfb, 0xff, 0xdf,
    0xbb, 0xdf, 0xff, 0xfe, 0xff, 0xf7, 0xf7, 0x6f, 0xff, 0xfd, 0x7f, 0xbf, 0xff, 0xfc, 0xfb, 0xef, 0xdf, 0xbf, 0xff, 0xfb,
    0xde, 0xff, 0xff, 0xf7, 0xe7, 0xff, 0x7f, 0x7f, 0xfd, 0xff, 0xbd, 0xfb, 0xfe, 0xdf, 0xaf, 0xff, 0xff, 0xfe, 0xfb, 0xdf,
    0xf7, 0xef, 0xff, 0xf7, 0x7f, 0x7f, 0xff, 0xfd, 0xfb, 0xbf, 0xdf, 0xbe, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xf3, 0xcf, 0xff,
    0xff, 0x7f, 0xf7, 0xff, 0x7d, 0xfb, 0xfd, 0xdf, 0xbf, 0xff, 0xfe, 0xfe, 0xef, 0xff, 0xf7, 0xef, 0xfb, 0xdf, 0x7f, 0xff,
    0xff, 0xf5, 0xfb, 0x7f, 0xdf, 0xbd, 0xff, 0xbf, 0xfe, 0xfe, 0xff, 0xe7, 0xef, 0xff, 0xff, 0x7b, 0xdf, 0xff, 0xfd, 0xfb,
    0xf7, 0xdf, 0x3f, 0xff, 0xfd, 0xfe, 0xbf, 0xff, 0xf6, 0xef, 0xef, 0xff, 0x7f, 0xff, 0xfb, 0xdd, 0xfb, 0xff, 0xdf, 0xb7,
    0xff, 0x7f, 0xfe, 0xfd, 0xff, 0xb7, 0xef, 0xfe, 0xff, 0x6f, 0xff, 0xff, 0xfd, 0xfb, 0xdf, 0xdf, 0xbf, 0xff, 0xf7, 0xfe,
    0x7f, 0xff, 0xf5, 0xef, 0xbf, 0xff, 0x7e, 0xff, 0xef, 0xfd, 0xfb, 0xff, 0xdb, 0x9f, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0x77,
    0xef, 0xfd, 0xff, 0x3f, 0xff, 0xfe, 0xfd, 0xeb, 0xff, 0xdf, 0xbf, 0xfb, 0xdf, 0xfe, 0xff, 0xff, 0xf7, 0xef, 0x7f, 0xff,
    0x7d, 0xff, 0xbf, 0xfd, 0xfa, 0xff, 0xcf, 0xbf, 0xff, 0xff, 0xfa, 0xdf, 0xff, 0xf7, 0xef, 0xf7, 0xff, 0x7f, 0xff, 0xfd,
    0xfd, 0xbb, 0xff, 0xde, 0xbf, 0xef, 0xff, 0xfe, 0xff, 0xfb, 0xd7, 0xef, 0xff, 0xff, 0x77, 0xff, 0x7f, 0xfd, 0xf9, 0xff,
    0x9f, 0xbf, 0xfe, 0xff, 0xee, 0xff, 0xff, 0xf7, 0xeb, 0xdf, 0xff, 0x7f, 0xff, 0xf7, 0xfd, 0x7b, 0xff, 0xdd, 0xbf, 0xbf,
    0xff, 0xfe, 0xff, 0xef, 0xf7, 0xef, 0xff, 0xfb, 0x5f, 0xff, 0xff, 0xfd, 0xf3, 0xff, 0x5f, 0xbf, 0xfd, 0xff, 0xbe, 0xff,
    0xfe, 0xf7, 0xef, 0xff, 0xff, 0x7f, 0xfb, 0xdf, 0xfd, 0xfb, 0xff, 0xd7, 0xbf, 0x7f, 0xff, 0xfc, 0xff, 0xbf, 0xf7, 0xee,
    0xff, 0xef, 0x7f, 0xff, 0xff, 0xf9, 0xdb, 0xff, 0xdf, 0xbf, 0xf7, 0xff, 0x7e, 0xff, 0xfd, 0xf7, 0xaf, 0xff, 0xfe, 0x7f,
    0xef, 0xff, 0xfd, 0xfb, 0xfb, 0xdf, 0xbf, 0xff, 0xff, 0xf6, 0xff, 0x7f, 0xf7, 0xed, 0xff, 0xbf, 0x7f, 0xfe, 0xff, 0xed,
    0xfb, 0xff, 0xdf, 0xbb, 0xdf, 0xff, 0xfe, 0xff, 0xf7, 0xf7, 0x6f, 0xff, 0xfd, 0x7f, 0xbf, 0xff, 0xfc, 0xfb, 0xef, 0xdf,
    0xbf, 0xff, 0xfb, 0xde, 0xff, 0xff, 0xf7, 0xe7, 0xff, 0x7f, 0x7f, 0xfd, 0xff, 0xbd, 0xfb, 0xfe, 0xdf, 0xaf, 0xff, 0xff,
    0xfe, 0xfb, 0xdf, 0xf7, 0xef, 0xff, 0xf7, 0x7f, 0x7f, 0xff, 0xfd, 0xfb, 0xbf, 0xdf, 0xbe, 0xff, 0xef, 0xfe, 0xff, 0xff,
    0xf3, 0xcf, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0x7d, 0xfb, 0xfd, 0xdf, 0xbf, 0xff, 0xfe, 0xfe, 0xef, 0xff, 0xf7, 0xef, 0xfb,
    0xdf, 0x7f, 0xff, 0xff, 0xf5, 0xfb, 0x7f, 0xdf, 0xbd, 0xff, 0xbf, 0xfe, 0xfe, 0xff, 0xe7, 0xef, 0xff, 0xff, 0x7b, 0xdf,
    0xff, 0xfd, 0xfb, 0xf7, 0xdf, 0x3f, 0xff, 0xfd, 0xfe, 0xbf, 0xff, 0xf6, 0xef, 0xef, 0xff, 0x7f, 0xff, 0xfb, 0xdd, 0xfb,
    0xff, 0xdf, 0xb7, 0xff, 0x7f, 0xfe, 0xfd, 0xff, 0xb7, 0xef, 0xfe, 0xff, 0x6f, 0xff, 0xff, 0xfd, 0xfb, 0xdf, 0xdf, 0xbf,
    0xff, 0xf7, 0xfe, 0x7f, 0xff, 0xf5, 0xef, 0xbf, 0xff, 0x7e, 0xff, 0xef, 0xfd, 0xfb, 0xff, 0xdb, 0x9f, 0xff, 0xff, 0xfe,
    0xf7, 0xff, 0x77, 0xef, 0xfd, 0xff, 0x3f, 0xff, 0xfe, 0xfd, 0xeb, 0xff, 0xdf, 0xbf, 0xfb, 0xdf, 0xfe, 0xff, 0xff, 0xf7,
    0xef, 0x7f, 0xff, 0x7d, 0xff, 0xbf, 0xfd, 0xfa, 0xff, 0xcf, 0xbf, 0xff, 0xff, 0xfa, 0xdf, 0xff, 0xf7, 0xef, 0xf7, 0xff,
    0x7f, 0xff, 0xfd, 0xfd, 0xbb, 0xff, 0xde, 0xbf, 0xef, 0xff, 0xfe, 0xff, 0xfb, 0xd7, 0xef, 0xff, 0xff, 0x77, 0xff, 0x7f,
    0xfd, 0xf9, 0xff, 0x9f, 0xbf, 0xfe, 0xff, 0xee, 0xff, 0xff, 0xf7, 0xeb, 0xdf, 0xff, 0x7f, 0xff, 0xf7, 0xfd, 0x7b, 0xff,
    0xdd, 0xbf, 0xbf, 0xff, 0xfe, 0xff, 0xef, 0xf7, 0xef, 0xff, 0xfb, 0x5f, 0xff, 0xff, 0xfd, 0xf3, 0xff, 0x5f, 0xbf, 0xfd,
    0xff, 0xbe, 0xff, 0xfe, 0xf7, 0xef, 0xff, 0xff, 0x7f, 0xfb, 0xdf, 0xfd, 0xfb, 0xff, 0xd7, 0xbf, 0x7f, 0xff, 0xfc, 0xff,
    0xbf, 0xf7, 0xee, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xf9, 0xdb, 0xff, 0xdf, 0xbf, 0xf7, 0xff, 0x7e, 0xff, 0xfd, 0xf7, 0xaf,
    0xff, 0xfe, 0x7f, 0xef, 0xff, 0xfd, 0xfb, 0xfb, 0xdf, 0xbf, 0xff, 0xff, 0xf6, 0xff, 0x7f, 0xf7, 0xed, 0xff, 0xbf, 0x7f,
    0xfe, 0xff, 0xed, 0xfb, 0xff, 0xdf, 0xbb, 0xdf, 0xff, 0xfe, 0xff, 0xf7, 0xf7, 0x6f, 0xff, 0xfd, 0x7f, 0xbf, 0xff, 0xfc,
    0xfb, 0xef, 0xdf, 0xbf, 0xff, 0xfb, 0xde, 0xff, 0xff, 0xf7, 0xe7, 0xff, 0x7f, 0x7f, 0xfd, 0xff, 0xbd, 0xfb, 0xfe, 0xdf,
    0xaf, 0xff, 0xff, 0xfe, 0xfb, 0xdf, 0xf7, 0xef, 0xff, 0xf7, 0x7f, 0x7f, 0xff, 0xfd, 0xfb, 0xbf, 0xdf, 0xbe, 0xff, 0xef,
    0xfe, 0xff, 0xff, 0xf3, 0xcf, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0x7d, 0xfb, 0xfd, 0xdf, 0xbf, 0xff, 0xfe, 0xfe, 0xef, 0xff,
    0xf7, 0xef, 0xfb, 0xdf, 0x7f, 0xff, 0xff, 0xf5, 0xfb, 0x7f, 0xdf, 0xbd, 0xff, 0xbf, 0xfe, 0xfe, 0xff, 0xe7, 0xef, 0xff,
    0xff, 0x7b, 0xdf, 0xff, 0xfd, 0xfb, 0xf7, 0xdf, 0x3f, 0xff, 0xfd, 0xfe, 0xbf, 0xff, 0xf6, 0xef, 0xef, 0xff, 0x7f, 0xff,
    0xfb, 0xdd, 0xfb, 0xff, 0xdf, 0xb7, 0xff, 0x7f, 0xfe, 0xfd, 0xff, 0xb7, 0xef, 0xfe, 0xff, 0x6f, 0xff, 0xff, 0xfd, 0xfb,
    0xdf, 0xdf, 0xbf, 0xff, 0xf7, 0xfe, 0x7f, 0xff, 0xf5, 0xef, 0xbf, 0xff, 0x7e, 0xff, 0xef, 0xfd, 0xfb, 0xff, 0xdb, 0x9f,
    0xff, 0xff, 0xfe, 0xf7, 0xff, 0x77, 0xef, 0xfd, 0xff, 0x3f, 0xff, 0xfe, 0xfd, 0xeb, 0xff, 0xdf, 0xbf, 0xfb, 0xdf, 0xfe,
    0xff, 0xff, 0xf7, 0xef, 0x7f, 0xff, 0x7d, 0xff, 0xbf, 0xfd, 0xfa, 0xff, 0xcf, 0xbf, 0xff, 0xff, 0xfa, 0xdf, 0xff, 0xf7,
    0xef, 0xf7, 0xff, 0x7f, 0xff, 0xfd, 0xfd, 0xbb, 0xff, 0xde, 0xbf, 0xef, 0xff, 0xfe, 0xff, 0xfb, 0xd7, 0xef, 0xff, 0xff,
    0x77, 0xff, 0x7f, 0xfd, 0xf9, 0xff, 0x9f, 0xbf, 0xfe, 0xff, 0xee, 0xff, 0xff, 0xf7, 0xeb, 0xdf, 0xff, 0x7f, 0xff, 0xf7,
    0xfd, 0x7b, 0xff, 0xdd, 0xbf, 0xbf, 0xff, 0xfe, 0xff, 0xef, 0xf7, 0xef, 0xff, 0xfb, 0x5f, 0xff, 0xff, 0xfd, 0xf3, 0xff,
    0x5f, 0xbf, 0xfd, 0xff, 0xbe, 0xff, 0xfe, 0xf7, 0xef, 0xff, 0xff, 0x7f, 0xfb, 0xdf, 0xfd, 0xfb, 0xff, 0xd7, 0xbf, 0x7f,
    0xff, 0xfc, 0xff, 0xbf, 0xf7, 0xee, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xf9, 0xdb, 0xff, 0xdf, 0xbf, 0xf7, 0xff, 0x7e, 0xff,
    0xfd, 0xf7, 0xaf, 0xff, 0xfe, 0x7f, 0xef, 0xff, 0xfd, 0xfb, 0xfb, 0xdf, 0xbf, 0xff, 0xff, 0xf6, 0xff, 0x7f, 0xf7, 0xed,
    0xff, 0xbf, 0x7f, 0xfe, 0xff, 0xed, 0xfb, 0xff, 0xdf, 0xbb, 0xdf, 0xff, 0xfe, 0xff, 0xf7, 0xf7, 0x6f, 0xff, 0xfd, 0x7f,
    0xbf, 0xff, 0xfc, 0xfb, 0xef, 0xdf, 0xbf, 0xff, 0xfb, 0xde, 0xff, 0xff, 0xf7, 0xe7, 0xff, 0x7f, 0x7f, 0xfd, 0xff, 0xbd,
    0xfb, 0xfe, 0xdf, 0xaf, 0xff, 0xff, 0xfe, 0xfb, 0xdf, 0xf7, 0xef, 0xff, 0xf7, 0x7f, 0x7f, 0xff, 0xfd, 0xfb, 0xbf, 0xdf,
    0xbe, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xf3, 0xcf, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0x7d, 0xfb, 0xfd, 0xdf, 0xbf, 0xff, 0xfe,
    0xfe, 0xef, 0xff, 0xf7, 0xef, 0xfb, 0xdf, 0x7f, 0xff, 0xff, 0xf5, 0xfb, 0x7f, 0xdf, 0xbd, 0xff, 0xbf, 0xfe, 0xfe, 0xff,
    0xe7, 0xef, 0xff, 0xff, 0x7b, 0xdf, 0xff, 0xfd, 0xfb, 0xf7, 0xdf, 0x3f, 0xff, 0xfd, 0xfe, 0xbf, 0xff, 0xf6, 0xef, 0xef,
    0xff, 0x7f, 0xff, 0xfb, 0xdd, 0xfb, 0xff, 0xdf, 0xb7, 0xff, 0x7f, 0xfe, 0xfd, 0xff, 0xb7, 0xef, 0xfe, 0xff, 0x6f, 0xff,
    0xff, 0xfd, 0xfb, 0xdf, 0xdf, 0xbf, 0xff, 0xf7, 0xfe, 0x7f, 0xff, 0xf5, 0xef, 0xbf, 0xff, 0x7e, 0xff, 0xef, 0xfd, 0xfb,
    0xff, 0xdb, 0x9f, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0x77, 0xef, 0xfd, 0xff, 0x3f, 0xff, 0xfe, 0xfd, 0xeb, 0xff, 0xdf, 0xbf,
    0xfb, 0xdf, 0xfe, 0xff, 0xff, 0xf7, 0xef, 0x7f, 0xff, 0x7d, 0xff, 0xbf, 0xfd, 0xfa, 0xff, 0xcf, 0xbf, 0xff, 0xff, 0xfa,
    0xdf, 0xff, 0xf7, 0xef, 0xf7, 0xff, 0x7f, 0xff, 0xfd, 0xfd, 0xbb, 0xff, 0xde, 0xbf, 0xef, 0xff, 0xfe, 0xff, 0xfb, 0xd7,
    0xef, 0xff, 0xff, 0x77, 0xff, 0x7f, 0xfd, 0xf9, 0xff, 0x9f, 0xbf, 0xfe, 0xff, 0xee, 0xff, 0xff, 0xf7, 0xeb, 0xdf, 0xff,
    0x7f, 0xff, 0xf7, 0xfd, 0x7b, 0xff, 0xdd, 0xbf, 0xbf, 0xff, 0xfe, 0xff, 0xef, 0xf7, 0xef, 0xff, 0xfb, 0x5f, 0xff, 0xff,
    0xfd, 0xf3, 0xff, 0x5f, 0xbf, 0xfd, 0xff, 0xbe, 0xff, 0xfe, 0xf7, 0xef, 0xff, 0xff, 0x7f, 0xfb, 0xdf, 0xfd, 0xfb, 0xff,
    0xd7, 0xbf, 0x7f, 0xff, 0xfc, 0xff, 0xbf, 0xf7, 0xee, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xf9, 0xdb, 0xff, 0xdf, 0xbf, 0xf7,
    0xff, 0x7e, 0xff, 0xfd, 0xf7, 0xaf, 0xff, 0xfe, 0x7f, 0xef, 0xff, 0xfd, 0xfb, 0xfb, 0xdf, 0xbf, 0xff, 0xff, 0xf6, 0xff,
    0x7f, 0xf7, 0xed, 0xff, 0xbf, 0x7f, 0xfe, 0xff, 0xed, 0xfb, 0xff, 0xdf, 0xbb, 0xdf, 0xff, 0xfe, 0xff, 0xf7, 0xf7, 0x6f,
    0xff, 0xfd, 0x7f, 0xbf, 0xff, 0xfc, 0xfb, 0xef, 0xdf, 0xbf, 0xff, 0xfb, 0xde, 0xff, 0xff, 0xf7, 0xe7, 0xff, 0x7f, 0x7f,
    0xfd, 0xff, 0xbd, 0xfb, 0xfe, 0xdf, 0xaf, 0xff, 0xff, 0xfe, 0xfb, 0xdf, 0xf7, 0xef, 0xff, 0xf7, 0x7f, 0x7f, 0xff, 0xfd,
    0xfb, 0xbf, 0xdf, 0xbe, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xf3 },
  { 0x3f, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xf7, 0x7f, 0xdf, 0xff, 0xef, 0xfe, 0xef, 0xff, 0xf7, 0xdf, 0xfb, 0xff,
    0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfa, 0x7f, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xbf, 0xfe, 0xef, 0xff, 0xdf,
    0xdf, 0xef, 0xff, 0xff, 0xff, 0xf5, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xfe, 0x7f, 0xfe,
    0xef, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xdd, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xf3, 0xff, 0xfb, 0xf7, 0x7f, 0xff,
    0xfd, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xde, 0x7f, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xef,
    0xf7, 0x7f, 0xff, 0xf7, 0xff, 0xfa, 0xef, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbe, 0x7f, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xbf, 0xf7, 0x7f, 0xff, 0xdf, 0xff, 0xee, 0xef, 0xff, 0xff, 0xd7, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xbd, 0xff,
    0xff, 0xff, 0xff, 0xfb, 0xfe, 0x7f, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xbe, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xff,
    0xfd, 0xb7, 0xff, 0xfb, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xfe, 0x7e, 0xef, 0xff, 0xff, 0xdf, 0xff,
    0xbf, 0xff, 0xff, 0xfd, 0x9f, 0xff, 0xef, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xf3, 0x7f, 0xff, 0xff, 0xfd, 0xfe, 0xef, 0xff,
    0xff, 0xdf, 0xfe, 0x7f, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xdf, 0xff, 0xe7, 0x7f, 0xff, 0xff, 0xf7,
    0xfe, 0xeb, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xfe, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xb7, 0x7f,
    0xff, 0xff, 0xdf, 0xfe, 0xef, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xfb, 0xff, 0xfd, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xfe, 0x77, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xaf, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xef, 0xff, 0xfd, 0xbf, 0xf7, 0xff, 0xfb,
    0xff, 0xfb, 0xff, 0xfd, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0x6f, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xbf,
    0xdf, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xf7, 0xf7, 0x7b, 0xff, 0xff, 0xff, 0xfc, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0x7f,
    0xff, 0xfd, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xdf, 0xf7, 0x6f, 0xff, 0xff, 0xff, 0xf6, 0xef, 0xfb, 0xff, 0xdf,
    0xff, 0xfd, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xfe, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0x3f, 0xff, 0xff, 0xff, 0xde, 0xef,
    0xef, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xfb, 0xfd, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf6, 0x7f, 0xff, 0xff,
    0xff, 0xfe, 0xef, 0xbf, 0xff, 0xdf, 0xff, 0xdf, 0xff, 0xef, 0xfd, 0xbf, 0xff, 0xf7, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xf5,
    0x7f, 0xff, 0xff, 0xff, 0xfe, 0xee, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xbf, 0xff, 0xdf, 0xff, 0xef, 0xfb,
    0xff, 0xff, 0xf7, 0x7f, 0xfb, 0xff, 0xff, 0xfe, 0xed, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0x7f, 0xfd, 0xbf, 0xff, 0xff,
    0xff, 0xbf, 0xfb, 0xff, 0xff, 0xd7, 0x7f, 0xef, 0xff, 0xff, 0xfe, 0xe7, 0xff, 0xfb, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xfd,
    0xbf, 0xff, 0xff, 0xfe, 0x7f, 0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xbf, 0xff, 0xff, 0xfe, 0xcf, 0xff, 0xef, 0xdf, 0xff, 0xff,
    0xf7, 0xff, 0xf9, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0x7e, 0x7f, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xbf,
    0xdf, 0xff, 0xff, 0xdf, 0xff, 0xed, 0xbf, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0x7d, 0xff, 0xff, 0xff, 0xfe,
    0xef, 0xfe, 0x7f, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xeb, 0xff, 0xff, 0xf7, 0x77, 0xff,
    0xfb, 0xff, 0xfe, 0xef, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0x7d, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff,
    0xf7, 0x5f, 0xff, 0xef, 0xff, 0xfe, 0xef, 0xf7, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xfe,
    0x7b, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xbf, 0xff, 0xfe, 0xef, 0xdf, 0xff, 0xcf, 0xff, 0xff, 0xff, 0xf7, 0xfd, 0xbb, 0xff,
    0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xfe, 0x7f, 0xff, 0xfe, 0xef, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xdf,
    0xfd, 0xaf, 0xff, 0xff, 0xff, 0xf7, 0xfb, 0xfb, 0xff, 0xf7, 0x7f, 0xfd, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xfe, 0x5f, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xfb, 0xef, 0xff, 0xf7, 0x7f, 0xf7, 0xff, 0xfb, 0xfe, 0xef, 0xff,
    0xfd, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xf7, 0x7f, 0xdf, 0xff, 0xef,
    0xfe, 0xef, 0xff, 0xf7, 0xdf, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfa, 0x7f, 0xff, 0xf7, 0x7f,
    0xff, 0xff, 0xbf, 0xfe, 0xef, 0xff, 0xdf, 0xdf, 0xef, 0xff, 0xff, 0xff, 0xf5, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xf9, 0xff,
    0xff, 0xf7, 0x7f, 0xff, 0xfe, 0x7f, 0xfe, 0xef, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xdd, 0xbf, 0xef, 0xff, 0xff,
    0xff, 0xf3, 0xff, 0xfb, 0xf7, 0x7f, 0xff, 0xfd, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xde, 0x7f, 0xff, 0xff, 0xff, 0xfd, 0xbf,
    0xbf, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xef, 0xf7, 0x7f, 0xff, 0xf7, 0xff, 0xfa, 0xef, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xbe, 0x7f, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xbf, 0xf7, 0x7f, 0xff, 0xdf, 0xff, 0xee, 0xef, 0xff, 0xff, 0xd7,
    0xff, 0xfb, 0xff, 0xff, 0xfd, 0xbd, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfe, 0x7f, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xbe, 0xef,
    0xff, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xff, 0xfd, 0xb7, 0xff, 0xfb, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xf7, 0x7f, 0xff, 0xff,
    0xfe, 0x7e, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0x9f, 0xff, 0xef, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xf3,
    0x7f, 0xff, 0xff, 0xfd, 0xfe, 0xef, 0xff, 0xff, 0xdf, 0xfe, 0x7f, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xfb,
    0xdf, 0xff, 0xe7, 0x7f, 0xff, 0xff, 0xf7, 0xfe, 0xeb, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xfe, 0x7f,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xb7, 0x7f, 0xff, 0xff, 0xdf, 0xfe, 0xef, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xfb, 0xff, 0xfd,
    0xbf, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0x77, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xaf, 0xff, 0xff, 0xdf, 0xdf, 0xff,
    0xef, 0xff, 0xfd, 0xbf, 0xf7, 0xff, 0xfb, 0xff, 0xfb, 0xff, 0xfd, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0x6f, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xbf, 0xdf, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xf7, 0xf7, 0x7b, 0xff, 0xff, 0xff, 0xfc,
    0xef, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0x7f, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xdf, 0xf7, 0x6f, 0xff,
    0xff, 0xff, 0xf6, 0xef, 0xfb, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xfe, 0x7f, 0xff, 0xfb, 0xff, 0xff,
    0xf7, 0x3f, 0xff, 0xff, 0xff, 0xde, 0xef, 0xef, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xfb, 0xfd, 0xbf, 0xff, 0xfd, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xf6, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xbf, 0xff, 0xdf, 0xff, 0xdf, 0xff, 0xef, 0xfd, 0xbf, 0xff,
    0xf7, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xf5, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xee, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf,
    0xfd, 0xbf, 0xff, 0xdf, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xfb, 0xff, 0xff, 0xfe, 0xed, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xfe, 0x7f, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xd7, 0x7f, 0xef, 0xff, 0xff, 0xfe, 0xe7, 0xff,
    0xfb, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xfe, 0x7f, 0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xbf, 0xff, 0xff,
    0xfe, 0xcf, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xf7, 0xff, 0xf9, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0x7e,
    0x7f, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xed, 0xbf, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0xff,
    0xff, 0xf7, 0x7d, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xfe, 0x7f, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xbf, 0xff, 0xff, 0xdf,
    0xff, 0xeb, 0xff, 0xff, 0xf7, 0x77, 0xff, 0xfb, 0xff, 0xfe, 0xef, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0x7d, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xf7, 0x5f, 0xff, 0xef, 0xff, 0xfe, 0xef, 0xf7, 0xff, 0xdb, 0xff, 0xff, 0xff,
    0xfd, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0x7b, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xbf, 0xff, 0xfe, 0xef, 0xdf, 0xff, 0xcf,
    0xff, 0xff, 0xff, 0xf7, 0xfd, 0xbb, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xfe, 0x7f, 0xff, 0xfe, 0xef,
    0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xaf, 0xff, 0xff, 0xff, 0xf7, 0xfb, 0xfb, 0xff, 0xf7, 0x7f, 0xfd, 0xff,
    0xff, 0xfe, 0xef, 0xff, 0xfe, 0x5f, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xfb, 0xef, 0xff, 0xf7,
    0x7f, 0xf7, 0xff, 0xfb, 0xfe, 0xef, 0xff, 0xfd, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xfb,
    0xbf, 0xff, 0xf7, 0x7f, 0xdf, 0xff, 0xef, 0xfe, 0xef, 0xff, 0xf7, 0xdf, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xfa, 0x7f, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xbf, 0xfe, 0xef, 0xff, 0xdf, 0xdf, 0xef, 0xff, 0xff, 0xff, 0xf5,
    0xbf, 0xfb, 0xff, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xfe, 0x7f, 0xfe, 0xef, 0xff, 0xff, 0xdf, 0xbf, 0xff,
    0xff, 0xff, 0xdd, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xf3, 0xff, 0xfb, 0xf7, 0x7f, 0xff, 0xfd, 0xff, 0xfe, 0xef, 0xff, 0xff,
    0xde, 0x7f, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xef, 0xf7, 0x7f, 0xff, 0xf7, 0xff, 0xfa,
    0xef, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbe, 0x7f, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xbf, 0xf7, 0x7f, 0xff,
    0xdf, 0xff, 0xee, 0xef, 0xff, 0xff, 0xd7, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xbd, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfe, 0x7f,
    0xf7, 0x7f, 0xff, 0xff, 0xff, 0xbe, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xff, 0xfd, 0xb7, 0xff, 0xfb, 0xff, 0xff,
    0xfb, 0xfd, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xfe, 0x7e, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0x9f, 0xff,
    0xef, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xf3, 0x7f, 0xff, 0xff, 0xfd, 0xfe, 0xef, 0xff, 0xff, 0xdf, 0xfe, 0x7f, 0xff, 0xff,
    0xfd, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xdf, 0xff, 0xe7, 0x7f, 0xff, 0xff, 0xf7, 0xfe, 0xeb, 0xff, 0xff, 0xdf, 0xfd,
    0xff, 0xff, 0xff, 0xfd, 0xbf, 0xfe, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xb7, 0x7f, 0xff, 0xff, 0xdf, 0xfe, 0xef, 0xff,
    0xff, 0xdf, 0xf7, 0xff, 0xfb, 0xff, 0xfd, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0x77, 0x7f, 0xff, 0xff, 0xff,
    0xfe, 0xaf, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xef, 0xff, 0xfd, 0xbf, 0xf7, 0xff, 0xfb, 0xff, 0xfb, 0xff, 0xfd, 0xf7, 0x7f,
    0xff, 0xff, 0xff, 0xfe, 0x6f, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xbf, 0xdf, 0xff, 0xef, 0xff, 0xfb, 0xff,
    0xf7, 0xf7, 0x7b, 0xff, 0xff, 0xff, 0xfc, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0x7f, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xbf,
    0xff, 0xfb, 0xff, 0xdf, 0xf7, 0x6f, 0xff, 0xff, 0xff, 0xf6, 0xef, 0xfb, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xbf,
    0xff, 0xfe, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0x3f, 0xff, 0xff, 0xff, 0xde, 0xef, 0xef, 0xff, 0xdf, 0xff, 0xf7, 0xff,
    0xfb, 0xfd, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf6, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xbf, 0xff, 0xdf,
    0xff, 0xdf, 0xff, 0xef, 0xfd, 0xbf, 0xff, 0xf7, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xf5, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xee,
    0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xbf, 0xff, 0xdf, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xfb, 0xff,
    0xff, 0xfe, 0xed, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0x7f, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xd7,
    0x7f, 0xef, 0xff, 0xff, 0xfe, 0xe7, 0xff, 0xfb, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xfe, 0x7f, 0xfb,
    0xff, 0xff, 0xf7, 0x7f, 0xbf, 0xff, 0xff, 0xfe, 0xcf, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xf7, 0xff, 0xf9, 0xbf, 0xff, 0xff,
    0xfd, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0x7e, 0x7f, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xed,
    0xbf, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0x7d, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xfe, 0x7f, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0xbd, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xeb, 0xff, 0xff, 0xf7, 0x77, 0xff, 0xfb, 0xff, 0xfe, 0xef, 0xfd, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xfe, 0x7d, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xf7, 0x5f, 0xff, 0xef, 0xff, 0xfe,
    0xef, 0xf7, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0x7b, 0xff, 0xff, 0xf7, 0x7f, 0xff,
    0xbf, 0xff, 0xfe, 0xef, 0xdf, 0xff, 0xcf, 0xff, 0xff, 0xff, 0xf7, 0xfd, 0xbb, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff,
    0xf7, 0x7f, 0xfe, 0x7f, 0xff, 0xfe, 0xef, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xaf, 0xff, 0xff, 0xff, 0xf7,
    0xfb, 0xfb, 0xff, 0xf7, 0x7f, 0xfd, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xfe, 0x5f, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff,
    0xff, 0xff, 0xdf, 0xfb, 0xef, 0xff, 0xf7, 0x7f, 0xf7, 0xff, 0xfb, 0xfe, 0xef, 0xff, 0xfd, 0xdf, 0xff, 0xff, 0xff, 0xff,
    0xfc, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xf7, 0x7f, 0xdf, 0xff, 0xef, 0xfe, 0xef, 0xff, 0xf7, 0xdf, 0xfb,
    0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfa, 0x7f, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xbf, 0xfe, 0xef, 0xff,
    0xdf, 0xdf, 0xef, 0xff, 0xff, 0xff, 0xf5, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xfe, 0x7f,
    0xfe, 0xef, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xdd, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xf3, 0xff, 0xfb, 0xf7, 0x7f,
    0xff, 0xfd, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xde, 0x7f, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xdb, 0xff,
    0xef, 0xf7, 0x7f, 0xff, 0xf7, 0xff, 0xfa, 0xef, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbe, 0x7f, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xbf, 0xf7, 0x7f, 0xff, 0xdf, 0xff, 0xee, 0xef, 0xff, 0xff, 0xd7, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xbd,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xfe, 0x7f, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xbe, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xef, 0xff,
    0xff, 0xfd, 0xb7, 0xff, 0xfb, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xfe, 0x7e, 0xef, 0xff, 0xff, 0xdf,
    0xff, 0xbf, 0xff, 0xff, 0xfd, 0x9f, 0xff, 0xef, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xf3, 0x7f, 0xff, 0xff, 0xfd, 0xfe, 0xef,
    0xff, 0xff, 0xdf, 0xfe, 0x7f, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xdf, 0xff, 0xe7, 0x7f, 0xff, 0xff,
    0xf7, 0xfe, 0xeb, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xfe, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xb7,
    0x7f, 0xff, 0xff, 0xdf, 0xfe, 0xef, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xfb, 0xff, 0xfd, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xfb,
    0xff, 0xfe, 0x77, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xaf, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xef, 0xff, 0xfd, 0xbf, 0xf7, 0xff,
    0xfb, 0xff, 0xfb, 0xff, 0xfd, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0x6f, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfd,
    0xbf, 0xdf, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xf7, 0xf7, 0x7b, 0xff, 0xff, 0xff, 0xfc, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xfe,
    0x7f, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xdf, 0xf7, 0x6f, 0xff, 0xff, 0xff, 0xf6, 0xef, 0xfb, 0xff,
    0xdf, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xfe, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0x3f, 0xff, 0xff, 0xff, 0xde,
    0xef, 0xef, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xfb, 0xfd, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf6, 0x7f, 0xff,
    0xff, 0xff, 0xfe, 0xef, 0xbf, 0xff, 0xdf, 0xff, 0xdf, 0xff, 0xef, 0xfd, 0xbf, 0xff, 0xf7, 0xff, 0xfb, 0xfb, 0xff, 0xff,
    0xf5, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xee, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xbf, 0xff, 0xdf, 0xff, 0xef,
    0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xfb, 0xff, 0xff, 0xfe, 0xed, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0x7f, 0xfd, 0xbf, 0xff,
    0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xd7, 0x7f, 0xef, 0xff, 0xff, 0xfe, 0xe7, 0xff, 0xfb, 0xdf, 0xff, 0xff, 0xfd, 0xff,
    0xfd, 0xbf, 0xff, 0xff, 0xfe, 0x7f, 0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xbf, 0xff, 0xff, 0xfe, 0xcf, 0xff, 0xef, 0xdf, 0xff,
    0xff, 0xf7, 0xff, 0xf9, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0x7e, 0x7f, 0xff, 0xff, 0xfe, 0xef, 0xff,
    0xbf, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xed, 0xbf, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0x7d, 0xff, 0xff, 0xff,
    0xfe, 0xef, 0xfe, 0x7f, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xeb, 0xff, 0xff, 0xf7, 0x77,
    0xff, 0xfb, 0xff, 0xfe, 0xef, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0x7d, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xbb, 0xff,
    0xff, 0xf7, 0x5f, 0xff, 0xef, 0xff, 0xfe, 0xef, 0xf7, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xbf, 0xff, 0xff, 0xff,
    0xfe, 0x7b, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xbf, 0xff, 0xfe, 0xef, 0xdf, 0xff, 0xcf, 0xff, 0xff, 0xff, 0xf7, 0xfd, 0xbb,
    0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xfe, 0x7f, 0xff, 0xfe, 0xef, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff,
    0xdf, 0xfd, 0xaf, 0xff, 0xff, 0xff, 0xf7, 0xfb, 0xfb, 0xff, 0xf7, 0x7f, 0xfd, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xfe, 0x5f,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xfb, 0xef, 0xff, 0xf7, 0x7f, 0xf7, 0xff, 0xfb, 0xfe, 0xef,
    0xff, 0xfd, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xf7, 0x7f, 0xdf, 0xff,
    0xef, 0xfe, 0xef, 0xff, 0xf7, 0xdf, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfa, 0x7f, 0xff, 0xf7,
    0x7f, 0xff, 0xff, 0xbf, 0xfe, 0xef, 0xff, 0xdf, 0xdf, 0xef, 0xff, 0xff, 0xff, 0xf5, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xf9,
    0xff, 0xff, 0xf7, 0x7f, 0xff, 0xfe, 0x7f, 0xfe, 0xef, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xdd, 0xbf, 0xef, 0xff,
    0xff, 0xff, 0xf3, 0xff, 0xfb, 0xf7, 0x7f, 0xff, 0xfd, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xde, 0x7f, 0xff, 0xff, 0xff, 0xfd,
    0xbf, 0xbf, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xef, 0xf7, 0x7f, 0xff, 0xf7, 0xff, 0xfa, 0xef, 0xff, 0xff, 0xdd, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xbe, 0x7f, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xbf, 0xf7, 0x7f, 0xff, 0xdf, 0xff, 0xee, 0xef, 0xff, 0xff,
    0xd7, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xbd, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfe, 0x7f, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xbe,
    0xef, 0xff, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xff, 0xfd, 0xb7, 0xff, 0xfb, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xf7, 0x7f, 0xff,
    0xff, 0xfe, 0x7e, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0x9f, 0xff, 0xef, 0xff, 0xff, 0xfb, 0xf7, 0xff,
    0xf3, 0x7f, 0xff, 0xff, 0xfd, 0xfe, 0xef, 0xff, 0xff, 0xdf, 0xfe, 0x7f, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xbf, 0xff, 0xff,
    0xfb, 0xdf, 0xff, 0xe7, 0x7f, 0xff, 0xff, 0xf7, 0xfe, 0xeb, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xfe,
    0x7f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xb7, 0x7f, 0xff, 0xff, 0xdf, 0xfe, 0xef, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xfb, 0xff,
    0xfd, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0x77, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xaf, 0xff, 0xff, 0xdf, 0xdf,
    0xff, 0xef, 0xff, 0xfd, 0xbf, 0xf7, 0xff, 0xfb, 0xff, 0xfb, 0xff, 0xfd, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0x6f, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xbf, 0xdf, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xf7, 0xf7, 0x7b, 0xff, 0xff, 0xff,
    0xfc, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0x7f, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xdf, 0xf7, 0x6f,
    0xff, 0xff, 0xff, 0xf6, 0xef, 0xfb, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xfe, 0x7f, 0xff, 0xfb, 0xff,
    0xff, 0xf7, 0x3f, 0xff, 0xff, 0xff, 0xde, 0xef, 0xef, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xfb, 0xfd, 0xbf, 0xff, 0xfd, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xf6, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xbf, 0xff, 0xdf, 0xff, 0xdf, 0xff, 0xef, 0xfd, 0xbf,
    0xff, 0xf7, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xf5, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xee, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xbf, 0xfd, 0xbf, 0xff, 0xdf, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xfb, 0xff, 0xff, 0xfe, 0xed, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xfe, 0x7f, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xd7, 0x7f, 0xef, 0xff, 0xff, 0xfe, 0xe7,
    0xff, 0xfb, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xfe, 0x7f, 0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xbf, 0xff,
    0xff, 0xfe, 0xcf, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xf7, 0xff, 0xf9, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0xf7,
    0x7e, 0x7f, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xed, 0xbf, 0xff, 0xff, 0xf7, 0xff, 0xfb,
    0xff, 0xff, 0xf7, 0x7d, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xfe, 0x7f, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xbf, 0xff, 0xff,
    0xdf, 0xff, 0xeb, 0xff, 0xff, 0xf7, 0x77, 0xff, 0xfb, 0xff, 0xfe, 0xef, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0x7d,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xf7, 0x5f, 0xff, 0xef, 0xff, 0xfe, 0xef, 0xf7, 0xff, 0xdb, 0xff, 0xff,
    0xff, 0xfd, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0x7b, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xbf, 0xff, 0xfe, 0xef, 0xdf, 0xff,
    0xcf, 0xff, 0xff, 0xff, 0xf7, 0xfd, 0xbb, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xf7, 0x7f, 0xfe, 0x7f, 0xff, 0xfe,
    0xef, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xaf, 0xff, 0xff, 0xff, 0xf7, 0xfb, 0xfb, 0xff, 0xf7, 0x7f, 0xfd,
    0xff, 0xff, 0xfe, 0xef, 0xff, 0xfe, 0x5f, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xfb, 0xef, 0xff,
    0xf7, 0x7f, 0xf7, 0xff, 0xfb, 0xfe, 0xef, 0xff, 0xfd, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfc },
  { 0xff, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xe7, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xdf,
    0x7f, 0xff, 0xff, 0xb7, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xf7, 0xdf, 0xff, 0xff, 0xef, 0xff, 0xcf, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0x77, 0xff, 0xff, 0xfe, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xdf, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xfe, 0x7f, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xfd, 0xff, 0xfb, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0x7f, 0xff, 0xef, 0xfe, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xf7, 0xfb, 0xff, 0xf7, 0xff, 0xfb,
    0xff, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xde, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xfd, 0xff, 0xfe, 0xff, 0xfb, 0xff, 0x77, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xdf, 0xff, 0xfb, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xbf, 0x7f, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xfb, 0xef, 0xf7,
    0xff, 0xfe, 0xff, 0xef, 0xff, 0x5f, 0xff, 0xff, 0xf7, 0xbf, 0xff, 0xff, 0xff, 0xfb, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff,
    0xbf, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xef, 0xef, 0xdf, 0xff, 0xfe, 0xff,
    0xbf, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xfe, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0xfb,
    0xfd, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xfe, 0xff, 0x7f, 0xff, 0x7f,
    0xfe, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xef, 0xfd, 0xdf, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xff, 0x7f, 0xef, 0xff, 0xfe, 0xfe, 0xff, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0xf7,
    0xff, 0xfb, 0xff, 0xf7, 0xfb, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xdf, 0xfe, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0xfb, 0x7f, 0xf7, 0xff, 0xf7, 0xff, 0xef, 0xff,
    0xdf, 0xfb, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0x7f, 0xfd, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfd,
    0xff, 0xff, 0xff, 0xeb, 0xff, 0xf7, 0xfe, 0xff, 0xff, 0xef, 0x7f, 0xdf, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xdf, 0xf7, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0x7f, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xbf, 0xff, 0xff, 0xf9, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xdf, 0xdf, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xff,
    0xfe, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xfe, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xbf,
    0xff, 0xff, 0xed, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xfd, 0xf7, 0xff, 0xff, 0xfb, 0xff, 0xf3, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xbd,
    0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xdf, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xff, 0x7b, 0xff,
    0xf7, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xff, 0xef, 0xfb, 0xff, 0xf6, 0xff, 0xff, 0xff, 0x6f, 0xff, 0xdf, 0xf7, 0xff,
    0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xbe, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xdb, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xef, 0xff, 0xde, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff,
    0xfb, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xcf, 0xff, 0xdf,
    0xff, 0xff, 0xef, 0xbf, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xf6, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xb7, 0xff, 0xff, 0xfd, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xff, 0xef,
    0x7f, 0xff, 0xfe, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xfb, 0xfb, 0xf7, 0xff, 0xff, 0xff, 0xef,
    0xff, 0x9f, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xfe, 0xff, 0xef, 0xff, 0xff, 0xfe,
    0xfd, 0xff, 0xff, 0x7f, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xfb, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xbf, 0xff,
    0xff, 0xfd, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xef, 0xff, 0xfb, 0xfe, 0xf7, 0xff, 0xff,
    0x7f, 0xef, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xbf, 0xfe, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xdf, 0xfb, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xef, 0xfe, 0xdf, 0xff, 0xff, 0x7f, 0xbf, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0x7f, 0xfb, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xf7,
    0xff, 0xff, 0xdf, 0xef, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xf7, 0xfe, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xbf, 0xf7, 0xff, 0xfd, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xdf,
    0xbf, 0xff, 0xff, 0xff, 0xef, 0xff, 0x7f, 0xfe, 0xff, 0xfe, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0xfd, 0xff, 0xff, 0xff, 0xfb,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xbf, 0xdf, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x7f, 0xff, 0xff,
    0xfe, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xfb, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfd, 0xef, 0xff,
    0xff, 0xfa, 0xff, 0xf7, 0xff, 0x7f, 0xff, 0xef, 0xf7, 0xdf, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xbf, 0xff, 0xfe, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0xfb, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xee, 0xff,
    0xdf, 0xff, 0x7f, 0xff, 0xbf, 0xf7, 0xff, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfd,
    0xfd, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xbe, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0x7f, 0xf7, 0xff, 0xfe, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xf7, 0xfd, 0xff, 0xff,
    0xef, 0xff, 0xdf, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xff, 0xef, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xfe, 0x7f, 0xff, 0xff, 0xf7,
    0xff, 0xfd, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0x7f, 0xff, 0xff, 0xee, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfd, 0x7f, 0xff, 0xff, 0xf3, 0xff, 0xf7, 0xff,
    0xff, 0xfb, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xdf, 0xff,
    0xff, 0xff, 0xff, 0xed, 0xff, 0xff, 0xfe, 0xfb, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xe7, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xdf, 0xff, 0xff, 0xfb, 0xff,
    0xe7, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xdf, 0x7f, 0xff, 0xff, 0xb7, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xdf, 0xff, 0xff, 0xef, 0xff, 0xcf, 0xff, 0xff,
    0xfe, 0xbf, 0xff, 0xff, 0x7f, 0xff, 0xff, 0x77, 0xff, 0xff, 0xfe, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf,
    0xfb, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xfe, 0x7f, 0xff,
    0xff, 0x7e, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xdd,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0x7f, 0xff, 0xef, 0xfe, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x7d, 0xff,
    0xff, 0xf7, 0xfb, 0xff, 0xf7, 0xff, 0xfb, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0x7f, 0xff, 0xff, 0xde, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfd, 0xff, 0xfe, 0xff, 0xfb, 0xff, 0x77, 0xff, 0xff, 0xf7, 0xef,
    0xff, 0xdf, 0xff, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x7f, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdd, 0xff, 0xff, 0xff, 0xfb, 0xef, 0xf7, 0xff, 0xfe, 0xff, 0xef, 0xff, 0x5f, 0xff, 0xff, 0xf7, 0xbf, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xd7, 0xff, 0xff,
    0xff, 0xef, 0xef, 0xdf, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xfe, 0xfb, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0xfb, 0xfd, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xef,
    0xff, 0xff, 0xfe, 0xff, 0x7f, 0xff, 0x7f, 0xfe, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xfb, 0xff, 0xf7,
    0xff, 0xbf, 0xff, 0xef, 0xfd, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xff, 0x7f, 0xef, 0xff, 0xfe, 0xfe,
    0xff, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0xf7, 0xff, 0xfb, 0xff, 0xf7, 0xfb, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xbf, 0xff,
    0xbf, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0xfb,
    0x7f, 0xf7, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xdf, 0xfb, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0x7f, 0xfd, 0xff,
    0xfe, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xf7, 0xfe, 0xff, 0xff, 0xef, 0x7f, 0xdf, 0xff,
    0xf7, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xfb, 0xdf, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0x7f,
    0xff, 0xff, 0xfa, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xf9, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xdf,
    0xdf, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xfe, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xf9,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xbf, 0xff, 0xff, 0xed, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0x6f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfd, 0xf7, 0xff, 0xff, 0xfb, 0xff, 0xf3, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xdf, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xfc, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xff, 0xef, 0xfb, 0xff, 0xf6, 0xff,
    0xff, 0xff, 0x6f, 0xff, 0xdf, 0xf7, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xbe, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xef, 0xff, 0xde, 0xff, 0xff, 0xff, 0x3f,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xfb, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xfd, 0xfb, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xcf, 0xff, 0xdf, 0xff, 0xff, 0xef, 0xbf, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xf6,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xb7, 0xff, 0xff, 0xfd, 0xef, 0xff, 0xdf, 0xff, 0xff,
    0xff, 0x9f, 0xff, 0xff, 0xff, 0xff, 0xef, 0x7f, 0xff, 0xfe, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff,
    0xfb, 0xfb, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0x9f, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5f, 0xff,
    0xff, 0xfe, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xfd, 0xff, 0xff, 0x7f, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xfb, 0xdf,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfd, 0xff,
    0xef, 0xff, 0xfb, 0xfe, 0xf7, 0xff, 0xff, 0x7f, 0xef, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xff,
    0x7f, 0xff, 0xbf, 0xfe, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xdf, 0xfb, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xef,
    0xfe, 0xdf, 0xff, 0xff, 0x7f, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0x7f, 0xfb, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf,
    0xfd, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xbf, 0xfe, 0xff, 0xff,
    0xff, 0x7f, 0x7f, 0xff, 0xf7, 0xfe, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xbf, 0xf7, 0xff, 0xfd,
    0xff, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xef, 0xff, 0x7f, 0xfe, 0xff, 0xfe, 0xff, 0x7f, 0xff,
    0xff, 0xf7, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xbf, 0xdf, 0xff, 0xfd, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0x7f, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xfb, 0xf7, 0xf7,
    0xff, 0xff, 0xff, 0xeb, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xfd, 0xef, 0xff, 0xff, 0xfa, 0xff, 0xf7, 0xff, 0x7f, 0xff, 0xef, 0xf7, 0xdf, 0xff, 0xff, 0xff,
    0xbb, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xbf, 0xff, 0xfe, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0xfb,
    0xff, 0xf7, 0xef, 0xff, 0xff, 0xee, 0xff, 0xdf, 0xff, 0x7f, 0xff, 0xbf, 0xf7, 0xff, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xff,
    0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xdf, 0xef,
    0xff, 0xff, 0xbe, 0xff, 0xff, 0xff, 0x7f, 0xff, 0x7f, 0xf7, 0xff, 0xfe, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xbb, 0xff, 0xf7, 0xfd, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xff, 0xef, 0xff, 0xff, 0x7e,
    0xff, 0xff, 0xfe, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xaf, 0xff,
    0xdf, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0x7f, 0xff, 0xff, 0xee, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfd,
    0x7f, 0xff, 0xff, 0xf3, 0xff, 0xf7, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xed, 0xff, 0xff, 0xfe, 0xfb, 0xff, 0xf7, 0x7f, 0xff, 0xff,
    0xe7, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xe7, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xdf, 0x7f, 0xff, 0xff, 0xb7, 0xff, 0xff,
    0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xdf,
    0xff, 0xff, 0xef, 0xff, 0xcf, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0xff, 0x7f, 0xff, 0xff, 0x77, 0xff, 0xff, 0xfe, 0xff, 0xfb,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xbf,
    0xff, 0xef, 0xff, 0xff, 0xfe, 0x7f, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xfb, 0xff,
    0xf7, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0x7f, 0xff, 0xef, 0xfe,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xf7, 0xfb, 0xff, 0xf7, 0xff, 0xfb, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xff,
    0xbf, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xde, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfd, 0xff, 0xfe, 0xff,
    0xfb, 0xff, 0x77, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xdf, 0xff, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x7f, 0xff,
    0xfd, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xfb, 0xef, 0xf7, 0xff, 0xfe, 0xff, 0xef, 0xff, 0x5f,
    0xff, 0xff, 0xf7, 0xbf, 0xff, 0xff, 0xff, 0xfb, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xfd, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xef, 0xef, 0xdf, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xf7,
    0x7f, 0xff, 0xff, 0xfe, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0xfb, 0xfd, 0xf7, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xfe, 0xff, 0x7f, 0xff, 0x7f, 0xfe, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xfd, 0xfb, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xef, 0xfd, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0x7f, 0xef, 0xff, 0xfe, 0xfe, 0xff, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0xf7, 0xff, 0xfb, 0xff, 0xf7, 0xfb, 0xff,
    0xff, 0xef, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0xfb, 0x7f, 0xf7, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xdf, 0xfb, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0x7f, 0xfd, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xf7,
    0xfe, 0xff, 0xff, 0xef, 0x7f, 0xdf, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xbf,
    0xff, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xdf, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xfe, 0xff, 0xff,
    0xbf, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0x7f, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xf9,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xdf, 0xdf, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0x7f, 0xff,
    0xfe, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xbf, 0xff, 0xff, 0xed, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfd, 0xf7, 0xff,
    0xff, 0xfb, 0xff, 0xf3, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xdf, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xef, 0xff,
    0xdb, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfd,
    0xff, 0xff, 0xef, 0xfb, 0xff, 0xf6, 0xff, 0xff, 0xff, 0x6f, 0xff, 0xdf, 0xf7, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xbe, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xf7, 0xff, 0xff, 0xef,
    0xef, 0xff, 0xde, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xfb, 0xfe, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xbd, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xcf, 0xff, 0xdf, 0xff, 0xff, 0xef, 0xbf, 0xff, 0xfe,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xf6, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xb7, 0xff,
    0xff, 0xfd, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xff, 0xef, 0x7f, 0xff, 0xfe, 0xfe, 0xff, 0xff,
    0x7f, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xfb, 0xfb, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0x9f, 0xff, 0xff, 0xfd, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xfe, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xfd, 0xff, 0xff, 0x7f, 0xfb, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xef, 0xfb, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0x7f, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xef, 0xff, 0xfb, 0xfe, 0xf7, 0xff, 0xff, 0x7f, 0xef, 0xff, 0xd7, 0xff, 0xff,
    0xff, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xbf, 0xfe, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xdf,
    0xfb, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xef, 0xfe, 0xdf, 0xff, 0xff, 0x7f, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0x7f, 0xfb,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xdf,
    0xff, 0xef, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xf7, 0xfe, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0xfb, 0xbf, 0xf7, 0xff, 0xfd, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xef, 0xff,
    0x7f, 0xfe, 0xff, 0xfe, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef,
    0xbf, 0xdf, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x7f, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xfe, 0xff,
    0xfd, 0xff, 0x7f, 0xff, 0xfb, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xbf, 0xff, 0xff,
    0xfd, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfd, 0xef, 0xff, 0xff, 0xfa, 0xff, 0xf7, 0xff, 0x7f,
    0xff, 0xef, 0xf7, 0xdf, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xbf, 0xff, 0xfe, 0xfd, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0xfb, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xee, 0xff, 0xdf, 0xff, 0x7f, 0xff, 0xbf, 0xf7,
    0xff, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xf7,
    0xff, 0xdf, 0xff, 0xef, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xbe, 0xff, 0xff, 0xff, 0x7f, 0xff, 0x7f, 0xf7, 0xff, 0xfe, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xf7, 0xfd, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xdf, 0xff,
    0xbf, 0xff, 0xff, 0xef, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xfe, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xfb,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0x7f, 0xff, 0xff,
    0xee, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfd, 0x7f, 0xff, 0xff, 0xf3, 0xff, 0xf7, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xed, 0xff, 0xff,
    0xfe, 0xfb, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xe7, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3f,
    0xff },
  { 0xff, 0xf3, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbe, 0xff, 0xbf, 0xdf, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xfb, 0x7f, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xef, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfe, 0xff, 0xbf,
    0xff, 0xbf, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfe,
    0xbf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xdf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0x7f, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xff, 0x7f, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfb, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xdf, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xef, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xbf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfd, 0xff, 0x3f, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfb,
    0xff, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xef, 0xff, 0xff, 0xf7,
    0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0xff, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xfb, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xf6, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfe, 0x7f, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xf7, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xdf, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xef, 0xfe, 0xff, 0xb7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xb7,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfb, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xef, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0xf7, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x6f,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xf7, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xef, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf5, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xdf, 0xdf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xfd, 0xef, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfd, 0xff, 0x77, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xef, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7e, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xef, 0x7f, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xdf, 0xff, 0xbf, 0xfb, 0xff, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfe, 0xff, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7f,
    0xef, 0xff, 0xff, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xfe, 0xff, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0x7f, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xfb, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfd,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfa, 0xff,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xff,
    0x7d, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfb,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbb, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xdf, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xfb, 0xff,
    0xff, 0xfd, 0xfd, 0xff, 0x7f, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xd7, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xef, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xde, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xfd, 0xfd, 0x7f, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xdf,
    0xff, 0xf7, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xee, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfe, 0xff, 0x9f, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xf9, 0xff, 0x7f, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xef, 0xff, 0xfb, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfe, 0xbf, 0xbf,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7b, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xf7, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfe, 0xff,
    0xbf, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xbe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0x5f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf3, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xef, 0xf7, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xbf, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfb,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xdf, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xf7, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xdf, 0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xef,
    0xff, 0xff, 0xf7, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0x7f, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xd7,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xdf, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xff, 0xef, 0xff, 0xfe, 0xf7, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7e, 0xff, 0xff, 0xff, 0xff, 0xbf,
    0xf7, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xee, 0xff, 0xbf, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xff,
    0xff, 0xf7, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfb, 0xfb, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xaf, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xf7, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xef, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xfd, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xfd, 0xff, 0xef, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xf7, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xdf, 0xfb, 0xff, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xed, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xed, 0xff, 0x7f,
    0xf7, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xfb, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfe, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0x6f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xdf, 0xff, 0xff, 0xbb, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xde, 0xff, 0xff, 0xfb, 0xff,
    0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfc, 0xff, 0xbf, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xef, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0xfb,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfe, 0xff, 0xbd, 0xff, 0xff, 0xff, 0xff,
    0x7f, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff,
    0xfb, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfb, 0xbf, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0xfd, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xfb, 0xff,
    0xff, 0xfd, 0xff, 0xfd, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xdf,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xbf, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xdf, 0xf7, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfe, 0xdf, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0x7d, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xcf, 0xff, 0xff, 0xf3, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbe, 0xff,
    0xbf, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0x7f, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xef, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfe,
    0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xfe, 0xff, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xfe, 0xbf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xdf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xe7, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xff, 0x7f, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfb,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xdf, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xef, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfd, 0xff, 0x3f, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xef,
    0xff, 0xff, 0xf7, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0xff, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xdf,
    0xf7, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xfb, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xf6, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfe, 0x7f, 0xff, 0xff, 0xff, 0xbf,
    0xff, 0xf7, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xdf, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xb7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xfe, 0xff, 0xff,
    0xff, 0xff, 0xb7, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfb, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff,
    0xfe, 0xff, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xff, 0x6f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xfb, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xef, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf5, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xdf, 0xdf, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xfd, 0xef, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfd, 0xff,
    0x77, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xdb, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xef, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7e, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xef, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xdf, 0xff, 0xbf, 0xfb, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfe, 0xff, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0x7f, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xff, 0xfb,
    0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xfe, 0xff, 0xbf, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff,
    0xfb, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xfd, 0xbf, 0xff, 0xff, 0xff,
    0x7f, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xfa, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xbf, 0xfd, 0xff,
    0xff, 0xff, 0xff, 0x7d, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbb, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff,
    0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0xff, 0xbf,
    0xff, 0xfb, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0x7f, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xd7, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xef, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xde, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfd, 0xfd, 0x7f, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xdf, 0xff, 0xf7, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xee, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfe,
    0xff, 0x9f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xff, 0x7f, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xfb, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfe, 0xff, 0xff, 0xff,
    0xfe, 0xbf, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7b, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xf7, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xfe, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xbe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0x5f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf3,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xef, 0xf7,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xdf, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xdf,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xdf, 0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0x7f, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xdf, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfe, 0xf7, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7e, 0xff, 0xff,
    0xff, 0xff, 0xbf, 0xf7, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xee, 0xff, 0xbf, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff,
    0xfe, 0xff, 0xff, 0xff, 0xf7, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfb, 0xfb, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xff, 0x7f, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xaf, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfd, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xf7, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xfd, 0xff,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xfd, 0xff, 0xef, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfd,
    0xf7, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xdf, 0xfb, 0xff,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xed, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xed, 0xff, 0x7f, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xfb,
    0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfe, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0x6f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xdf, 0xff, 0xff,
    0xbb, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0xbf, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xde, 0xff,
    0xff, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfc, 0xff, 0xbf, 0xff, 0xff,
    0x7f, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xef, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xfe, 0xfb, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfe, 0xff, 0xbd, 0xff,
    0xff, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xfb, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfb, 0xbf,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0xfd, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0xff,
    0xbf, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xf7, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xfe,
    0xff, 0xbf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xdf, 0xf7, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xfe, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0x7d, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xcf, 0xff },
  { 0xff, 0xaf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xf7,
    0xfb, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xdf, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xdf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xdf, 0xff, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xe7,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff,
    0xbf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xf7, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xee, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xf7, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xfd, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xfd,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0x7f,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xfb, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0x9f, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xdf, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xfd, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xf3, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xfe,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xff, 0xff, 0x3f,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xdf, 0xef, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xff, 0xfb, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xf7, 0xef, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0x9f, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0x7f, 0xef,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0x7f, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xdf, 0xff, 0xfb, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xed, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfd, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0x7f, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xef, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xbf, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xf7, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0x7f, 0xff, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xde, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xde, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xef, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xef, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf,
    0xf7, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0xef, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xfb, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfb,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xfe, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xdf, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0x6f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xdf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xef, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xfb, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xfe, 0xff, 0xff, 0xef,
    0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xb7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfd,
    0xff, 0x7f, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xed, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xdf, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0xee, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xf7, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf,
    0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfe,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xdf, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xf7, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff,
    0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf3, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff,
    0xfe, 0xff, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xf7, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf5, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x5f, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xb7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xef, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfb, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xfe, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfe, 0xff, 0xef, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xdf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xf7, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xf7, 0xef, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfe, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfb, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xdf, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xfd, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xcf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbe, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xdf, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfe, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xd7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xbf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbb, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfe, 0xef, 0xff, 0xef,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff,
    0xbf, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0x77, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf,
    0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xe7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x7f, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0xef, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfe, 0xff, 0xfb, 0xef, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0x7f, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xdf, 0xef, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff,
    0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xf7, 0xfb, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff,
    0xbf, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xfb, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xdf, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xdf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0x7f,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xe7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xee, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb,
    0xff, 0xfd, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xf7, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xbf, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xeb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfe,
    0xfb, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfd, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xf3, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xfb, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xff, 0xff, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfe, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfd,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0x7f, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf7, 0xef, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0x7f, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0x7f,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xdf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbd, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xf7,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xed, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xde, 0xff, 0xff, 0xaf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xde, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xef, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xf7, 0xff,
    0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0xef, 0xef,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xfb, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0x7f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xdf, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xdf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xbf, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xef, 0xff, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xb7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfd, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf,
    0xfe, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xed, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0xff,
    0xfb, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xee, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0x3f, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xf7, 0xbf, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfe, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xfd, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xdf, 0xff, 0xbf, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xef, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xf3, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd,
    0xff, 0xef, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xf7, 0xbf, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf5, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xfe, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xef, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb,
    0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfe,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0x7f, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xb7,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xfb, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfe, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xfe, 0xff, 0xef, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xfe, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xf7, 0xef,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xfb, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfc, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xcf, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xbf,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xfb,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xdf, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xfe, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xbf, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xf7, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xbf, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfe, 0xef, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfb, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x77, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xfd, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xe7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff,
    0xff, 0xfb, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbe, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfe, 0xff, 0xfb, 0xef, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xdf, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xdf, 0xef, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xf5, 0xff },
  { 0xff, 0x7f, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xfb, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xfb, 0xdf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x7f, 0xff,
    0x7f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xdf, 0xff,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xdf, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xfd, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfd,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xff, 0xf7, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef,
    0xfd, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef,
    0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfb, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xf7, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xf7, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xf7, 0xfe, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xfe, 0xf7, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xfe,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb,
    0xfe, 0xff, 0xff, 0x77, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff, 0xff, 0xfb,
    0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xfb, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0x7f, 0xff, 0xfb, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0x7b, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0x7f, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff,
    0xdf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x7f, 0xff, 0xfd, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7d, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xb7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xff, 0xff,
    0xff, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xb7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xef, 0xff,
    0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xfe, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xef, 0xff, 0xf7, 0xff, 0xdf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xe7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xe7, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7e, 0xff, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfb, 0xff, 0xef, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff,
    0xf7, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xbf, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xbe, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xed, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf3, 0xff, 0xff,
    0xff, 0xf3, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xed, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xbe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xbf, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0xff,
    0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xff, 0xde, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xde, 0xff, 0xff,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xdf, 0xff, 0xfe, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfe, 0xff, 0xdf, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xdf,
    0xff, 0xff, 0xff, 0xee, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xee, 0xff, 0xff, 0x7f, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x7f,
    0xff, 0xff, 0xef, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xef, 0x7f, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff,
    0x7f, 0xef, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xef, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xef, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xfe, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xf7,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xbf, 0xf7,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xbf,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xef, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xbf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xfb, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xff, 0xf7, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xff, 0xff, 0xfb,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfb,
    0xff, 0xff, 0xdf, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xdf, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff,
    0xfe, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfb, 0xdf, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xff, 0xdf, 0xfb, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfe, 0xff, 0xff,
    0x7f, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xfb, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xfb, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x7f, 0xff, 0x7f,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xfb, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xdf, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xdf, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xfd, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfd, 0xef,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xff, 0xf7, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfd,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfb, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xef,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xfd, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xef, 0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xf7, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xf7, 0xfe, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xfe, 0xf7, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xfe, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfe,
    0xff, 0xff, 0x77, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff, 0xff, 0xfb, 0xfe,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfb,
    0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0x7f, 0xff, 0xfb, 0xff, 0xff,
    0xfe, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0x7b, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xef, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xdf,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xfd, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xb7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xff, 0xff, 0xff,
    0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xb7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x7d, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xef,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xef,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xfe, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xf7, 0xff, 0xdf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xbd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xe7, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xe7, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7e, 0xff, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfb, 0xff, 0xef, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xf7,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf,
    0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xbe, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xed, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf3, 0xff, 0xff, 0xff,
    0xf3, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xed, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xbe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xbf, 0xff, 0xfe, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0xff, 0xfb,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfb,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0xff, 0xff,
    0x7f, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xde, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xde, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xdf,
    0xff, 0xfe, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xfd, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfe, 0xff, 0xdf, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xdf, 0xff,
    0xff, 0xff, 0xee, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf9, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xee, 0xff, 0xff, 0x7f, 0xdf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x7f, 0xff,
    0xff, 0xef, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0x7f,
    0xef, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xef, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xfe, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xf7, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xbf, 0xf7, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xbf, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xef, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xbf,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfb,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfe,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xf7, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xdf, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xfe,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb,
    0xdf, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xff, 0xdf, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfe, 0xff },
  { 0xff, 0xff, 0xf9, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfe, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xfd, 0xff, 0xf7, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfe, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfe,
    0xff, 0xff, 0xff, 0xb7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xf7, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xef,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x3f,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xef, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfb, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xde, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0x7f, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xff, 0xff, 0xf3, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfe, 0xff, 0xff, 0xfd,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xdf, 0xf7,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xfb, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xef,
    0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xef, 0xfd, 0xff, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xde, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xbf, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xff,
    0x7f, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf3, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xef, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0x7f, 0xff, 0xbf,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xee, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xff, 0xf7, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfe, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x6f, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0x7d, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xdf, 0xff, 0xff, 0xfe, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xf7, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xe7, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xdf, 0xff, 0xff, 0xff, 0xf7, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xdf, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xf7, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xef, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfb, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xed, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xdf,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff, 0xff, 0xfe, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xaf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef,
    0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xee, 0xff, 0xff, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0x7f, 0xfd, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xfe, 0xfe, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0x7f, 0xff, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xb7, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xdf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xf7, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xef, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfb, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xef, 0xff,
    0xff, 0xff, 0xfb, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xfd, 0xff, 0xff, 0xfd, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbe,
    0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xdf, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xbf, 0xff, 0xef, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xef, 0xff, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x77, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xff, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfd, 0xff, 0xfe,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xbb, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0x7f, 0xff, 0x7f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xff,
    0xbe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfd, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff,
    0xfd, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0x3f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xef,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xff, 0xbf, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf9, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xf7, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xfe, 0xf7, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xfb, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff,
    0x7f, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xbf, 0xbf,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x7f, 0xff,
    0xff, 0xff, 0xfb, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xdf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xf7, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xff, 0xbf, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xed, 0xff, 0xff, 0xff, 0x7f, 0xfe, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfb, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xef, 0xff, 0xbf, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0x7f, 0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xff, 0xf9,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xfe, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfd, 0xff, 0xf7,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xfb, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfe, 0xff, 0xff, 0xff,
    0xb7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff,
    0xdf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xef,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfb, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xff, 0x7f, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xde, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0x7f, 0xfb, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0x7f, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xff, 0xff, 0xf3, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfe, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xf7, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xaf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xef, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfb,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xef, 0xfd, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x9f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef,
    0xfd, 0xff, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xde, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xef, 0xff, 0xff, 0xbf, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xff, 0x7f, 0xf7, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf3, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xbf, 0xdf, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x7f, 0xff, 0xff, 0xff, 0xfd, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xef, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0x7f, 0xff, 0xbf, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xdb,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xee, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xfe, 0xff, 0xff, 0xf7, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xf7, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xdf, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xfb, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xef, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xdf, 0xff,
    0xff, 0xfe, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0x7f, 0xff, 0xfb, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xef,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xdf, 0xff, 0xff, 0xff, 0xf7, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xfe, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xdf, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xf7, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xef, 0xff,
    0x7f, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xfb, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xed, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff, 0xff, 0xfe, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff,
    0xff, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xaf,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xbf,
    0xff, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xee, 0xff, 0xff, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f,
    0xfd, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfe, 0xfe, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xdf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xef, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0x7f, 0xff, 0xff, 0xbf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xff, 0xff, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xb7, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xdf, 0xff, 0xff, 0xff, 0xdf, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xfe, 0xff, 0xf7, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xf7, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf,
    0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xeb, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x7f, 0xef, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfb, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xbf, 0xff, 0x7f, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xef, 0xff, 0xff, 0xff, 0xfb,
    0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfb, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfd, 0xff, 0xff,
    0xfd, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xef, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xfb, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0x7f, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbe, 0xff, 0xff, 0xff,
    0xbf, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x5f, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xef, 0xf7, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xdf, 0xff,
    0xff, 0xf7, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xdd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xfb, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xef,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xef, 0xff, 0xff, 0x7f, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xdf, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0x7f, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xf7, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff,
    0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xef, 0xff,
    0xff, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xf7,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xbb, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfe, 0xff, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x7f, 0xff,
    0x7f, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xdf, 0xbf, 0xff, 0xff, 0xff, 0xfb, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xfe, 0xff, 0xff, 0xff, 0xbe, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xfd, 0xff, 0xff, 0xf7, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0xff, 0xff,
    0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfd, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf,
    0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x3f,
    0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff,
    0xff, 0xff, 0xfb, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xef, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7e, 0xff, 0xff, 0xff, 0xbf, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xf9, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xf7, 0xff, 0xff, 0xff, 0xfd, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xdf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xfe, 0xf7, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff,
    0xbf, 0xff, 0xff, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff,
    0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xef, 0xfb, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xef, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe,
    0xff, 0xfb, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xff, 0xff, 0x7f, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7,
    0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xbf, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xdf,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff,
    0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0x7b, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xeb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xbf, 0xbf, 0xff, 0xff, 0xff,
    0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xf7, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x7f, 0xff, 0xff, 0xff, 0xfb,
    0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xbf, 0xfe, 0xff, 0xff, 0xff, 0xbd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff,
    0xdf, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xf7, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0x7f, 0xff, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xff, 0xff, 0xbf, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff,
    0xff, 0xfe, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xdf, 0xff, 0xff, 0xff,
    0xff, 0xfb, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xbf, 0xff, 0xff, 0xff,
    0xff, 0xfd, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfd,
    0xff, 0xff, 0xff, 0x6f, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff,
    0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfb, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff,
    0xbf, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xed, 0xff, 0xff, 0xff, 0x7f, 0xfe, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0x7f, 0xf7, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xdf, 0xff,
    0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xdf, 0xff, 0xff, 0xbf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
    0xff, 0xfd, 0xfe, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xbf, 0xff, 0xff,
    0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xff, 0xbf, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0xff,
    0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xdf, 0xff, 0xff, 0xff, 0xbf, 0xfb, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef,
    0xff, 0xff, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xef, 0xff, 0xbf, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xfd, 0x7f, 0xff, 0xff, 0x7f, 0xff, 0xf7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfb, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9f, 0xff, 0xff }
}};

} // namespace

#endif
