/*
Copyright 2022 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package reconciler

import (
	"context"

	"k8s.io/apimachinery/pkg/util/wait"
	"k8s.io/klog/v2"
)

func (rc *reconciler) Run(ctx context.Context, stopCh <-chan struct{}) {
	logger := klog.FromContext(ctx)
	rc.reconstructVolumes(logger)
	logger.Info("Reconciler: start to sync state")
	wait.Until(func() { rc.reconcile(ctx) }, rc.loopSleepDuration, stopCh)
}

func (rc *reconciler) reconcile(ctx context.Context) {
	logger := klog.FromContext(ctx)
	readyToUnmount := rc.readyToUnmount()
	if readyToUnmount {
		// Unmounts are triggered before mounts so that a volume that was
		// referenced by a pod that was deleted and is now referenced by another
		// pod is unmounted from the first pod before being mounted to the new
		// pod.
		rc.unmountVolumes(logger)
	}

	// Next we mount required volumes. This function could also trigger
	// attach if kubelet is responsible for attaching volumes.
	// If underlying PVC was resized while in-use then this function also handles volume
	// resizing.
	rc.mountOrAttachVolumes(logger)

	// Unmount volumes only when DSW and ASW are fully populated to prevent unmounting a volume
	// that is still needed, but it did not reach DSW yet.
	if readyToUnmount {
		// Ensure devices that should be detached/unmounted are detached/unmounted.
		rc.unmountDetachDevices(logger)

		// Clean up any orphan volumes that failed reconstruction.
		rc.cleanOrphanVolumes(logger)
	}

	if len(rc.volumesNeedUpdateFromNodeStatus) != 0 {
		rc.updateReconstructedFromNodeStatus(ctx)
	}
	if len(rc.volumesNeedUpdateFromNodeStatus) == 0 {
		// ASW is fully populated only after both devicePaths and uncertain volume attach-ability
		// were reconstructed from the API server.
		// This will start reconciliation of node.status.volumesInUse.
		rc.updateLastSyncTime()
	}
}
