/*
 * Copyright (c) 2018, 2025, Oracle and/or its affiliates. All rights reserved.
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The contents of this file are subject to the terms of either the Universal Permissive License
 * v 1.0 as shown at https://oss.oracle.com/licenses/upl
 *
 * or the following license:
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted
 * provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions
 * and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other materials provided with
 * the distribution.
 * 
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY
 * WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.openjdk.jmc.rjmx.test;

import static org.junit.Assert.assertTrue;

import org.junit.Test;
import org.openjdk.jmc.common.util.Environment;
import org.openjdk.jmc.rjmx.IServerHandle;
import org.openjdk.jmc.rjmx.RJMXPlugin;
import org.openjdk.jmc.rjmx.common.ConnectionDescriptorBuilder;
import org.openjdk.jmc.rjmx.common.ConnectionException;
import org.openjdk.jmc.rjmx.common.IConnectionDescriptor;
import org.openjdk.jmc.rjmx.common.IConnectionHandle;
import org.openjdk.jmc.rjmx.common.IServerDescriptor;
import org.openjdk.jmc.rjmx.common.subscription.IMBeanHelperService;
import org.openjdk.jmc.rjmx.common.subscription.IMRISubscription;
import org.openjdk.jmc.rjmx.common.subscription.IMRIValueListener;
import org.openjdk.jmc.rjmx.common.subscription.ISubscriptionService;
import org.openjdk.jmc.rjmx.common.subscription.MRI;
import org.openjdk.jmc.rjmx.common.subscription.MRIValueEvent;
import org.openjdk.jmc.rjmx.common.subscription.MRI.Type;
import org.openjdk.jmc.rjmx.servermodel.IServer;
import org.openjdk.jmc.rjmx.servermodel.IServerModel;
import org.openjdk.jmc.rjmx.subscription.PolicyFactory;

/**
 * This test suite is supposed to test the example code that we ship with the documentation for the
 * org.openjdk.jmc.rjmx bundle. That is, for each code example included in
 * org.openjdk.jmc.rjmx/src/org/openjdk/jmc/rjmx/package.html, there should be a test method in here
 * with a verbatim copy of that code.
 * <p>
 * Included in the RJMXTestSuite.
 */
// NOTE: If you change the verbatim test YOU MUST update the corresponding package.html document.
public class PackageExampleTest {
	private volatile boolean gotEvent;

	/**
	 * Tests that the package documentation example actually makes sense and compiles as expected.
	 *
	 * @throws ConnectionException
	 */
	@Test
	public void testPackageExampleVerbatim() throws Exception {
		IConnectionDescriptor descriptor = new ConnectionDescriptorBuilder().hostName("localhost").port(0).build();
		IServerHandle serverHandle = IServerHandle.create(descriptor);
		try (IConnectionHandle handle = serverHandle.connect("Usage description")) {
			ISubscriptionService service = handle.getServiceOrThrow(ISubscriptionService.class);
			MRI attribute = new MRI(Type.ATTRIBUTE, "java.lang:type=Threading", "ThreadCount");
			service.addMRIValueListener(attribute, new IMRIValueListener() {
				@Override
				public void valueChanged(MRIValueEvent event) {
					System.out.println(event.getValue());
				}
			});
			IMRISubscription subscription = service.getMRISubscription(attribute);
			subscription.setUpdatePolicy(PolicyFactory.createSimpleUpdatePolicy(1500));
		}
	}

	@Test
	public void testPackageExampleFunctionality() throws Exception {
		ConnectionDescriptorBuilder builder = new ConnectionDescriptorBuilder();
		IConnectionDescriptor descriptor = builder.hostName("localhost").port(0).build();
		try (IConnectionHandle handle = IServerHandle.create(descriptor).connect("Usage description")) {
			ISubscriptionService service = handle.getServiceOrThrow(ISubscriptionService.class);
			gotEvent = false;
			MRI attribute = new MRI(Type.ATTRIBUTE, "java.lang:type=Threading", "ThreadCount");
			service.addMRIValueListener(attribute, new IMRIValueListener() {
				@Override
				public void valueChanged(MRIValueEvent event) {
					synchronized (PackageExampleTest.this) {
						gotEvent = true;
						PackageExampleTest.this.notifyAll();
					}
				}
			});
			IMRISubscription subscription = service.getMRISubscription(attribute);
			subscription.setUpdatePolicy(PolicyFactory.createSimpleUpdatePolicy(1500));
			synchronized (PackageExampleTest.this) {
				this.wait(4000);
			}
		}
		assertTrue("Never got any event!", gotEvent);
	}

	@Test
	public void testUseServerModel() throws Exception {
		IServerModel model = RJMXPlugin.getDefault().getService(IServerModel.class);
		for (IServer server : model.elements()) {
			IServerDescriptor descriptor = server.getServerHandle().getServerDescriptor();
			if (descriptor.getJvmInfo() != null
					&& Integer.valueOf(Environment.getThisPID()).equals(descriptor.getJvmInfo().getPid())) {
				try (IConnectionHandle handle = server.getServerHandle().connect("Usage description")) {
					handle.getServiceOrThrow(IMBeanHelperService.class).getMBeanNames().size();
					return;
				}
			}
		}
	}
}
