# frozen_string_literal: true

require_relative 'gruff_test_case'

class TestGruffLine < GruffTestCase
  def test_should_render_with_transparent_theme
    skip 'This spec fails on ARM platform' if arm_platform?

    g = Gruff::Line.new(400)
    g.title = 'Transparent Background'
    g.theme = {
      colors: %w[black grey],
      marker_color: 'grey',
      font_color: 'black',
      background_colors: 'transparent'
    }

    g.labels = {
      0 => '5/6',
      1 => '5/15',
      2 => '5/24',
      3 => '5/30'
    }
    g.data(:apples, [-1, 0, 4, -4])
    g.data(:peaches, [10, 8, 6, 3])
    g.write('test/output/line_transparent.png')

    assert_same_image('test/expected/line_transparent.png', 'test/output/line_transparent.png')
  end

  def test_very_small
    g = Gruff::Line.new(200)
    g.title = 'Very Small Line Chart 200px'
    g.labels = @labels
    @datasets.each do |data|
      g.data(data[0], data[1])
    end
    g.write('test/output/line_very_small.png')

    assert_same_image('test/expected/line_very_small.png', 'test/output/line_very_small.png')
  end

  def test_line_graph_with_themes
    line_graph_with_themes
    line_graph_with_themes(400)
  end

  def test_one_value
    g = Gruff::Line.new
    g.title = 'One Value'
    g.labels = {
      0 => '1',
      1 => '2'
    }
    g.data('one', 1)

    g.write('test/output/line_one_value.png')

    assert_same_image('test/expected/line_one_value.png', 'test/output/line_one_value.png')
  end

  def test_one_value_array
    g = Gruff::Line.new
    g.title = 'One Value in an Array'
    g.labels = {
      0 => '1',
      1 => '2'
    }
    g.data('one', [1])

    g.write('test/output/line_one_value_array.png')

    assert_same_image('test/expected/line_one_value_array.png', 'test/output/line_one_value_array.png')
  end

  def test_should_not_hang_with_0_0_100
    g = Gruff::Line.new(320)
    g.title = 'Hang Value Graph Test'
    g.data('test', [0, 0, 100])

    g.write('test/output/line_hang_value.png')

    assert_same_image('test/expected/line_hang_value.png', 'test/output/line_hang_value.png')
  end

  def test_line_small_values
    @datasets = [
      [:small, [0.1, 0.14356, 0.0, 0.5674839, 0.456]],
      [:small2, [0.2, 0.3, 0.1, 0.05, 0.9]]
    ]

    g = Gruff::Line.new
    g.title = 'Small Values Line Graph Test'
    @datasets.each do |data|
      g.data(data[0], data[1])
    end
    g.write('test/output/line_small_values.png')

    assert_same_image('test/expected/line_small_values.png', 'test/output/line_small_values.png')

    g = Gruff::Line.new(400)
    g.title = 'Small Values Line Graph Test 400px'
    @datasets.each do |data|
      g.data(data[0], data[1])
    end
    g.write('test/output/line_small_values_small.png')

    assert_same_image('test/expected/line_small_values_small.png', 'test/output/line_small_values_small.png')
  end

  def test_line_starts_with_zero
    @datasets = [
      [:first0, [0, 5, 10, 8, 18]],
      [:normal, [1, 2, 3, 4, 5]]
    ]

    g = Gruff::Line.new
    g.title = 'Small Values Line Graph Test'
    @datasets.each do |data|
      g.data(data[0], data[1])
    end
    g.write('test/output/line_small_zero.png')

    assert_same_image('test/expected/line_small_zero.png', 'test/output/line_small_zero.png')

    g = Gruff::Line.new(400)
    g.title = 'Small Values Line Graph Test 400px'
    @datasets.each do |data|
      g.data(data[0], data[1])
    end
    g.write('test/output/line_small_small_zero.png')

    assert_same_image('test/expected/line_small_small_zero.png', 'test/output/line_small_small_zero.png')
  end

  def test_line_large_values
    @datasets = [
      [:large, [100_005, 35_000, 28_000, 27_000]],
      [:large2, [35_000, 28_000, 27_000, 100_005]],
      [:large3, [28_000, 27_000, 100_005, 35_000]],
      [:large4, [1_238, 39_092, 27_938, 48_876]]
    ]

    g = Gruff::Line.new
    g.title = 'Very Large Values Line Graph Test'
    g.baseline_value = 50_000
    g.baseline_color = 'green'
    g.dot_radius = 15
    g.line_width = 3
    @datasets.each do |data|
      g.data(data[0], data[1])
    end

    g.write('test/output/line_large.png')

    assert_same_image('test/expected/line_large.png', 'test/output/line_large.png')
  end

  def test_request_too_many_colors
    g = Gruff::Line.new
    g.title = 'More Sets Than in Color Array'
    @datasets.each do |data|
      g.data(data[0], data[1])
      g.data("#{data[0]}-B", data[1].map { |d| d + 20 })
    end

    g.replace_colors(
      %w[
        #0779e4
        #4cbbb9
        #77d8d8
        #2c003e
        #ffa372
        #ffbd69
        #85a392
        #efa8e4
        #5a3f11
        #2b580c
        #323232
        #bae5e5
      ]
    )
    g.write('test/output/line_more_sets_than_colors.png')

    assert_same_image('test/expected/line_more_sets_than_colors.png', 'test/output/line_more_sets_than_colors.png')
  end

  def test_many_datapoints
    g = Gruff::Line.new
    g.title = 'Many Multi-Line Graph Test'
    g.labels = {
      0 => 'June',
      10 => 'July',
      30 => 'August',
      50 => 'September'
    }
    g.data('many points', (0..50).map { rand(100) })
    g.x_axis_label = 'Months'

    # Default theme
    g.write('test/output/line_many.png')

    assert_same_image('test/expected/line_many.png', 'test/output/line_many.png')
  end

  def test_dot_style_square
    skip 'This spec fails on ARM platform' if arm_platform?

    g = Gruff::Line.new
    g.title = 'Square points'
    g.labels = {
      0 => 'June',
      10 => 'July',
      30 => 'August',
      50 => 'September'
    }
    g.dot_style = :square
    g.data('many points', (0..50).map { rand(100) })
    g.x_axis_label = 'Months'

    # Default theme
    g.write('test/output/line_dot_style_square.png')

    assert_same_image('test/expected/line_dot_style_square.png', 'test/output/line_dot_style_square.png')
  end

  def test_dot_style_diamond
    g = Gruff::Line.new
    g.title = 'Diamond points'
    g.labels = {
      0 => 'June',
      10 => 'July',
      30 => 'August',
      50 => 'September'
    }
    g.dot_style = :diamond
    g.data('many points', (0..50).map { rand(100) })
    g.x_axis_label = 'Months'

    g.write('test/output/line_dot_style_diamond.png')

    assert_same_image('test/expected/line_dot_style_diamond.png', 'test/output/line_dot_style_diamond.png')
  end

  def test_similar_high_end_values
    @dataset = %w[29.43 29.459 29.498 29.53 29.548 29.589 29.619 29.66 29.689 29.849 29.878 29.74 29.769 29.79 29.808 29.828].map(&:to_f)

    g = Gruff::Line.new
    g.title = 'Similar High End Values Test'
    g.data('similar points', @dataset)
    g.write('test/output/line_similar_high_end_values.png')

    assert_same_image('test/expected/line_similar_high_end_values.png', 'test/output/line_similar_high_end_values.png')

    g = Gruff::Line.new
    g.title = 'Similar High End Values With Floor'
    g.data('similar points', @dataset)
    g.minimum_value = 0
    g.y_axis_label = 'Barometric Pressure'
    g.write('test/output/line_similar_high_end_values_with_floor.png')

    assert_same_image('test/expected/line_similar_high_end_values_with_floor.png', 'test/output/line_similar_high_end_values_with_floor.png')
  end

  def test_many_lines_graph_small
    g = Gruff::Line.new(400)
    g.title = 'Many Values Line Test 400px'
    g.labels = {
      0 => '5/6',
      10 => '5/15',
      20 => '5/24',
      30 => '5/30',
      40 => '6/4',
      50 => '6/16'
    }
    %w[jimmy jane philip arthur julie bert].each do |student_name|
      g.data(student_name, (0..50).map { rand(100) })
    end

    # Default theme
    g.write('test/output/line_many_lines_small.png')

    assert_same_image('test/expected/line_many_lines_small.png', 'test/output/line_many_lines_small.png')
  end

  def test_graph_tiny
    g = Gruff::Line.new(300)
    g.title = 'Tiny Test 300px'
    g.labels = {
      0 => '5/6',
      10 => '5/15',
      20 => '5/24',
      30 => '5/30',
      40 => '6/4',
      50 => '6/16'
    }
    %w[jimmy jane philip arthur julie bert].each do |student_name|
      g.data(student_name, (0..50).map { rand(100) })
    end

    # Default theme
    g.write('test/output/line_tiny.png')

    assert_same_image('test/expected/line_tiny.png', 'test/output/line_tiny.png')
  end

  def test_no_data
    g = Gruff::Line.new(400)
    g.title = 'No Data'
    # Default theme
    g.write('test/output/line_no_data.png')

    assert_same_image('test/expected/line_no_data.png', 'test/output/line_no_data.png')

    g = Gruff::Line.new(400)
    g.title = 'No Data Title'
    g.no_data_message = 'There is no data'
    g.write('test/output/line_no_data_msg.png')

    assert_same_image('test/expected/line_no_data_msg.png', 'test/output/line_no_data_msg.png')

    g = Gruff::Line.new(400)
    g.data 'A', []
    g.data 'B', []
    g.write('test/output/line_no_data_with_empty.png')

    assert_same_image('test/expected/line_no_data_with_empty.png', 'test/output/line_no_data_with_empty.png')
  end

  def test_all_zeros
    g = Gruff::Line.new(400)
    g.title = 'All Zeros'

    g.data(:gus, [0, 0, 0, 0])

    # Default theme
    g.write('test/output/line_no_data_other.png')

    assert_same_image('test/expected/line_no_data_other.png', 'test/output/line_no_data_other.png')
  end

  def test_some_nil_points
    g = Gruff::Line.new
    g.title = 'Some Nil Points'

    @datasets = [
      [:data1, [1, 2, 3, nil, 3, 5, 6]],
      [:data2, [5, nil, nil, 5, nil, nil, 5]],
      [:data3, [4, nil, 2, 1, 0]],
      [:data4, [nil, nil, 3, 1, 2]]
    ]

    @datasets.each do |data|
      g.data(*data)
    end

    # Default theme
    g.write('test/output/line_some_nil_points.png')

    assert_same_image('test/expected/line_some_nil_points.png', 'test/output/line_some_nil_points.png')
  end

  def test_no_title
    g = Gruff::Line.new(400)
    g.labels = @labels
    @datasets.each do |data|
      g.data(data[0], data[1])
    end

    g.write('test/output/line_no_title.png')

    assert_same_image('test/expected/line_no_title.png', 'test/output/line_no_title.png')
  end

  def test_no_line_markers
    g = setup_basic_graph(400)
    g.title = 'No Line Markers'
    g.hide_line_markers = true
    g.write('test/output/line_no_line_markers.png')

    assert_same_image('test/expected/line_no_line_markers.png', 'test/output/line_no_line_markers.png')
  end

  def test_no_legend
    g = setup_basic_graph(400)
    g.title = 'No Legend'
    g.hide_legend = true
    g.write('test/output/line_no_legend.png')

    assert_same_image('test/expected/line_no_legend.png', 'test/output/line_no_legend.png')
  end

  def test_nothing_but_the_graph
    g = setup_basic_graph(400)
    g.title = 'THIS TITLE SHOULD NOT DISPLAY!!!'
    g.hide_line_markers = true
    g.hide_legend = true
    g.hide_title = true
    g.write('test/output/line_nothing_but_the_graph.png')

    assert_same_image('test/expected/line_nothing_but_the_graph.png', 'test/output/line_nothing_but_the_graph.png')
  end

  def test_legend_below_the_chart
    g = setup_basic_graph(400)
    g.title = 'Legend below the chart'
    g.legend_at_bottom = true
    g.write('test/output/line_legend_at_bottom.png')

    assert_same_image('test/expected/line_legend_at_bottom.png', 'test/output/line_legend_at_bottom.png')
  end

  def test_legend_below_the_chart_without_title
    g = setup_basic_graph(400)
    g.title = 'Legend below the chart'
    g.hide_title = true
    g.legend_at_bottom = true
    g.write('test/output/line_legend_at_bottom_no_title.png')

    assert_same_image('test/expected/line_legend_at_bottom_no_title.png', 'test/output/line_legend_at_bottom_no_title.png')
  end

  def test_baseline_larger_than_data
    g = setup_basic_graph(400)
    g.title = 'Baseline Larger Than Data'
    g.baseline_value = 150
    g.write('test/output/line_large_baseline.png')

    assert_same_image('test/expected/line_large_baseline.png', 'test/output/line_large_baseline.png')
  end

  def test_hide_dots
    g = setup_basic_graph(400)
    g.title = 'Hide Dots'
    g.hide_dots = true
    g.write('test/output/line_hide_dots.png')

    assert_same_image('test/expected/line_hide_dots.png', 'test/output/line_hide_dots.png')
  end

  def test_hide_lines
    g = setup_basic_graph(400)
    g.title = 'Hide Lines'
    g.hide_lines = true
    g.write('test/output/line_hide_lines.png')

    assert_same_image('test/expected/line_hide_lines.png', 'test/output/line_hide_lines.png')
  end

  def test_wide_graph
    g = setup_basic_graph('800x400')
    g.title = 'Wide Graph'
    g.write('test/output/line_wide_graph.png')

    assert_same_image('test/expected/line_wide_graph.png', 'test/output/line_wide_graph.png')

    g = setup_basic_graph('400x200')
    g.title = 'Wide Graph Small'
    g.write('test/output/line_wide_graph_small.png')

    assert_same_image('test/expected/line_wide_graph_small.png', 'test/output/line_wide_graph_small.png')
  end

  def test_negative
    g = setup_pos_neg(800)
    g.write('test/output/line_pos_neg.png')

    assert_same_image('test/expected/line_pos_neg.png', 'test/output/line_pos_neg.png')

    g = setup_pos_neg(400)
    g.title = 'Pos/Neg Line Test Small'
    g.write('test/output/line_pos_neg_400.png')

    assert_same_image('test/expected/line_pos_neg_400.png', 'test/output/line_pos_neg_400.png')
  end

  def test_all_negative
    g = setup_all_neg(800)
    g.maximum_value = 0
    g.write('test/output/line_all_neg.png')

    assert_same_image('test/expected/line_all_neg.png', 'test/output/line_all_neg.png')
  end

  def test_all_negative_no_max_value
    g = setup_all_neg(800)
    g.write('test/output/line_all_neg_no_max.png')

    assert_same_image('test/expected/line_all_neg_no_max.png', 'test/output/line_all_neg_no_max.png')
  end

  def test_all_negative_400
    g = setup_all_neg(400)
    g.maximum_value = 0
    g.title = 'All Neg Line Test Small'
    g.write('test/output/line_all_neg_400.png')

    assert_same_image('test/expected/line_all_neg_400.png', 'test/output/line_all_neg_400.png')
  end

  def test_many_numbers
    g = Gruff::Line.new('400x170')
    g.title = 'Line Test, Many Numbers'

    data = [
      {
        date: '01',
        wpm: 0,
        errors: 0,
        accuracy: 0
      },
      {
        date: '02',
        wpm: 10,
        errors: 2,
        accuracy: 80
      },
      {
        date: '03',
        wpm: 15,
        errors: 0,
        accuracy: 100
      },
      {
        date: '04',
        wpm: 16,
        errors: 2,
        accuracy: 87
      },
      { date: '05' },
      {
        date: '06',
        wpm: 18,
        errors: 1,
        accuracy: 94
      },
      { date: '07' },
      { date: '08' },
      {
        date: '09',
        wpm: 21,
        errors: 1,
        accuracy: 95
      },
      { date: '10' },
      { date: '11' },
      { date: '12' },
      { date: '13' },
      { date: '14' },
      { date: '15' },
      { date: '16' },
      { date: '17' },
      { date: '18' },
      {
        date: '19',
        wpm: 28,
        errors: 5,
        accuracy: 82
      },
      { date: '20' },
      { date: '21' },
      { date: '22' },
      { date: '23' },
      { date: '24' },
      { date: '25' },
      { date: '26' },
      {
        date: '27',
        wpm: 37,
        errors: 3,
        accuracy: 92
      }
    ]

    %i[wpm errors accuracy].each do |field|
      g.dataxy(field, data.each_with_index.filter_map { |d, i| [i + 1, d[field]] if d[field] })
    end

    labels = {}
    data.each_with_index do |d, i|
      labels[i + 1] = d[:date] if d.size > 1
    end
    g.labels = labels

    g.write('test/output/line_many_numbers.png')

    assert_same_image('test/expected/line_many_numbers.png', 'test/output/line_many_numbers.png')
  end

  def test_no_hide_line_no_labels
    g = Gruff::Line.new
    g.title = 'No Hide Line No Labels'
    @datasets.each do |data|
      g.data(data[0], data[1])
    end
    g.hide_line_markers = false
    g.write('test/output/line_no_hide.png')

    assert_same_image('test/expected/line_no_hide.png', 'test/output/line_no_hide.png')
  end

  def test_xy_data
    g = Gruff::Line.new
    g.title = 'X/Y Dataset'
    g.dataxy('Apples', [1, 3, 4, 5, 6, 10], [1, 2, 3, 4, 4, 3])
    g.dataxy('Bapples', [1, 3, 4, 5, 7, 9], [1, 1, 2, 2, 3, 3])
    g.data('Capples', [1, 1, 2, 2, 3, 3])
    g.labels = { 0 => '2003', 2 => '2004', 4 => '2005', 6 => '2006', 8 => '2007', 10 => '2008' }
    g.write('test/output/line_xy.png')

    assert_same_image('test/expected/line_xy.png', 'test/output/line_xy.png')
  end

  def test_xy_data_color
    g = Gruff::Line.new
    g.title = 'X/Y Dataset'
    g.dataxy('Apples', [1, 3, 4, 5, 6, 10], [1, 2, 3, 4, 4, 3], 'lightblue')
    g.dataxy('Bapples', [1, 3, 4, 5, 7, 9], [1, 1, 2, 2, 3, 3], '#22259F')
    g.data('Capples', [1, 1, 2, 2, 3, 3], '#A2EA00')
    g.labels = { 0 => '2003', 2 => '2004', 4 => '2005', 6 => '2006', 8 => '2007', 10 => '2008' }
    g.write('test/output/line_xy_color.png')

    assert_same_image('test/expected/line_xy_color.png', 'test/output/line_xy_color.png')
  end

  def test_xy_data_pairs
    g = Gruff::Line.new
    g.title = 'X/Y Dataset Pairs'
    g.dataxy('Apples', [[1, 1], [3, 2], [4, 3], [5, 4], [6, 4], [10, 3]])
    g.dataxy('Bapples', [[1, 1], [3, 1], [4, 2], [5, 2], [7, 3], [9, 3]])
    g.data('Capples', [1, 1, 2, 2, 3, 3])
    g.dataxy('Dapples', [[1, 1], [2, 2], [5, 6], [13, 13], [15, nil], [2, 17], [3, nil], [3, 17], [13, nil], [3, 18], [5, nil], [2, 18]])
    g.dataxy('Eapples', [[1, 1], [2, 3], [5, 8], [13, 21], [13, 8], [5, 3], [2, 1], [1, 1]])
    g.labels = { 0 => '2003', 2 => '2004', 4 => '2005', 6 => '2006', 8 => '2007', 10 => '2008', 12 => '2009' }
    g.write('test/output/line_xy_pairs.png')

    assert_same_image('test/expected/line_xy_pairs.png', 'test/output/line_xy_pairs.png')
  end

  def test_xy_data_pairs_color
    g = Gruff::Line.new
    g.title = 'X/Y Dataset Pairs'
    g.dataxy('Apples', [[1, 1], [3, 2], [4, 3], [5, 4], [6, 4], [10, 3]], 'lightblue')
    g.dataxy('Bapples', [[1, 1], [3, 1], [4, 2], [5, 2], [7, 3], [9, 3]], '#22259F')
    g.data('Capples', [1, 1, 2, 2, 3, 3], '#A2EA00')
    g.dataxy('Dapples', [[1, 1], [2, 2], [5, 6], [13, 13], [15, nil], [2, 17], [3, nil], [3, 17], [13, nil], [3, 18], [5, nil], [2, 18]], 'red')
    g.dataxy('Eapples', [[1, 1], [2, 3], [5, 8], [13, 21], [13, 8], [5, 3], [2, 1], [1, 1]])
    g.labels = { 0 => '2003', 2 => '2004', 4 => '2005', 6 => '2006', 8 => '2007', 10 => '2008', 12 => '2009' }
    g.write('test/output/line_xy_pairs_color.png')

    assert_same_image('test/expected/line_xy_pairs_color.png', 'test/output/line_xy_pairs_color.png')
  end

  def test_jruby_error
    g = Gruff::Line.new
    g.theme = {
      colors: %w[#7F0099 #2F85ED #2FED09 #EC962F],
      marker_color: '#aaa',
      background_colors: %w[#E8E8E8 #B9FD6C]
    }
    g.hide_title = true

    g.legend_font_size = 12
    g.marker_font_size = 16
    g.hide_dots = false

    g.write('test/output/line_jruby_error.png')

    assert_same_image('test/expected/line_jruby_error.png', 'test/output/line_jruby_error.png')
  end

  def test_marker_label_accuracy
    g = Gruff::Line.new
    g.title = 'Marker label accuracy'
    g.labels = {
      0 => '1',
      1 => '2',
      2 => '3',
      3 => '4',
      4 => '5'
    }
    g.data('first', [0.5, 0.51, 0.52, 0.53, 0.54])
    g.data('second', [0.6, 0.61, 0.62, 0.63, 0.64])
    g.data('third', [0.7, 0.71, 0.72, 0.73, 0.74])

    g.write('test/output/line_marker_label_accuracy.png')

    assert_same_image('test/expected/line_marker_label_accuracy.png', 'test/output/line_marker_label_accuracy.png')
  end

  def test_y_axis_increment
    g = Gruff::Line.new
    g.title = 'y axis increment'

    g.data('data', [1, 2, 3])

    g.y_axis_increment = 1

    g.write('test/output/line_y_axis_increment.png')

    assert_same_image('test/expected/line_y_axis_increment.png', 'test/output/line_y_axis_increment.png')
  end

  def test_multiple_reference_lines
    g = Gruff::Line.new
    g.title = 'Line Chart with Multiple Reference Lines'

    g.data('Apples', [3, 2, 3, 4, 4, 3])
    g.data('Oranges', [4, 8, 7, 9, 8, 9])
    g.data('Watermelon', [2, 3, 4, 5, 6, 8])
    g.data('Peaches', [9, 9, 10, 8, 7, 9])

    g.labels = { 0 => '2003', 2 => '2004', 4 => '2005' }

    g.reference_line_default_width = 1

    g.reference_lines[:baseline]  = { value: 5 }
    g.reference_lines[:lots]      = { value: 9 }
    g.reference_lines[:little]    = { value: 3 }
    g.reference_lines[:horiz_one] = { index: 1, color: 'green' }
    g.reference_lines[:horiz_two] = { index: 3, color: 'green' }

    g.write('test/output/line_reference_lines.png')

    assert_same_image('test/expected/line_reference_lines.png', 'test/output/line_reference_lines.png')
  end

  def test_baseline
    g = Gruff::Line.new
    g.title = 'Line Chart with Baseline = 5'

    g.data('Apples', [3, 2, 3, 4, 4, 3])
    g.data('Oranges', [4, 8, 7, 9, 8, 9])
    g.data('Watermelon', [2, 3, 4, 5, 6, 8])
    g.data('Peaches', [9, 9, 10, 8, 7, 9])

    g.labels = { 0 => '2003', 2 => '2004', 4 => '2005' }
    g.baseline_value = 5
    g.baseline_color = 'green'

    g.write('test/output/line_baseline.png')

    assert_same_image('test/expected/line_baseline.png', 'test/output/line_baseline.png')

    assert_equal(5, g.baseline_value)
    assert_equal('green', g.baseline_color)
  end

  def test_show_vertical_markers
    g = Gruff::Line.new
    g.title = 'Show vertical markers'

    g.data('Apples', [3, 2, 3, 4, 4, 3])
    g.data('Oranges', [4, 8, 7, 9, 8, 9])
    g.data('Watermelon', [2, 3, 4, 5, 6, 8])
    g.data('Peaches', [9, 9, 10, 8, 7, 9])

    g.labels = { 0 => '2003', 2 => '2004', 4 => '2005' }
    g.show_vertical_markers = true
    g.marker_shadow_color = '#888888'

    g.write('test/output/line_show_vertical_markers.png')

    assert_same_image('test/expected/line_show_vertical_markers.png', 'test/output/line_show_vertical_markers.png')
  end

  def test_hide_line_numbers
    g = Gruff::Line.new
    g.title = 'Show vertical markers'

    g.data('Apples', [3, 2, 3, 4, 4, 3])
    g.data('Oranges', [4, 8, 7, 9, 8, 9])
    g.data('Watermelon', [2, 3, 4, 5, 6, 8])
    g.data('Peaches', [9, 9, 10, 8, 7, 9])

    g.labels = { 0 => '2003', 2 => '2004', 4 => '2005' }
    g.show_vertical_markers = true
    g.hide_line_numbers = true
    g.marker_shadow_color = '#888888'

    g.write('test/output/line_hide_line_numbers.png')

    assert_same_image('test/expected/line_hide_line_numbers.png', 'test/output/line_hide_line_numbers.png')
  end

  def test_marker_x_count
    g = Gruff::Line.new
    g.title = 'Marker x count'

    g.data('Apples', [3, 2, 3, 4, 4, 3])
    g.data('Oranges', [4, 8, 7, 9, 8, 9])
    g.data('Watermelon', [2, 3, 4, 5, 6, 8])
    g.data('Peaches', [9, 9, 10, 8, 7, 9])

    g.labels = { 0 => '2003', 2 => '2004', 4 => '2005' }
    g.show_vertical_markers = true
    g.marker_x_count = 10
    g.marker_shadow_color = '#888888'

    g.write('test/output/line_marker_x_count.png')

    assert_same_image('test/expected/line_marker_x_count.png', 'test/output/line_marker_x_count.png')
  end

  def test_sorted_drawing
    g = setup_basic_graph('800x400')
    g.title = 'Sorted drawing'
    g.sorted_drawing = true
    g.write('test/output/line_sorted_drawing.png')

    assert_same_image('test/expected/line_sorted_drawing.png', 'test/output/line_sorted_drawing.png')
  end

  def test_webp
    g = setup_basic_graph('800x400')
    g.title = 'Line Chart WEBP'
    g.write('test/output/line_webp.webp')
  rescue Magick::ImageMagickError
    # assert_match(/no encode delegate for this image format .*\.webp/, $!.message)
  end

  def test_data_duck_typing
    g = Gruff::Line.new

    obj = Object.new
    def obj.to_a
      [1, 2, 3, 4, 5]
    end
    g.dataxy('Apples', obj, obj)

    pass
  end

  def test_minimum_x
    g = Gruff::Line.new
    g.minimum_x_value = 3
    g.dataxy('foo', [1, 2, 3, 4, 5], [21, 22, 23, 24, 25])
    g.dataxy('bar', [6, 7, 8, 9, 10], [26, 27, 28, 29, 30])

    g = Gruff::Line.new
    g.dataxy('foo', [1, 2, 3, 4, 5], [21, 22, 23, 24, 25])
    g.dataxy('bar', [6, 7, 8, 9, 10], [26, 27, 28, 29, 30])
    g.minimum_x_value = 3

    pass
  end

  def test_maximum
    g = Gruff::Line.new
    g.maximum_x_value = 3
    g.dataxy('foo', [1, 2, 3, 4, 5], [21, 22, 23, 24, 25])
    g.dataxy('bar', [6, 7, 8, 9, 10], [26, 27, 28, 29, 30])

    g = Gruff::Line.new
    g.dataxy('foo', [1, 2, 3, 4, 5], [21, 22, 23, 24, 25])
    g.dataxy('bar', [6, 7, 8, 9, 10], [26, 27, 28, 29, 30])
    g.maximum_x_value = 3

    pass
  end

  def test_empty_data
    g = Gruff::Dot.new
    g.title = 'Contained Empty Data'
    g.data('Apples', [3, 2, 3, 4, 4, 3])
    g.data('Oranges', [])
    g.data('Watermelon', nil)
    g.data('Peaches', [9, 9, 10, 8, 7, 9])

    g.write('test/output/line_empty_data.png')

    assert_same_image('test/expected/line_empty_data.png', 'test/output/line_empty_data.png')
  end

  def test_duck_typing
    g = Gruff::Line.new
    g.dataxy('foo', [1, 2, 3, 4, 5], [21, 22, 23, 24, 25])
    g.dataxy('bar', GruffCustomData.new([6, 7, 8, 9, 10]), GruffCustomData.new([26, 27, 28, 29, 30]))

    g.data :Bob, GruffCustomData.new([50, 19, 31, 89, 20, 54, 37, 65]), '#33A6B8'
    g.write('test/output/line_duck_typing.png')

    assert_same_image('test/expected/line_duck_typing.png', 'test/output/line_duck_typing.png')
  end

private

  # TODO: Reset data after each theme
  def line_graph_with_themes(size = nil)
    g = Gruff::Line.new(size)
    g.title = "Multi-Line Graph Test #{size}"
    g.labels = @labels
    g.baseline_value = 90
    @datasets.each do |data|
      g.data(data[0], data[1])
    end
    # Default theme
    g.write("test/output/line_theme_keynote_#{size}.png")

    assert_same_image("test/expected/line_theme_keynote_#{size}.png", "test/output/line_theme_keynote_#{size}.png")

    g = Gruff::Line.new(size)
    g.title = "Multi-Line Graph Test #{size}"
    g.labels = @labels
    g.baseline_value = 90
    g.theme = Gruff::Themes::THIRTYSEVEN_SIGNALS
    @datasets.each do |data|
      g.data(data[0], data[1])
    end
    g.write("test/output/line_theme_37signals_#{size}.png")

    assert_same_image("test/expected/line_theme_37signals_#{size}.png", "test/output/line_theme_37signals_#{size}.png")

    g = Gruff::Line.new(size)
    g.title = "Multi-Line Graph Test #{size}"
    g.labels = @labels
    g.baseline_value = 90
    g.theme = Gruff::Themes::RAILS_KEYNOTE
    @datasets.each do |data|
      g.data(data[0], data[1])
    end
    g.write("test/output/line_theme_rails_keynote_#{size}.png")

    assert_same_image("test/expected/line_theme_rails_keynote_#{size}.png", "test/output/line_theme_rails_keynote_#{size}.png")

    g = Gruff::Line.new(size)
    g.title = "Multi-Line Graph Test #{size}"
    g.labels = @labels
    g.baseline_value = 90
    g.theme = Gruff::Themes::ODEO
    @datasets.each do |data|
      g.data(data[0], data[1])
    end
    g.write("test/output/line_theme_odeo_#{size}.png")

    assert_same_image("test/expected/line_theme_odeo_#{size}.png", "test/output/line_theme_odeo_#{size}.png")
  end

  def setup_pos_neg(size = 800)
    g = Gruff::Line.new(size)
    g.title = 'Pos/Neg Line Graph Test'
    g.labels = {
      0 => '5/6',
      1 => '5/15',
      2 => '5/24',
      3 => '5/30'
    }
    g.data(:apples, [-1, 0, 4, -4])
    g.data(:peaches, [10, 8, 6, 3])
    g
  end

  def setup_all_neg(size = 800)
    g = Gruff::Line.new(size)
    g.title = 'All Neg Line Graph Test'
    g.labels = {
      0 => '5/6',
      1 => '5/15',
      2 => '5/24',
      3 => '5/30'
    }
    g.data(:apples, [-1, -5, -20, -4])
    g.data(:peaches, [-10, -8, -6, -3])
    g
  end
end
