/*
 * pi3's Linux kernel Runtime Guard
 *
 * Component:
 *  - Intercept SECCOMP policy update
 *
 * Notes:
 *  - Process SECCOMP Exploit Detection validation
 *
 * Caveats:
 *  - None
 *
 * Timeline:
 *  - Created: 18.XI.2017
 *
 * Author:
 *  - Adam 'pi3' Zabrocki (http://pi3.com.pl)
 *
 */

#include "../../../../p_lkrg_main.h"

#if defined(P_LKRG_EXPLOIT_DETECTION_SECCOMP_H)
char p_seccomp_kretprobe_state = 0;

static struct kretprobe p_seccomp_kretprobe = {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,17,0)
    .kp.symbol_name = "do_seccomp",
#else
    .kp.symbol_name = "prctl_set_seccomp",
#endif
    .handler = p_seccomp_ret,
    .entry_handler = p_seccomp_entry,
    .data_size = sizeof(struct p_seccomp_data),
};

/*
 * x86-64 syscall ABI:
 *  *rax - syscall_number
 *    rdi - 1st argument
 *    rsi - 2nd argument
 *    rdx - 3rd argument
 *    rcx - 4th argument
 *
 *    r8  - 5th one
 *    r9  - 6th one
 */

int p_seccomp_entry(struct kretprobe_instance *p_ri, struct pt_regs *p_regs) {

   struct p_ed_process *p_tmp;
   unsigned long p_flags;

//   p_ed_enforce_validation();

   p_tasks_write_lock(&p_flags);
   if ( (p_tmp = p_find_ed_by_pid(task_pid_nr(current))) != NULL) {
      // This process is on the ED list - set temporary 'disable' flag!
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,17,0)
      if (p_regs_get_arg2(p_regs) & SECCOMP_FILTER_FLAG_TSYNC) { // SECCOMP_FILTER_FLAG_TSYNC
         struct task_struct *p_father = current;
         struct task_struct *p_threads = p_father;
         struct p_ed_process *p_child_tmp;

         p_tmp->p_ed_task.p_sec.flag_sync_thread = 1;

         rcu_read_lock();
         // Available since 3.14.0
         for_each_thread(p_father, p_threads) {
            get_task_struct(p_threads);
            p_child_tmp = p_find_ed_by_pid(task_pid_nr(p_threads));
            if (p_child_tmp) {
#ifdef P_LKRG_TASK_OFF_DEBUG
               if (p_threads != current) {
                  p_debug_off_flag_off(p_child_tmp, 35);
               } else {
                  p_debug_off_flag_off(p_child_tmp, 33);
               }
#endif
               p_child_tmp->p_ed_task.p_sec.flag_sync_thread = 1;
               p_set_ed_process_off(p_child_tmp);
            }
            put_task_struct(p_threads);
         }
         rcu_read_unlock();
      } else {
#endif
#ifdef P_LKRG_TASK_OFF_DEBUG
         p_debug_off_flag_off(p_tmp, 33);
#endif
         p_set_ed_process_off(p_tmp);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,17,0)
      }
#endif
   }
   p_tasks_write_unlock(&p_flags);

   return 0;
}


int p_seccomp_ret(struct kretprobe_instance *p_ri, struct pt_regs *p_regs) {

   struct p_ed_process *p_tmp;
   char p_update = ((long)p_regs_get_ret(p_regs) >= 0) ? 1 : 0;
   unsigned long p_flags;

   // Update process
   p_tasks_write_lock(&p_flags);
   if ( (p_tmp = p_find_ed_by_pid(task_pid_nr(current))) != NULL) {
      // This process is on the ED list - update information!
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,17,0)
      if (p_tmp->p_ed_task.p_sec.flag_sync_thread) { // SECCOMP_FILTER_FLAG_TSYNC
         struct task_struct *p_father = current;
         struct task_struct *p_threads = p_father;
         struct p_ed_process *p_child_tmp;

         rcu_read_lock();
         // Available since 3.14.0
         for_each_thread(p_father, p_threads) {
            get_task_struct(p_threads);
            p_child_tmp = p_find_ed_by_pid(task_pid_nr(p_threads));
            if (p_child_tmp) {
               if (p_update)
                  p_update_ed_process(p_child_tmp, p_threads, 0);
               if (p_threads != p_father) {
#ifdef P_LKRG_TASK_OFF_DEBUG
                  if (p_threads != current) {
                     p_debug_off_flag_off(p_child_tmp, 36);
                  } else {
                     p_debug_off_flag_off(p_child_tmp, 34);
                  }
#endif
                  p_set_ed_process_on(p_child_tmp);
                  p_child_tmp->p_ed_task.p_sec.flag_sync_thread = 0;
               }
            }
            put_task_struct(p_threads);
         }
         rcu_read_unlock();
      } else {
#endif
         if (p_update)
            p_update_ed_process(p_tmp, current, 0);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,17,0)
      }
#endif
#ifdef P_LKRG_TASK_OFF_DEBUG
      p_debug_off_flag_on(p_tmp, 34);
#endif
      p_set_ed_process_on(p_tmp);
      // In case of seccomp failing with SECCOMP_FILTER_FLAG_TSYNC flag
      p_tmp->p_ed_task.p_sec.flag_sync_thread = 0;
   }
   p_tasks_write_unlock(&p_flags);

//   p_ed_enforce_validation();

   return 0;
}


GENERATE_INSTALL_FUNC(seccomp)
#endif
