import { ReadonlyJSONValue } from '@lumino/coreutils';
import { IDisposable } from '@lumino/disposable';
import { ISignal } from '@lumino/signaling';
/**
 * Notification manager
 */
export declare class NotificationManager implements IDisposable {
    constructor();
    /**
     * Signal emitted whenever a notification changes.
     */
    get changed(): ISignal<NotificationManager, Notification.IChange>;
    /**
     * Total number of notifications.
     */
    get count(): number;
    /**
     * Whether the manager is disposed or not.
     */
    get isDisposed(): boolean;
    /**
     * The list of notifications.
     */
    get notifications(): Notification.INotification[];
    /**
     * Dismiss one notification (specified by its id) or all if no id provided.
     *
     * @param id Notification id
     */
    dismiss(id?: string): void;
    /**
     * Dispose the manager.
     */
    dispose(): void;
    /**
     * Test whether a notification exists or not.
     *
     * @param id Notification id
     * @returns Notification status
     */
    has(id: string): boolean;
    /**
     * Add a new notification.
     *
     * This will trigger the `changed` signal with an `added` event.
     *
     * @param message Notification message
     * @param type Notification type
     * @param options Notification option
     * @returns Notification unique id
     */
    notify<T extends ReadonlyJSONValue = ReadonlyJSONValue>(message: string, type: Notification.TypeOptions, options: Notification.IOptions<T>): string;
    /**
     * Update an existing notification.
     *
     * If the notification does not exists this won't do anything.
     *
     * Once updated the notification will be moved at the begin
     * of the notification stack.
     *
     * @param args Update options
     * @returns Whether the update was successful or not.
     */
    update<T extends ReadonlyJSONValue = ReadonlyJSONValue>(args: Notification.IUpdate<T>): boolean;
    private _isDisposed;
    private _changed;
    private _queue;
}
/**
 * Notification namespace
 */
export declare namespace Notification {
    /**
     * Enumeration of available action display type.
     */
    type ActionDisplayType = 'default' | 'accent' | 'warn' | 'link';
    /**
     * Interface describing an action linked to a notification.
     */
    interface IAction {
        /**
         * The action label.
         *
         * This should be a short description.
         */
        label: string;
        /**
         * Callback function to trigger
         *
         * ### Notes
         * By default execution of the callback will close the toast
         * and dismiss the notification. You can prevent this by calling
         * `event.preventDefault()` in the callback.
         */
        callback: (event: MouseEvent) => void;
        /**
         * The action caption.
         *
         * This can be a longer description of the action.
         */
        caption?: string;
        /**
         * The action display type.
         *
         * This will be used to modify the action button style.
         */
        displayType?: ActionDisplayType;
    }
    /**
     * Notification interface
     */
    interface INotification<T extends ReadonlyJSONValue = ReadonlyJSONValue> {
        /**
         * Notification unique identifier
         */
        id: string;
        /**
         * Notification message
         *
         * #### Notes
         * The message will be truncated if longer than 140 characters.
         */
        message: string;
        /**
         * Notification creation date
         */
        createdAt: number;
        /**
         * Notification modification date
         */
        modifiedAt: number;
        /**
         * Notification type
         */
        type: TypeOptions;
        /**
         * Notification options
         */
        options: IOptions<T>;
    }
    /**
     * Notification change interface
     */
    interface IChange {
        /**
         * Change type
         */
        type: 'added' | 'removed' | 'updated';
        /**
         * Notification that changed
         */
        notification: INotification;
    }
    /**
     * Notification options
     */
    interface IOptions<T extends ReadonlyJSONValue> {
        /**
         * Autoclosing behavior - false (not closing automatically)
         * or number (time in milliseconds before hiding the notification)
         *
         * Set to zero if you want the notification to be retained in the notification
         * center but not displayed as toast. This is the default behavior.
         */
        autoClose?: number | false;
        /**
         * List of associated actions
         */
        actions?: Array<IAction>;
        /**
         * Data associated with a notification
         */
        data?: T;
        /**
         * Task progression
         *
         * ### Notes
         * This should be a number between 0 (not started) and 1 (completed).
         */
        progress?: number;
    }
    /**
     * Parameters for notification depending on a promise.
     */
    interface IPromiseOptions<Pending extends ReadonlyJSONValue, Success extends ReadonlyJSONValue = Pending, Error extends ReadonlyJSONValue = Pending> {
        /**
         * Promise pending message and options
         *
         * #### Notes
         * The message will be truncated if longer than 140 characters.
         */
        pending: {
            message: string;
            options?: IOptions<Pending>;
        };
        /**
         * Message when promise resolves and options
         *
         * The message factory receives as first argument the result
         * of the promise and as second the success `options.data`.
         *
         * #### Notes
         * The message will be truncated if longer than 140 characters.
         */
        success: {
            message: (result: unknown, data?: Success) => string;
            options?: IOptions<Success>;
        };
        /**
         * Message when promise rejects and options
         *
         * The message factory receives as first argument the error
         * of the promise and as second the error `options.data`.
         *
         * #### Notes
         * The message will be truncated if longer than 140 characters.
         */
        error: {
            message: (reason: unknown, data?: Error) => string;
            options?: IOptions<Error>;
        };
    }
    /**
     * Type of notifications
     */
    type TypeOptions = 'info' | 'in-progress' | 'success' | 'warning' | 'error' | 'default';
    /**
     * Options for updating a notification
     */
    interface IUpdate<T extends ReadonlyJSONValue> extends IOptions<T> {
        /**
         * Notification unique id
         */
        id: string;
        /**
         * New notification message
         */
        message?: string;
        /**
         * New notification type
         */
        type?: TypeOptions;
    }
    /**
     * The global notification manager.
     */
    const manager: NotificationManager;
    /**
     * Dismiss one notification (specified by its id) or all if no id provided
     *
     * @param id notification id
     */
    function dismiss(id?: string): void;
    /**
     * Helper function to emit a notification.
     *
     * #### Notes
     * The message will be truncated if longer than 140 characters.
     *
     * @param message Notification message
     * @param type Notification type
     * @param options Options for the error notification
     * @returns Notification unique id
     */
    function emit<T extends ReadonlyJSONValue = ReadonlyJSONValue>(message: string, type?: TypeOptions, options?: IOptions<T>): string;
    /**
     * Helper function to emit an error notification.
     *
     * #### Notes
     * The message will be truncated if longer than 140 characters.
     *
     * @param message Notification message
     * @param options Options for the error notification
     * @returns Notification unique id
     */
    function error<T extends ReadonlyJSONValue = ReadonlyJSONValue>(message: string, options?: IOptions<T>): string;
    /**
     * Helper function to emit an info notification.
     *
     * #### Notes
     * The message will be truncated if longer than 140 characters.
     *
     * @param message Notification message
     * @param options Options for the info notification
     * @returns Notification unique id
     */
    function info<T extends ReadonlyJSONValue = ReadonlyJSONValue>(message: string, options?: IOptions<T>): string;
    /**
     * Helper function to show an in-progress notification.
     *
     * #### Notes
     * The message will be truncated if longer than 140 characters.
     *
     * @param promise Promise to wait for
     * @param options Options for the in-progress notification
     * @returns Notification unique id
     */
    function promise<Pending extends ReadonlyJSONValue = ReadonlyJSONValue, Success extends ReadonlyJSONValue = Pending, Error extends ReadonlyJSONValue = Pending>(promise: Promise<Success>, options: IPromiseOptions<Pending, Success, Error>): string;
    /**
     * Helper function to emit a success notification.
     *
     * #### Notes
     * The message will be truncated if longer than 140 characters.
     *
     * @param message Notification message
     * @param options Options for the success notification
     * @returns Notification unique id
     */
    function success<T extends ReadonlyJSONValue = ReadonlyJSONValue>(message: string, options?: IOptions<T>): string;
    /**
     * Helper function to update a notification.
     *
     * If the notification does not exists, nothing will happen.
     *
     * Once updated the notification will be moved at the begin
     * of the notification stack.
     *
     * #### Notes
     * The message will be truncated if longer than 140 characters.
     *
     * @param args Update options
     * @returns Whether the update was successful or not.
     */
    function update<T extends ReadonlyJSONValue = ReadonlyJSONValue>(args: IUpdate<T>): boolean;
    /**
     * Helper function to emit a warning notification.
     *
     * #### Notes
     * The message will be truncated if longer than 140 characters.
     *
     * @param message Notification message
     * @param options Options for the warning notification
     * @returns Notification unique id
     */
    function warning<T extends ReadonlyJSONValue = ReadonlyJSONValue>(message: string, options?: IOptions<T>): string;
}
