/* 
 * tools.c
 * Copyright (C) 2002, AVM GmbH. All rights reserved.
 * 
 * This Software is  free software. You can redistribute and/or
 * modify such free software under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * The free software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this Software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA, or see
 * http://www.opensource.org/licenses/lgpl-license.html
 * 
 * Contact: AVM GmbH, Alt-Moabit 95, 10559 Berlin, Germany, email: info@avm.de
 */

#include <linux/string.h>
#include <linux/vmalloc.h>
#include <linux/kernel.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,18)
#include <linux/slab.h>
#else
#include <linux/malloc.h>
#endif

#include "defs.h"
#include "libdefs.h"
#include "driver.h"
#include "tools.h"

typedef struct __header {

	unsigned	type;
#if !defined (NDEBUG)
	unsigned	size;
	unsigned	from;
	unsigned	tag;
#endif
} header_t;

#define	TYPE_NONE	'?'
#define TYPE_KMALLOCED	'k'
#define	TYPE_VMALLOCED	'v'
#define	TYPE_SMALLOCED	's'

#define	ALLOC_NORMAL	0
#define	ALLOC_SPECIAL	1

#define	PRIORITY	GFP_ATOMIC

static void *		lib_heap_base	= NULL;
static unsigned		lib_heap_size	= 0;

#if !defined (NDEBUG)
#include <asm/atomic.h>

#define	FENCE_TAG	0xDEADBEEF
#define	FENCE1_OK(h,m)	((h)->tag==FENCE_TAG)
#define	FENCE2_OK(h,m)	(*(unsigned *)(((char *) m)+(h)->size)==FENCE_TAG)

static atomic_t		alloc_count	= ATOMIC_INIT (0);

#if !defined (NDEBUG) && defined (LOG_TIMER)
static struct timeval	zero_time;
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned hallocated (void) {
    
	return atomic_read (&alloc_count);
} /* hallocated */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned hallocator (void * mem) {
	header_t * hdr;

	if (mem != NULL) {
		hdr = ((header_t *) mem) - 1;
		return FENCE1_OK(hdr, mem) ? 0 : hdr->from;
	} else {
		return 0;
	}
} /* hallocator */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int hvalid (void * mem) {
	header_t *	hdr;
	int		flag = TRUE;

	if (mem != NULL) {
		hdr  = ((header_t *) mem) - 1;
		flag = FENCE1_OK(hdr, mem) && FENCE2_OK(hdr, mem);
	} 
	return flag;
} /* hvalid */
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void * ALLOCATE (unsigned n, unsigned * t, unsigned m) {
	void *		temp;
	unsigned	type;

	temp = kmalloc (n, PRIORITY);
	type = TYPE_KMALLOCED;
	if (temp == NULL) {
		if (ALLOC_NORMAL == m) {
			assert (capi_lib->lib_heap_alloc != NULL);
			info (lib_heap_base != NULL);
			temp = (*capi_lib->lib_heap_alloc) (lib_heap_base, n);
			type = TYPE_SMALLOCED;
		} else {
			temp = vmalloc (n);
			type = TYPE_VMALLOCED;
		}
	}
	info (temp != NULL);
	*t = (temp != NULL) ? type : TYPE_NONE;
#if defined (LOG_ALLOC)
	log ("ALLOC (%u,%u) = 0x%p, type %c\n", n, m, temp, (char) *t);
#endif
	return temp;
} /* ALLOCATE */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if !defined (NDEBUG)
#define	PATCH(n)	sizeof(header_t)+sizeof(unsigned)+((n)?(n):1)
#else
#define	PATCH(n)	sizeof(header_t)+((n)?(n):1)
#endif

static void * halloc (unsigned size, unsigned addr, unsigned mode) {
	unsigned	n, t = TYPE_NONE;
	void *		mem;
	header_t *	hdr;

	n = PATCH(size);
	if (NULL == (hdr = (header_t *) ALLOCATE (n, &t, mode))) {
		log (
			"Memory request (%u/%u bytes) failed. (mode %u)\n", 
			size, 
			PATCH(size),
			mode
		);
		mem = NULL;
	} else {
		mem = (void *) (hdr + 1);
		hdr->type = t;
#if !defined (NDEBUG)
		hdr->size = size ? size : 1;
		hdr->from = addr;
		hdr->tag  = FENCE_TAG;
		* (unsigned *) (((char *) mem) + size) = FENCE_TAG;
		atomic_add (size, &alloc_count);
#else
		UNUSED_ARG (addr);
#endif
	}
	return mem;
} /* halloc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned libheap_init (unsigned heap_size) {
	void *	heap_base = NULL;

	assert (lib_heap_base == NULL);
	assert (heap_size > MIN_LIB_HEAP_SIZE);
	do {
		if (NULL != (heap_base = halloc (heap_size, (unsigned) &libheap_init, ALLOC_SPECIAL))) {
			lib_heap_base = heap_base;
			lib_heap_size = heap_size;
			(*capi_lib->lib_heap_init) (heap_base, heap_size);
			return heap_size;
		}
		heap_size /= 2;
	} while (heap_size > 0);
	return 0;
} /* libheap_init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void libheap_exit (void) {

	assert (lib_heap_base != NULL);
	(*capi_lib->lib_heap_exit) (lib_heap_base);
	hfree (lib_heap_base);
	lib_heap_base = NULL;
	lib_heap_size = 0;
} /* libheap_exit */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * hmalloc (unsigned size) {

	return halloc (size, *(((unsigned *) &size) - 1), ALLOC_NORMAL);
} /* hmalloc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * hcalloc (unsigned size) {
	void * mem;

	mem = halloc (size, *(((unsigned *) &size) - 1), ALLOC_NORMAL);
	if ((mem != NULL) && (size != 0)) {
		memset (mem, 0, size);
	}
	return mem;
} /* hcalloc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void hfree (void * mem) {
	header_t *	hdr;

	if (mem != NULL) {
		hdr = ((header_t *) mem) - 1;
#if !defined (NDEBUG)
		if (!(FENCE1_OK(hdr, mem) && FENCE2_OK(hdr, mem))) {
			log (
				"FENCE VIOLATED (%u/0x%08X)!\n", 
				hdr->size, 
				hdr->from
			);
		}
		atomic_sub (hdr->size, &alloc_count);
#if defined (LOG_ALLOC)
		log ("FREE (0x%p, %u), type %c\n", hdr, hdr->size, hdr->type);
#endif
#endif
		switch (hdr->type) {

		case TYPE_KMALLOCED:	
			kfree (hdr);	
			break;
		case TYPE_VMALLOCED:	
			vfree (hdr);	
			break;
		case TYPE_SMALLOCED:	
			assert (capi_lib->lib_heap_free != NULL);
			info (lib_heap_base != NULL);
			(*capi_lib->lib_heap_free) (lib_heap_base, hdr);	
			break;
		default:		
			assert (0);	
			break;
		}
	}
} /* hfree */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if !defined (NDEBUG) && defined (LOG_TIMER)

#include <linux/time.h>

void setup_timer (dbg_timer * t, long dsec, long dusec) {
	
	assert (t != NULL);
	memset (&t->t, 0, sizeof (t->t));
	t->d.tv_sec  = dsec;
	t->d.tv_usec = dusec;
} /* dbg_setup_timer */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int check_timer (dbg_timer * t) {
	int		res = 1;
	struct timeval	now;
	struct timeval	delta;
	
	assert (t != NULL);
	do_gettimeofday (&now);
	timeval_less (now, zero_time, &delta);
	now = delta;
	timeval_less (now, t->t, &delta);
	if ((delta.tv_sec > t->d.tv_sec) 
	|| ((delta.tv_sec == t->d.tv_sec) && (delta.tv_usec > t->d.tv_usec))
	) {
		lprintf (
			KERN_INFO, 
			"Timer '%s' exceeded: %ld s, %ld s\n", 
			t->name,
			delta.tv_sec,
			delta.tv_usec
		);
		res = 0;
	} 
	return res;
} /* check_timer */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void touch_timer (dbg_timer * t) {
	struct timeval	temp, delta;
	
	assert (t != NULL);
	do_gettimeofday (&temp);
	timeval_less (temp, zero_time, &delta);
	t->t = delta;
} /* touch_timer */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void start_watch (dbg_timer * w) {
	struct timeval	temp, delta;
	
	assert (w != NULL);
	do_gettimeofday (&temp);
	timeval_less (temp, zero_time, &delta);
	w->t = delta;
} /* start_watch */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void stop_watch (dbg_timer * w) {
	struct timeval	temp, delta;
	
	assert (w != NULL);
	do_gettimeofday (&temp);
	timeval_less (temp, zero_time, &delta);
	temp = delta;
	timeval_less (temp, w->t, &delta);
	w->t = delta;
} /* stop_watch */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int timers_start (void) {

	do_gettimeofday (&zero_time);
	return 1;
} /* timers_start */
     
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void timers_stop (void) {

} /* timers_stop */

#endif /* !NDEBUG && LOG_TIMER */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void vlprintf (const char * level, const char * fmt, va_list args) {
	static char line[256];

	VSNPRINTF (line, sizeof (line), fmt, args);
	printk ("%s%s: %s", level, TARGET, line);
} /* vlprintf */
 
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void lprintf (const char * level, const char * fmt, ...) {
	va_list args;

	va_start (args, fmt);
	vlprintf (level, fmt, args);
	va_end (args);
} /* lprintf */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void message (const char * fmt, ...) {
	va_list args;

	va_start (args, fmt);
	vlprintf (KERN_INFO, fmt, args);
	va_end (args);
} /* message */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#ifndef NDEBUG
void _OSassert (void * exp, void * file, unsigned line) {

	message ("assert (%s) in %s(%u)\n", exp, file, line);
} /* _OSassert */

void _OSinfo (void * exp, void * file, unsigned line) {

	message ("info (%s) in %s(%u)\n", exp, file, line);
} /* _OSinfo */
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
