;;; dribble.lisp
;;;
;;; Copyright (C) 2004 Peter Graves
;;; $Id$
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License
;;; as published by the Free Software Foundation; either version 2
;;; of the License, or (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;;;
;;; As a special exception, the copyright holders of this library give you
;;; permission to link this library with independent modules to produce an
;;; executable, regardless of the license terms of these independent
;;; modules, and to copy and distribute the resulting executable under
;;; terms of your choice, provided that you also meet, for each linked
;;; independent module, the terms and conditions of the license of that
;;; module.  An independent module is a module which is not derived from
;;; or based on this library.  If you modify this library, you may extend
;;; this exception to your version of the library, but you are not
;;; obligated to do so.  If you do not wish to do so, delete this
;;; exception statement from your version.


;;; Each time we start dribbling to a new stream, we put it in
;;; *DRIBBLE-STREAM*, and push a list of *DRIBBLE-STREAM*, *STANDARD-INPUT*,
;;; *STANDARD-OUTPUT* and *ERROR-OUTPUT* in *PREVIOUS-DRIBBLE-STREAMS*.
;;; *STANDARD-OUTPUT* and *ERROR-OUTPUT* is changed to a broadcast stream that
;;; broadcasts to *DRIBBLE-STREAM* and to the old values of the variables.
;;; *STANDARD-INPUT* is changed to an echo stream that echos input from the old
;;; value of standard input to *DRIBBLE-STREAM*.
;;;
;;; When dribble is called with no arguments, *DRIBBLE-STREAM* is closed,
;;; and the values of *DRIBBLE-STREAM*, *STANDARD-INPUT*, and
;;; *STANDARD-OUTPUT* are popped from *PREVIOUS-DRIBBLE-STREAMS*.

;;; From SBCL.

(in-package "SYSTEM")

(defvar *previous-dribble-streams* nil)
(defvar *dribble-stream* nil)

(defun dribble (&optional pathname &key (if-exists :append))
  "With a file name as an argument, dribble opens the file and sends a
  record of further I/O to that file. Without an argument, it closes
  the dribble file, and quits logging."
  (cond (pathname
         (let* ((new-dribble-stream
                 (open pathname
                       :direction :output
                       :if-exists if-exists
                       :if-does-not-exist :create))
                (new-standard-output
                 (make-broadcast-stream *standard-output* new-dribble-stream))
                (new-error-output
                 (make-broadcast-stream *error-output* new-dribble-stream))
                (new-standard-input
                 (make-echo-stream *standard-input* new-dribble-stream)))
           (push (list *dribble-stream* *standard-input* *standard-output*
                       *error-output*)
                 *previous-dribble-streams*)
           (setf *dribble-stream* new-dribble-stream)
           (setf *standard-input* new-standard-input)
           (setf *standard-output* new-standard-output)
           (setf *error-output* new-error-output)
           ;; Starting a new internal REPL for dribbling
           (loop do
             (format t "~a> " (package-name *package*))
             (with-simple-restart (abort "Error detected in dribbling")
               (handler-case
                 (let ((input (read *standard-input*)))
                   (print (eval input) *standard-output*)
                   (terpri)
                   (when (equal input '(dribble))
                     (return)))
                 (error (c)
                   (format *error-output* "~a~%" c)
                   (error c)))))))
        ((null *dribble-stream*)
         (error "Not currently dribbling."))
        (t
         (let ((old-streams (pop *previous-dribble-streams*)))
           (close *dribble-stream*)
           (setf *dribble-stream* (first old-streams))
           (setf *standard-input* (second old-streams))
           (setf *standard-output* (third old-streams))
           (setf *error-output* (fourth old-streams)))))
  (values))
