// Copyright © by Jeff Foley 2017-2025. All rights reserved.
// Use of this source code is governed by Apache 2 LICENSE that can be found in the LICENSE file.
// SPDX-License-Identifier: Apache-2.0

package server

// This file will be automatically regenerated based on the schema, any resolver implementations
// will be copied through when generating and any unknown code will be moved to the end.
// Code generated by github.com/99designs/gqlgen version v0.17.41

import (
	"context"
	"encoding/json"
	"errors"

	"github.com/google/uuid"
	"github.com/owasp-amass/amass/v5/config"
	"github.com/owasp-amass/amass/v5/engine/api/graphql/server/model"
	et "github.com/owasp-amass/amass/v5/engine/types"
	oam "github.com/owasp-amass/open-asset-model"
	"github.com/owasp-amass/open-asset-model/account"
	oamcert "github.com/owasp-amass/open-asset-model/certificate"
	"github.com/owasp-amass/open-asset-model/contact"
	oamdns "github.com/owasp-amass/open-asset-model/dns"
	"github.com/owasp-amass/open-asset-model/file"
	"github.com/owasp-amass/open-asset-model/financial"
	"github.com/owasp-amass/open-asset-model/general"
	"github.com/owasp-amass/open-asset-model/network"
	"github.com/owasp-amass/open-asset-model/org"
	"github.com/owasp-amass/open-asset-model/people"
	"github.com/owasp-amass/open-asset-model/platform"
	oamreg "github.com/owasp-amass/open-asset-model/registration"
	"github.com/owasp-amass/open-asset-model/url"
)

// CreateSession is the resolver for the createSession field.
func (r *mutationResolver) CreateSession(ctx context.Context, input model.CreateSessionInput) (*model.Session, error) {
	testSession := &model.Session{
		SessionToken: "00000000-0000-0000-0000-0000000000033", //?
	}
	return testSession, nil
}

// CreateSessionFromJSON is the resolver for the createSessionFromJson field.
func (r *mutationResolver) CreateSessionFromJSON(ctx context.Context, input model.CreateSessionJSONInput) (*model.Session, error) {
	var config config.Config

	if err := json.Unmarshal([]byte(input.Config), &config); err != nil {
		return nil, err
	}
	// Populate FROM/TO in transformations
	for k, t := range config.Transformations {
		_ = t.Split(k)
	}

	s, err := r.Manager.NewSession(&config)
	if err != nil {
		return nil, err
	}

	return &model.Session{SessionToken: s.ID().String()}, nil
}

// CreateAsset is the resolver for the createAsset field.
func (r *mutationResolver) CreateAsset(ctx context.Context, input model.CreateAssetInput) (*model.Asset, error) {
	// Check if the session token is valid
	token, err := uuid.Parse(input.SessionToken)
	if err != nil {
		return nil, errors.New("invalid session token")
	}
	// Check if the session exists
	// and if the session is not already terminated
	session := r.Manager.GetSession(token)
	if session == nil {
		return nil, errors.New("invalid session token")
	}

	data, ok := input.Data.(map[string]interface{})
	if !ok {
		return nil, errors.New("failed to cast the asset data")
	}
	atype := data["type"].(string)

	j, err := json.Marshal(input.Data)
	if err != nil {
		return nil, err
	}
	// Unmarshal json into AssetData struct
	assetData := &et.AssetData{
		OAMAsset: createSeedAsset(atype),
	}
	if err := json.Unmarshal(j, assetData); err != nil {
		return nil, err
	}

	dba, err := session.Cache().CreateAsset(assetData.OAMAsset)
	if err != nil {
		return nil, err
	}

	// Create and schedule new event
	event := &et.Event{
		Name:       *input.AssetName,
		Entity:     dba,
		Dispatcher: r.Dispatcher,
		Session:    session,
	}

	if err := r.Dispatcher.DispatchEvent(event); err != nil {
		return nil, errors.New("failed to create asset")
	}

	return &model.Asset{ID: token.String()}, nil
}

// TerminateSession is the resolver for the terminateSession field.
func (r *mutationResolver) TerminateSession(ctx context.Context, sessionToken string) (*bool, error) {
	var result bool
	// Check if the session token is valid
	token, err := uuid.Parse(sessionToken)
	if err != nil {
		return &result, errors.New("invalid session token")
	}
	// Check if the session exists
	// and if the session is not already terminated
	if r.Manager.GetSession(token) == nil {
		return &result, errors.New("invalid session token")
	}

	result = true
	go r.Manager.CancelSession(token)
	return &result, nil
}

// SessionStats is the resolver for the sessionStats field.
func (r *queryResolver) SessionStats(ctx context.Context, sessionToken string) (*model.SessionStats, error) {
	// Check if the session token is valid
	token, err := uuid.Parse(sessionToken)
	if err != nil {
		return nil, errors.New("invalid session token")
	}
	// Check if the session exists
	// and if the session is not already terminated
	session := r.Manager.GetSession(token)
	if session == nil {
		return nil, errors.New("invalid session token")
	}

	stats := session.Stats()
	stats.Lock()
	completed := stats.WorkItemsCompleted
	total := stats.WorkItemsTotal
	stats.Unlock()

	return &model.SessionStats{
		WorkItemsCompleted: &completed,
		WorkItemsTotal:     &total,
	}, nil
}

// LogMessages is the resolver for the logMessages field.
func (r *subscriptionResolver) LogMessages(ctx context.Context, sessionToken string) (<-chan *string, error) {
	token, _ := uuid.Parse(sessionToken)
	session := r.Manager.GetSession(token)

	if session != nil {
		session.PubSub().Publish("Channel created")
		ch := session.PubSub().Subscribe()
		return ch, nil
	}
	return nil, nil
}

// Mutation returns MutationResolver implementation.
func (r *Resolver) Mutation() MutationResolver { return &mutationResolver{r} }

// Query returns QueryResolver implementation.
func (r *Resolver) Query() QueryResolver { return &queryResolver{r} }

// Subscription returns SubscriptionResolver implementation.
func (r *Resolver) Subscription() SubscriptionResolver { return &subscriptionResolver{r} }

type mutationResolver struct{ *Resolver }
type queryResolver struct{ *Resolver }
type subscriptionResolver struct{ *Resolver }

func createSeedAsset(atype string) oam.Asset {
	switch atype {
	case string(oam.Account):
		return &account.Account{}
	case string(oam.AutnumRecord):
		return &oamreg.AutnumRecord{}
	case string(oam.AutonomousSystem):
		return &network.AutonomousSystem{}
	case string(oam.ContactRecord):
		return &contact.ContactRecord{}
	case string(oam.DomainRecord):
		return &oamreg.DomainRecord{}
	case string(oam.File):
		return &file.File{}
	case string(oam.FQDN):
		return &oamdns.FQDN{}
	case string(oam.FundsTransfer):
		return &financial.FundsTransfer{}
	case string(oam.Identifier):
		return &general.Identifier{}
	case string(oam.IPAddress):
		return &network.IPAddress{}
	case string(oam.IPNetRecord):
		return &oamreg.IPNetRecord{}
	case string(oam.Location):
		return &contact.Location{}
	case string(oam.Netblock):
		return &network.Netblock{}
	case string(oam.Organization):
		return &org.Organization{}
	case string(oam.Phone):
		return &contact.Phone{}
	case string(oam.Person):
		return &people.Person{}
	case string(oam.Product):
		return &platform.Product{}
	case string(oam.ProductRelease):
		return &platform.ProductRelease{}
	case string(oam.Service):
		return &platform.Service{}
	case string(oam.TLSCertificate):
		return &oamcert.TLSCertificate{}
	case string(oam.URL):
		return &url.URL{}
	}
	return nil
}
