#ifndef _SERVERPOOL_H
#define _SERVERPOOL_H 1

#include <stddef.h>
#include <memory>
#include <string>
#include <vector>

#include "server.h"
#include "state.pb.h"
#include "stream.h"
#include "udpstream.h"

class Acceptor;
class Server;
class UDPStream;
struct ClientStats;
struct sockaddr_in6;

// Provides services such as load-balancing between a number of Server instances.
class ServerPool {
public:
	ServerPool(int num_servers);

	// Fills streams() and clients().
	CubemapStateProto serialize();

	// Picks a server (round-robin) and allocates the given client to it.
	void add_client(int sock, Acceptor *acceptor);
	void add_client_from_serialized(const ClientProto &client, const std::vector<std::shared_ptr<const std::string>> &short_responses);

	// Picks a srever (round-robin) and adds the given HLS zombie to it.
	void add_hls_zombie_from_serialized(const HLSZombieProto &client);

	// Adds the given stream to all the servers. Returns the stream index.
	int add_stream(const std::string &url,
	               const std::string &hls_url,
	               size_t backlog_size,
	               size_t prebuffering_bytes,
	               Stream::Encoding encoding,
	               Stream::Encoding src_encoding,
                       unsigned hls_frag_duration,
                       size_t hls_backlog_margin,
                       const std::string &allow_origin);
	int add_stream_from_serialized(const StreamProto &stream, const std::vector<int> &data_fds);
	void delete_stream(const std::string &url);
	int add_udpstream(const sockaddr_in6 &dst, int pacing_rate, int ttl, int multicast_iface_index);

	// Returns the stream index for the given URL (e.g. /foo.ts). Returns -1 on failure.
	int lookup_stream_by_url(const std::string &url) const;

	// Adds the given data to all the servers.
	void set_header(int stream_index,
	                const std::string &http_header,
	                const std::string &stream_header);
	void add_data(int stream_index, const char *data, size_t bytes, uint16_t metacube_flags, const RationalPTS &pts);

	// Sets the given stream as unavailable on all the servers.
	void set_unavailable(int stream_index);

	// Sets the max pacing rate for all the servers.
	void set_pacing_rate(int stream_index, uint32_t pacing_rate);

	// Changes the given stream's backlog size on all the servers.
	void set_backlog_size(int stream_index, size_t new_size);

	// Changes the given stream's amount of forced prebuffering on all the servers.
	void set_prebuffering_bytes(int stream_index, size_t new_amount);

	// Changes the given stream's output encoding type on all the servers.
	void set_encoding(int stream_index, Stream::Encoding encoding);

	// Changes the given stream's input encoding type on all the servers.
	void set_src_encoding(int stream_index, Stream::Encoding encoding);

	// Changes the given stream's maximum HLS fragment duration (in seconds) on all the servers.
	void set_hls_frag_duration(int stream_index, unsigned hls_frag_duration);

	// Changes the given stream's backlog margin for HLS fragments (in bytes) on all the servers.
	void set_hls_backlog_margin(int stream_index, size_t hls_backlog_margin);

	// Changes the given stream's CORS header on all the servers.
	void set_allow_origin(int stream_index, const std::string &allow_origin);

	// Register the given stream under the given URL on all the servers.
	// Used only for deserialized streams (for new ones, we do this registration
	// in add_stream()).
	void register_hls_url(int stream_index, const std::string &hls_url);

	// Adds the given gen204 endpoint to all the servers.
	void add_gen204(const std::string &url, const std::string &allow_origin);

	// Prepares all the servers for accepting TLS connections from the given acceptor.
	// (They need a private context, since the contexts are not definde to be thread-safe.)
	void create_tls_context_for_acceptor(const Acceptor *acceptor);

	// Starts all the servers.
	void run();

	// Stops all the servers.
	void stop();

	std::vector<ClientStats> get_client_stats() const;
	std::vector<HLSZombie> get_hls_zombies() const;

private:
	std::unique_ptr<Server[]> servers;
	int num_servers, clients_added = 0;

	// Our indexing is currently rather primitive; every stream_index in
	// [0, num_http_streams) maps to a HTTP stream (of which every Server
	// has exactly one copy), and after that, it's mapping directly into
	// <udp_streams>.
	int num_http_streams = 0;
	std::vector<std::unique_ptr<UDPStream>> udp_streams;

	ServerPool(const ServerPool &);
};

#endif  // !defined(_SERVERPOOL_H)
