import json

import dateutil
from cvss import parser as cvss_parser

from dojo.models import Finding


class GithubVulnerabilityParser:
    def get_scan_types(self):
        return ["Github Vulnerability Scan"]

    def get_label_for_scan_types(self, scan_type):
        return scan_type  # no custom label for now

    def get_description_for_scan_types(self, scan_type):
        return "Import vulnerabilities from Github API (GraphQL Query)"

    def get_findings(self, filename, test):
        data = json.load(filename)
        if "data" in data:
            vulnerabilityAlerts = self._search_vulnerability_alerts(data["data"])
            if not vulnerabilityAlerts:
                msg = "Invalid report, no 'vulnerabilityAlerts' node found"
                raise ValueError(msg)
            repository_url = None
            if "repository" in data["data"]:
                if "nameWithOwner" in data["data"]["repository"]:
                    repository_url = "https://github.com/{}".format(
                        data["data"]["repository"]["nameWithOwner"],
                    )
                if "url" in data["data"]["repository"]:
                    repository_url = data["data"]["repository"]["url"]
            dupes = {}
            for alert in vulnerabilityAlerts["nodes"]:
                description = alert["securityVulnerability"]["advisory"][
                    "description"
                ]
                if "number" in alert and repository_url is not None:
                    dependabot_url = (
                        repository_url
                        + "/security/dependabot/{}".format(alert["number"])
                    )
                    description = (
                        f"[{dependabot_url}]({dependabot_url})\n"
                        + description
                    )
                finding = Finding(
                    title=alert["securityVulnerability"]["advisory"]["summary"],
                    test=test,
                    description=description,
                    severity=self._convert_security(
                        alert["securityVulnerability"].get("severity", "MODERATE"),
                    ),
                    static_finding=True,
                    dynamic_finding=False,
                    unique_id_from_tool=alert["id"],
                )
                if "vulnerableManifestPath" in alert:
                    finding.file_path = alert["vulnerableManifestPath"]
                if "vulnerableRequirements" in alert and alert["vulnerableRequirements"].startswith("= "):
                    finding.component_version = alert["vulnerableRequirements"][2:]
                if "createdAt" in alert:
                    finding.date = dateutil.parser.parse(alert["createdAt"])
                if "state" in alert and (
                    "FIXED" == alert["state"] or "DISMISSED" == alert["state"]
                ):
                    finding.active = False
                    finding.is_mitigated = True
                # if the package is present
                if "package" in alert["securityVulnerability"]:
                    finding.component_name = alert["securityVulnerability"][
                        "package"
                    ].get("name")
                if "references" in alert["securityVulnerability"]["advisory"]:
                    finding.references = ""
                    for ref in alert["securityVulnerability"]["advisory"][
                        "references"
                    ]:
                        finding.references += ref["url"] + "\r\n"
                if "identifiers" in alert["securityVulnerability"]["advisory"]:
                    unsaved_vulnerability_ids = []
                    for identifier in alert["securityVulnerability"]["advisory"][
                        "identifiers"
                    ]:
                        if identifier.get("value"):
                            unsaved_vulnerability_ids.append(
                                identifier.get("value"),
                            )
                    if unsaved_vulnerability_ids:
                        finding.unsaved_vulnerability_ids = (
                            unsaved_vulnerability_ids
                        )
                if "cvss" in alert["securityVulnerability"]["advisory"]:
                    if (
                        "score"
                        in alert["securityVulnerability"]["advisory"]["cvss"]
                    ):
                        score = alert["securityVulnerability"]["advisory"]["cvss"][
                            "score"
                        ]
                        if score is not None:
                            finding.cvssv3_score = score
                    if (
                        "vectorString"
                        in alert["securityVulnerability"]["advisory"]["cvss"]
                    ):
                        cvss_vector_string = alert["securityVulnerability"][
                            "advisory"
                        ]["cvss"]["vectorString"]
                        if cvss_vector_string is not None:
                            cvss_objects = cvss_parser.parse_cvss_from_text(
                                cvss_vector_string,
                            )
                            if len(cvss_objects) > 0:
                                finding.cvssv3 = cvss_objects[0].clean_vector()
                if (
                    "cwes" in alert["securityVulnerability"]["advisory"]
                    and "nodes"
                    in alert["securityVulnerability"]["advisory"]["cwes"]
                ):
                    cwe_nodes = alert["securityVulnerability"]["advisory"]["cwes"][
                        "nodes"
                    ]
                    if cwe_nodes and len(cwe_nodes) > 0:
                        finding.cwe = int(cwe_nodes[0].get("cweId")[4:])
                dupe_key = finding.unique_id_from_tool
                if dupe_key in dupes:
                    find = dupes[dupe_key]
                    find.nb_occurences += 1
                else:
                    dupes[dupe_key] = finding
            return list(dupes.values())
        elif isinstance(data, list):
            findings = []
            for vuln in data:
                url = vuln["url"]
                html_url = vuln["html_url"]
                if vuln["state"] == "open":
                    active = True
                else:
                    active = False
                ruleid = vuln["rule"]["id"]
                ruleseverity = vuln["rule"]["severity"]
                ruledescription = vuln["rule"]["description"]
                rulename = vuln["rule"]["name"]
                ruletags = vuln["rule"]["tags"]
                severity = vuln["rule"]["security_severity_level"]
                most_recent_instanceref = vuln["most_recent_instance"]["ref"]
                most_recent_instanceanalysis_key = vuln["most_recent_instance"]["analysis_key"]
                most_recent_instanceenvironment = vuln["most_recent_instance"]["environment"]
                most_recent_instancecategory = vuln["most_recent_instance"]["category"]
                most_recent_instancestate = vuln["most_recent_instance"]["state"]
                most_recent_instancecommit_sha = vuln["most_recent_instance"]["commit_sha"]
                most_recent_instancemessage = vuln["most_recent_instance"]["message"]["text"]
                location = vuln["most_recent_instance"]["location"]
                instancesurl = vuln["instances_url"]
                description = ruledescription + "\n"
                description += "**url:** " + url + "\n"
                description += "**html_url:** " + html_url + "\n"
                description += "**ruleid:** " + ruleid + "\n"
                description += "**ruleseverity:** " + ruleseverity + "\n"
                description += "**ruledescription:** " + ruledescription + "\n"
                description += "**rulename:** " + rulename + "\n"
                description += "**ruletags:** " + str(ruletags) + "\n"
                description += "**most_recent_instanceref:** " + most_recent_instanceref + "\n"
                description += "**most_recent_instanceanalysis_key:** " + most_recent_instanceanalysis_key + "\n"
                description += "**most_recent_instanceenvironment:** " + most_recent_instanceenvironment + "\n"
                description += "**most_recent_instancecategory:** " + most_recent_instancecategory + "\n"
                description += "**most_recent_instancestate:** " + most_recent_instancestate + "\n"
                description += "**most_recent_instancecommit_sha:** " + most_recent_instancecommit_sha + "\n"
                description += "**most_recent_instancemessage:** " + most_recent_instancemessage + "\n"
                description += "**location:** " + str(location) + "\n"
                description += "**instancesurl:** " + instancesurl + "\n"
                uniqueid = ruleid + url + most_recent_instanceanalysis_key + str(location)
                finding = Finding(
                    title=ruleid,
                    test=test,
                    description=description,
                    severity=severity.capitalize(),
                    active=active,
                    static_finding=True,
                    dynamic_finding=False,
                    unique_id_from_tool=uniqueid,
                )
                findings.append(finding)
            return findings

    def _search_vulnerability_alerts(self, data):
        if isinstance(data, list):
            for item in data:
                result = self._search_vulnerability_alerts(item)
                if result:
                    return result
        elif isinstance(data, dict):
            for key in data:
                if key == "vulnerabilityAlerts":
                    return data[key]
                result = self._search_vulnerability_alerts(data[key])
                if result:
                    return result
        return None

    def _convert_security(self, val):
        if val.lower() == "moderate":
            return "Medium"
        else:
            return val.title()
