from dojo.models import Test
from dojo.tools.harbor_vulnerability.parser import HarborVulnerabilityParser
from unittests.dojo_test_case import DojoTestCase


class TestHarborVulnerabilityParser(DojoTestCase):

    def test_parse_file_with_no_vuln_has_no_findings(self):
        with open("unittests/scans/harbor_vulnerability/harbor-0-vuln.json") as testfile:
            parser = HarborVulnerabilityParser()
            findings = parser.get_findings(testfile, Test())
            self.assertEqual(0, len(findings))

    # Sample with One Test
    # + also verify data with one test
    def test_parse_file_with_one_vuln_has_one_findings(self):
        with open("unittests/scans/harbor_vulnerability/harbor-1-vuln.json") as testfile:
            parser = HarborVulnerabilityParser()
            findings = parser.get_findings(testfile, Test())
            self.assertEqual(1, len(findings))

            finding = findings[0]
            self.assertEqual(finding.title, "CVE-YYYY-NNN - package (exploitable-version)")
            self.assertEqual(
                finding.description,
                "This is a sample description for sample description from Harbor API.",
            )
            self.assertEqual(finding.severity, "Info")
            self.assertEqual(finding.mitigation, "Upgrade package to version unexploitable-version")
            self.assertEqual(1, len(finding.unsaved_vulnerability_ids))
            self.assertEqual(finding.unsaved_vulnerability_ids[0], "CVE-YYYY-NNN")
            self.assertEqual(finding.component_name, "package")
            self.assertEqual(finding.component_version, "exploitable-version")
            self.assertEqual(finding.references, "https://github.com/goharbor/harbor\n")

    # Sample with Multiple Test
    def test_parse_file_with_multiple_vuln_has_multiple_findings(self):
        with open("unittests/scans/harbor_vulnerability/harbor-5-vuln.json") as testfile:
            parser = HarborVulnerabilityParser()
            findings = parser.get_findings(testfile, Test())
            self.assertEqual(5, len(findings))

            finding = findings[1]
            self.assertEqual(finding.severity, "High")
            self.assertIsNone(finding.mitigation)
            self.assertIsNone(finding.references)

    # Sample with Trivy Test
    def test_parse_file_with_multiple_vuln_has_multiple_trivy_findings(self):
        with open("unittests/scans/harbor_vulnerability/harbor-trivy-vuln.json") as testfile:
            parser = HarborVulnerabilityParser()
            findings = parser.get_findings(testfile, Test())

            finding = findings[0]
            self.assertEqual(finding.severity, "High")
            self.assertEqual(finding.cwe, "125")

    # Sample with harborapi pip
    def test_parse_file_with_multiple_vuln_has_harborapi_pip_package(self):
        with open("unittests/scans/harbor_vulnerability/harborapipip.json") as testfile:
            parser = HarborVulnerabilityParser()
            findings = parser.get_findings(testfile, Test())
            self.assertEqual(2, len(findings))

            finding = findings[0]
            self.assertEqual(finding.severity, "Medium")
            self.assertEqual(finding.cwe, "787")
