"""
Faraday Penetration Test IDE
Copyright (C) 2013  Infobyte LLC (http://www.infobytesec.com/)
See the file 'doc/LICENSE' for the license information

"""

import xml.etree.ElementTree as ET

from dateutil.parser import parse
from faraday_plugins.plugins.plugin import PluginXMLFormat
from faraday_plugins.plugins.plugins_utils import CVE_regex
from faraday_plugins.plugins.plugins_utils import get_severity_from_cvss

__author__ = "Blas"
__copyright__ = "Copyright (c) 2019, Infobyte LLC"
__credits__ = ["Blas", "Nicolas Rebagliati"]
__license__ = ""
__version__ = "1.0.0"
__maintainer__ = "Blas"
__email__ = "bmoyano@infobytesec.com"
__status__ = "Development"

from faraday_plugins.plugins.repo.nessus.DTO import ReportHost, Report, ReportItem

class NessusParser:
    """
    The objective of this class is to parse an xml file generated by the nessus tool.

    TODO: Handle errors.
    TODO: Test nessus output version. Handle what happens if the parser doesn't support it.
    TODO: Test cases.

    @param nessus_filepath A proper simple report generated by nessus
    """

    def __init__(self, output):
        self.tree = ET.fromstring(output)
        self.report = []
        if self.tree:
            self.report = self.__get_report()

    def __get_report(self) -> Report:
        report = self.tree.find('Report')
        return Report(report) if report else None


class NessusPlugin(PluginXMLFormat):
    """
    Example plugin to parse nessus output.
    """

    def __init__(self, *arg, **kwargs):
        super().__init__(*arg, **kwargs)
        self.extension = ".nessus"
        self.identifier_tag = "NessusClientData_v2"
        self.id = "Nessus"
        self.name = "Nessus XML Output Plugin"
        self.plugin_version = "0.0.1"
        self.version = "5.2.4"
        self.framework_version = "1.0.1"
        self.options = None

    @staticmethod
    def parse_compliance_data(data: dict):
        compliance_data = {}
        for key, value in data.items():
            if 'compliance-' in key:
                compliance_name = key.split("}")[-1]
                compliance_data[compliance_name] = value
        return compliance_data

    def map_properties(self, host: ReportHost):
        if self.hostname_resolution:
            name = host.host_properties.host_ip if host.host_properties.host_ip else host.name
        else:
            name = host.name
        hostnames = [host.host_properties.host_fqdn]
        if host.host_properties.host_rdns and host.host_properties.host_rdns not in hostnames:
            hostnames.append(host.host_properties.host_rdns)
        return {
            "name": name,
            "hostnames": hostnames,
            "mac": host.host_properties.mac_address,
            "os": host.host_properties.operating_system
        }

    @staticmethod
    def map_item(host_id, run_date, plugin_name, item: ReportItem) -> dict:
        data = item.plugin_output
        data += f'{item.exploit_available}'
        return {
            "host_id": host_id,
            "name": plugin_name,
            "severity": item.risk_factor,
            "data": data,
            "external_id": item.plugin_id_attr,
            "run_date": run_date,
            "desc": item.description,
            "resolution": item.solution,
            "ref": [],
        }

    def map_policy_general(self, kwargs, item: ReportItem):
        kwargs.update({"policyviolations": []})
        if item.plugin_family_attr == 'Policy Compliance':
            data = item.get_data()
            bis_benchmark_data = kwargs["desc"].split('\n')
            compliance_data = self.parse_compliance_data(data)
            compliance_info = compliance_data.get('compliance-info', '')
            if compliance_info and not kwargs["desc"]:
                kwargs["desc"] = compliance_info
            compliance_reference = compliance_data.get(
                'compliance-reference', '').replace('|', ':').split(',')
            compliance_result = compliance_data.get('compliance-result', '')
            for reference in compliance_reference:
                kwargs["ref"].append(reference)
            compliance_check_name = compliance_data.get('compliance-check-name', '')
            compliance_solution = compliance_data.get('compliance-solution', '')
            if compliance_solution and not kwargs["resolution"]:
                kwargs["resolution"] = compliance_solution
            policy_item = f'{compliance_check_name} - {compliance_result}'
            for policy_check_data in bis_benchmark_data:
                if 'ref.' in policy_check_data:
                    kwargs["ref"].append(policy_check_data)
            if 'compliance-see-also' in compliance_data:
                kwargs["ref"].append(compliance_data.get('compliance-see-also'))
            # We used this info from tenable: https://community.tenable.com/s/article/Compliance-checks-in-SecurityCenter
            kwargs["policyviolations"].append(policy_item)
            kwargs["name"] = f'{kwargs["name"]}: {policy_item}'

        return kwargs

    def parseOutputString(self, output):
        """
        This method will discard the output the shell sends, it will read it from
        the xml where it expects it to be present.

        NOTE: if 'debug' is true then it is being run from a test case and the
        output being sent is valid.
        """

        try:
            parser = NessusParser(output)
        except Exception as e:
            self.logger.error(str(e))
            return None
        if not parser.report:
            self.logger.error('Parser did not find any reports')
            return 1  # this value is used to exit the execution of the executor. Can't use False or None because if this function doesn't fail, it returns none
        report_hosts = parser.report.report_hosts
        if report_hosts:
            for host in report_hosts:
                run_date = host.host_properties.host_end
                if run_date:
                    run_date = parse(run_date)
                website = host.host_properties.host_fqdn
                host_id = self.createAndAddHost(**self.map_properties(host))

                for item in host.report_items:
                    vulnerability_name = item.plugin_name
                    if not vulnerability_name:
                        continue
                    item_name = item.svc_name_attr

                    main_data = self.map_item(
                        host_id, run_date, vulnerability_name, item)

                    main_data = self.map_add_ref(main_data, item)
                    if item_name == 'general':
                        main_data = self.map_policy_general(main_data, item)
                        self.createAndAddVulnToHost(**main_data)
                    else:
                        main_data["service_id"] = self.createAndAddServiceToHost(
                            host_id, name=item_name, protocol=item.protocol_attr,
                            ports=item.port_attr)
                        if item_name == 'www' or item_name == 'http':
                            main_data.update({"website": website})
                            self.createAndAddVulnWebToService(**main_data)
                        else:
                            self.createAndAddVulnToService(**main_data)

    @staticmethod
    def map_add_ref(main_data, item: ReportItem):
        main_data["cvss2"] = {}
        main_data["cvss3"] = {}
        
        found_cves = set()

        if item.cve:
            if isinstance(item.cve, str):
                cves_in_item_cve_field = CVE_regex.findall(item.cve)
                found_cves.update(cves_in_item_cve_field)
            elif isinstance(item.cve, list):
                for c_val in item.cve:
                    if isinstance(c_val, str):
                        found_cves.add(c_val)
        
        if item.see_also:
            main_data["ref"].append(item.see_also)
            # Extract CVEs from see_also string
            cves_from_see_also = CVE_regex.findall(item.see_also)
            found_cves.update(cves_from_see_also)

        if found_cves:
            main_data["cve"] = sorted(list(found_cves))
        
        if item.cpe:
            main_data["ref"].append(item.cpe)
        if item.cwe:
            main_data["cwe"] = item.cwe

        if item.cvss3_vector:
            cvss3_vector_string = item.cvss3_vector
            # Prepend "CVSS:3.0/" if no "CVSS:3.x" prefix exists, similar to Nessus SC plugin
            if not cvss3_vector_string.startswith("CVSS:3."):
                cvss3_vector_string = "CVSS:3.0/" + cvss3_vector_string
            main_data["cvss3"]["vector_string"] = cvss3_vector_string
        
        if item.cvss3_base_score:
            main_data["cvss3"]["base_score"] = item.cvss3_base_score
            main_data["severity"] = get_severity_from_cvss(item.cvss3_base_score)
        
        if item.cvss_vector:
            main_data["cvss2"]["vector_string"] = item.cvss_vector
        
        if item.cvss2_base_score:
            main_data["cvss2"]["base_score"] = item.cvss2_base_score
            # If CVSSv3 score is not present, use CVSSv2 score for severity
            if not item.cvss3_base_score:
                main_data["severity"] = get_severity_from_cvss(item.cvss2_base_score)
        
        return main_data


def createPlugin(*args, **kwargs):
    return NessusPlugin(*args, **kwargs)
