/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

/* (C) Marcin Kwadrans <quarter@users.sourceforge.net> */

#include "include/support.h"
#include "include/boardset.h"

/*! \brief Tłumaczenie nazw zakładek

	Tłumaczy nazwę zakładki zawartej w *title.
	Funkcja zapisuje w title wskaźnik do przetłumaczonego
	komunikatu lub nie modyfikuje wskaźnika jeśli komunikat
	nie może być stłumaczony 
	\param title Wskaźnik do nazwy zakładki do przetłumaczenia
*/
static void try_translate_title (gchar **title)
{
gchar *newtitle = NULL;
	
	if (!g_ascii_strcasecmp (*title, "Icons"))
		newtitle = g_strdup (_("Icons"));
	else
	if (!g_ascii_strcasecmp (*title, "House"))
		newtitle = g_strdup (_("House"));
	else
	if (!g_ascii_strcasecmp (*title, "Blackboard"))
		newtitle = g_strdup (_("Blackboard"));
	else
	if (!g_ascii_strcasecmp (*title, "Digits"))
		newtitle = g_strdup (_("Digits"));
	else
	if (!g_ascii_strcasecmp (*title, "Alphabet"))
		newtitle = g_strdup (_("Alphabet"));
	else
	if (!g_ascii_strcasecmp (*title, "Commands"))
		newtitle = g_strdup (_("Commands"));
	else
	if (!g_ascii_strcasecmp (*title, "Wizard"))
		newtitle = g_strdup (_("Wizard"));
	else
	if (!g_ascii_strcasecmp (*title, "Variables"))
		newtitle = g_strdup (_("Variables"));
	else
	if (!g_ascii_strcasecmp (*title, "Math"))
		newtitle = g_strdup (_("Math"));
	else
	if (!g_ascii_strcasecmp (*title, "Conditions and loops"))
		newtitle = g_strdup (_("Conditions and loops"));
	else
	if (!g_ascii_strcasecmp (*title, "Other"))
		newtitle = g_strdup (_("Other"));
	
	if (newtitle != NULL) {
		g_free (*title);
		*title = newtitle;
	}
}

/*! \brief Konstruktor zbioru plansz

	Tworzy zbiór plansz typu 
	\param a_type Typ plansz przechowywanych w zbiorze
*/
LWBoardSet::LWBoardSet  (LWType a_type): list_board(NULL), enable_grid(FALSE)
{
	type = a_type;
	
	widget = gtk_notebook_new ();
	gtk_widget_show (widget);
}

/*! \brief Destruktor

	Destruktor zbioru plansz. Rekurencyjnie niszczy również wszystkie
	plansze zawarte w zbiorze
*/
LWBoardSet::~LWBoardSet ()
{
	clear();
	gtk_widget_destroy (widget);
}

/*! \brief Odtwarza zawartość na podstawie drzewa XML

	Odtworzenie zawartości plansz na podstawie drzewa XML
	\param node Węzeł drzewa zawierający opis zbioru
*/
void LWBoardSet::restoreFromXML (xmlNode *node)
{
	g_return_if_fail (!xmlStrcasecmp (node->name, (xmlChar *) "BoardSet"));
	
	clear();
	
	for (xmlNode *n=node->children; n != NULL; n=n->next) {
		LWBoard *board = new LWBoard (this);
		gchar *title = (gchar *) xmlGetProp (node, (xmlChar *) "title");
		try_translate_title (&title);
		addBoard (board, title);
		g_free (title);
		board->restoreFromXML (n);
	}
}

/*! \brief Dadanie planszy do zbioru

	Dodaje planszę do zbioru nadając jej etykietę
	\param board Plansza
	\param title Etykieta planszy
*/
void LWBoardSet::addBoard (LWBoard *board, gchar *title)
{
	g_return_if_fail (board != NULL);
	g_return_if_fail (title != NULL);
	
	GtkWidget *w = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (w),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport (
			GTK_SCROLLED_WINDOW (w), board->getWidget());
	gtk_widget_show (w);
	
	gtk_notebook_append_page (GTK_NOTEBOOK (widget), w,
									gtk_label_new (title));
	
	list_board = g_slist_append (list_board, board);
}

/*! \brief Zmiana etykiety

	Zmienia etykietę planszy dodanej do zbioru
	\param borad Plansza
	\param title Etykieta planszy
*/
void LWBoardSet::changeBoardTitle (LWBoard *board, gchar *title)
{
	g_return_if_fail (board != NULL);
	g_return_if_fail (title != NULL);
	
	GtkWidget *w = gtk_widget_get_ancestor (board->getWidget(),
		GTK_TYPE_SCROLLED_WINDOW);
	
	g_return_if_fail (w != NULL);
	
	try_translate_title (&title);
	
	gtk_notebook_set_tab_label_text (GTK_NOTEBOOK (widget), w, title);
	
	g_free (title);
}

/*! \brief Czyści zbiór

	Niszczy wszystkie plansze wchodzące w skład zbioru
*/
void LWBoardSet::clear ()
{
	for (GSList *l = list_board; l != NULL; l = l->next) {
		LWBoard *board = (LWBoard *) l->data;
		delete board;
	}

	g_slist_free (list_board);
	list_board = NULL;
}

/*! \brief Włączanie siatki

	Włącza siatkę na zbiorze plansz
	\param enable Jeśli wartość TRUE siatka zostanie włączona,
	w przeciwnym razie będzie wyłączona
*/
void LWBoardSet::enableGrid (gboolean enable)
{
	enable_grid = enable;
	
	for (GSList *l = list_board; l != NULL; l = l->next) {
		LWBoard *board = (LWBoard *) l->data;
		board->enableGrid (enable);
	}
}

/*! \brief Sprawdzanie czy siatka jest włączona

	Sprawdza czy siatka jest włączona.
	\return Zwracana jest prawda jeśli siatka jest włączona, w przeciwnym
	razie fałsz.
*/
gboolean LWBoardSet::isGridEnabled ()
{
	return enable_grid;
}

/*! \brief Zwraca kontrolkę

	Zwraca kontrolkę zbioru
	\return Zwrócona kontrolka
*/

GtkWidget *LWBoardSet::getWidget ()
{
	return widget;
}

/*! \brief Zwraca typ plansz

	Zwraca typ plansz przechowywanych w zbiorze
	\return Zwrócony typ
*/
LWType LWBoardSet::getType()
{
	return type;
}
