#--- podExamples.t ------------------------------------------------------------
# function: Test HTML::ToC.  In particular test the examples as described in
#           the POD documentation.

use strict;
use Test::More tests => 16;
use Test::Differences;

use HTML::Toc;
use HTML::TocGenerator;
use HTML::TocInsertor;
use HTML::TocUpdator;

my ($filename, $filename2);


BEGIN {
      # Create test file
   $filename = "tmp.htm";
   die "$filename is already there" if -e $filename;
      # Create test file 2
   $filename2 = "tmp2.htm";
   die "$filename2 is already there" if -e $filename2;
}


END {
      # Remove test file
   unlink($filename) or warn "Can't unlink $filename: $!";
      # Remove test file 2
   unlink($filename2) or warn "Can't unlink $filename2: $!";
}


#--- TestExtendFromFile() --------------------------------------------------
# function: Test HTML::TocGenerator->extendFromFile()

sub TestExtendFromFile {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<body>
   <h1>Chapter of document 1</h1>
</body>
EOT

      # Assemble test file 2
   open(FILE, ">$filename2") || die "Can't create $filename2: $!";
   print FILE <<'EOT'; close(FILE);
<body>
   <h1>Chapter of document 2</h1>
</body>
EOT

      # Create objects
   my $toc          = HTML::Toc->new();
   my $tocGenerator = HTML::TocGenerator->new();

      # Set ToC options
   $toc->setOptions({'doLinkToFile' => 1});
      # Generate ToC
   $tocGenerator->generateFromFile($toc, $filename);
   $tocGenerator->extendFromFile($toc, $filename2);
      # Test ToC
   eq_or_diff($toc->format(), <<EOT, 'Test HTML::TocGenerator->extendFromFile()', {max_width=>120});

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="tmp.htm#h-1">Chapter of document 1</a></li>
   <li><a href="tmp2.htm#h-2">Chapter of document 2</a></li>
</ul>
<!-- End of generated Table of Contents -->
EOT
}  # TestExtendFromFile()


#--- TestGenerateFromFiles() --------------------------------------------------
# function: Test HTML::TocGenerator->generateFromFile()

sub TestGenerateFromFiles {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<body>
   <h1>Chapter of document 1</h1>
</body>
EOT

      # Assemble test file 2
   open(FILE, ">$filename2") || die "Can't create $filename2: $!";
   print FILE <<'EOT'; close(FILE);
<body>
   <h1>Chapter of document 2</h1>
</body>
EOT

      # Create objects
   my $toc          = HTML::Toc->new();
   my $tocGenerator = HTML::TocGenerator->new();

      # Set ToC options
   $toc->setOptions({'doLinkToFile' => 1});
      # Generate ToC
   $tocGenerator->generateFromFile($toc, [$filename, $filename2]);
      # Test ToC
   eq_or_diff($toc->format(), <<EOT, 'Test HTML::TocGenerator->generateFromFile()', {max_width=>120});

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="tmp.htm#h-1">Chapter of document 1</a></li>
   <li><a href="tmp2.htm#h-2">Chapter of document 2</a></li>
</ul>
<!-- End of generated Table of Contents -->
EOT
}  # TestGenerateFromFiles()


#--- TestGenerateFromFile() --------------------------------------------------
# function: Test HTML::TocGenerator->generateFromFile() using multiple files.

sub TestGenerateFromFile {
      # Assemble test file 1
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<html>
<body>
   <h1>Chapter</h1>
</body>
</html>
EOT

      # Create objects
   my $toc          = HTML::Toc->new();
   my $tocGenerator = HTML::TocGenerator->new();

      # Generate ToC
   $tocGenerator->generateFromFile($toc, $filename);
      # Test ToC
   eq_or_diff($toc->format(), <<EOT, 'Test HTML::TocGenerator->generateFromFile() using multiple files', {max_width=>120});

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#h-1">Chapter</a></li>
</ul>
<!-- End of generated Table of Contents -->
EOT
}  # TestGenerateFromFile()


#--- TestInsertIntoFile() -----------------------------------------------------
# function: Test HTML::TocInsertor->insertIntoFile()

sub TestInsertIntoFile {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<html>
<body>
   <h1>Chapter</h1>
</body>
</html>
EOT

      # Create objects
   my $toc         = HTML::Toc->new();
   my $tocInsertor = HTML::TocInsertor->new();
   my $output;

      # Generate ToC
   $tocInsertor->insertIntoFile($toc, $filename, {'output' => \$output});
      # Test ToC
   eq_or_diff($output, <<EOT, 'Test HTML::TocInsertor->insertIntoFile()', {max_width=>120});
<html>
<body>
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#h-1">Chapter</a></li>
</ul>
<!-- End of generated Table of Contents -->

   <h1><a name="h-1"></a>Chapter</h1>
</body>
</html>
EOT
}  # TestInsertIntoFile()


#--- TestInsertIntoFileUsingTocUpdator() --------------------------------------
# function: Test HTML::TocUpdator->insertIntoFile()

sub TestInsertIntoFileUsingTocUpdator {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<html>
<body>
   <h1>
   Chapter
   </h1>
</body>
</html>
EOT

      # Create objects
   my $toc        = HTML::Toc->new();
   my $tocUpdator = HTML::TocUpdator->new();
   my $output;

      # Generate ToC
   $tocUpdator->insertIntoFile($toc, $filename, {'output' => \$output});
      # Test ToC
   eq_or_diff($output, <<EOT, 'Test HTML::TocUpdator->insertIntoFile()', {max_width=>120});
<html>
<body><!-- #BeginToc -->
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#h-1"> Chapter </a></li>
</ul>
<!-- End of generated Table of Contents -->
<!-- #EndToc -->
   <h1><!-- #BeginTocAnchorNameBegin --><a name="h-1"></a><!-- #EndTocAnchorNameBegin -->
   Chapter
   </h1>
</body>
</html>
EOT
}  # TestInsertIntoFileUsingTocUpdator()


#--- TestUpdateFileUsingTocUpdator() -----------------------------------
# function: Test HTML::TocUpdator->updateFile()

sub TestUpdateFileUsingTocUpdator {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<html>
<body><!-- #BeginToc -->
foo
<!-- #EndToc -->
   <h1><!-- #BeginTocAnchorNameBegin -->bar<!-- #EndTocAnchorNameBegin -->
   Chapter
   </h1>
</body>
</html>
EOT

      # Create objects
   my $toc        = HTML::Toc->new();
   my $tocUpdator = HTML::TocUpdator->new();
   my $output;

      # Generate ToC
   $tocUpdator->updateFile($toc, $filename, {'output' => \$output});
      # Test ToC
   eq_or_diff($output, <<EOT, 'Test HTML::TocUpdator->updateFile()', {max_width=>120});
<html>
<body><!-- #BeginToc -->
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#h-1"> Chapter </a></li>
</ul>
<!-- End of generated Table of Contents -->
<!-- #EndToc -->
   <h1><!-- #BeginTocAnchorNameBegin --><a name="h-1"></a><!-- #EndTocAnchorNameBegin -->
   Chapter
   </h1>
</body>
</html>
EOT
}  # TestInsertIntoFileUsingTocUpdator()


#--- TestUsingAttributeValueAsTocText() -----------------------------------
# function: Test HTML::TocInsertor->insertIntoFile()

sub TestUsingAttributeValueAsTocText {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<body>
   <img src="test1.gif" alt="First picture">
   <img src="test2.gif" alt="Second picture">
</body>
EOT

      # Create objects
   my $toc         = HTML::Toc->new();
   my $tocInsertor = HTML::TocInsertor->new();
   my $output;

   $toc->setOptions({
      'tokenToToc'   => [{
         'groupId'    => 'image',
         'tokenBegin' => '<img alt=@>'
      }],
   });

      # Generate ToC
   $tocInsertor->insertIntoFile($toc, $filename, {'output' => \$output});
      # Test ToC
   eq_or_diff($output, <<EOT, 'TestUsingAttributeValueAsTocText', {max_width=>120});
<body>
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#image-1">First picture</a></li>
   <li><a href="#image-2">Second picture</a></li>
</ul>
<!-- End of generated Table of Contents -->

   <a name="image-1"></a><img src="test1.gif" alt="First picture">
   <a name="image-2"></a><img src="test2.gif" alt="Second picture">
</body>
EOT
}  # TestUsingAttributeValueAsTocText()


#--- TestGlobalGroups0() ------------------------------------------------------
# function: Test 'HTML::TocGenerator' option 'doUseGroupsGlobal = 0'.

sub TestGlobalGroups0 {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<h1>Chapter</h1>
<h2>Paragraph</h2>
EOT

      # Create objects
   my $toc1         = HTML::Toc->new();
   my $toc2         = HTML::Toc->new();
   my $tocGenerator = HTML::TocGenerator->new();

      # Set options
   $toc1->setOptions({
      'header'      => '',
      'footer'      => '',
      'tokenToToc' => [{'tokenBegin' => '<h1>'}]
   });
   $toc2->setOptions({
      'header'      => '',
      'footer'      => '',
      'tokenToToc' => [{'tokenBegin' => '<h2>'}]
   });
      # Generate ToC
   $tocGenerator->generateFromFile([$toc1, $toc2], $filename);
      # Test ToC
   eq_or_diff($toc1->format() . $toc2->format() . "\n", <<'EOT', "Test 'HTML::TocGenerator' option 'doUseGroupsGlobal = 0'", {max_width=>120});
<ul>
   <li><a href="#h-1">Chapter</a></li>
</ul><ul>
   <li><a href="#h-1">Paragraph</a></li>
</ul>
EOT
}  # TestGlobalGroups0()


#--- TestGlobalGroups1() ------------------------------------------------------
# function: Test 'HTML::TocGenerator' option 'doUseGroupsGlobal = 1'.

sub TestGlobalGroups1 {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT';
<h1>Chapter</h1>
<h2>Paragraph</h2>
EOT
   close(FILE);

      # Create objects
   my $toc1         = HTML::Toc->new();
   my $toc2         = HTML::Toc->new();
   my $tocGenerator = HTML::TocGenerator->new();

      # Set options
   $toc1->setOptions({
      'header'      => '',
      'footer'      => '',
      'tokenToToc' => [{'tokenBegin' => '<h1>'}]
   });
   $toc2->setOptions({
      'header'      => '',
      'footer'      => '',
      'tokenToToc' => [{'tokenBegin' => '<h2>'}]
   });
      # Generate ToC
   $tocGenerator->generateFromFile(
      [$toc1, $toc2], $filename, {'doUseGroupsGlobal' => 1}
   );
      # Test ToC
   eq_or_diff($toc1->format() . $toc2->format() . "\n", <<'EOT', "Test 'HTML::TocGenerator' option 'doUseGroupsGlobal = 1'", {max_width=>120});
<ul>
   <li><a href="#h-1">Chapter</a></li>
</ul><ul>
   <li><a href="#h-2">Paragraph</a></li>
</ul>
EOT
}  # TestGlobalGroups1()


#--- TestMultipleGroupsAppendix() ---------------------------------------------
# function: Test multiple ToCs

sub TestMultipleGroupsAppendix() {
      # Create objects
   my $toc         = HTML::Toc->new();
   my $tocInsertor = HTML::TocInsertor->new();
   my $output;

      # Set ToC options
   $toc->setOptions({
      'tokenToToc' => [{
            'tokenBegin' => '<h1 class="-appendix">'
         }, {
            'tokenBegin' => '<h2 class="-appendix">',
            'level'      => 2
         }, {
            'groupId'    => 'appendix',
            'tokenBegin' => '<h1 class="appendix">',
         }, {
            'groupId'    => 'appendix',
            'tokenBegin' => '<h2 class="appendix">',
            'level'      => 2
         }],
   });
      # Generate ToC
   $tocInsertor->insert($toc, <<EOT, {'output' => \$output});
<body>
   <h1>Chapter</h1>
   <h2>Paragraph</h2>
   <h3>Subparagraph</h3>
   <h1>Chapter</h1>
   <h1 class="appendix">Appendix Chapter</h1>
   <h2 class="appendix">Appendix Paragraph</h2>
</body>
EOT
      # Test ToC
   eq_or_diff($output, <<HTML, 'TestMultipleGroupsAppendix', {max_width=>120});
<body>
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#h-1">Chapter</a>
      <ul>
         <li><a href="#h-1.1">Paragraph</a></li>
      </ul>
   </li>
   <li><a href="#h-2">Chapter</a></li>
</ul>
<ul>
   <li><a href="#appendix-1">Appendix Chapter</a>
      <ul>
         <li><a href="#appendix-1.1">Appendix Paragraph</a></li>
      </ul>
   </li>
</ul>
<!-- End of generated Table of Contents -->

   <h1><a name="h-1"></a>Chapter</h1>
   <h2><a name="h-1.1"></a>Paragraph</h2>
   <h3>Subparagraph</h3>
   <h1><a name="h-2"></a>Chapter</h1>
   <h1 class="appendix"><a name="appendix-1"></a>Appendix Chapter</h1>
   <h2 class="appendix"><a name="appendix-1.1"></a>Appendix Paragraph</h2>
</body>
HTML
}  # TestMultipleGroupsAppendix()


#--- TestMultipleGroupsPart() -------------------------------------------------
# function: Test multiple ToCs

sub TestMultipleGroupsPart() {
      # Create objects
   my $toc         = HTML::Toc->new();
   my $tocInsertor = HTML::TocInsertor->new();
   my $output;

      # Set ToC options
   $toc->setOptions({
      'tokenToToc' => [{
            'tokenBegin' => '<h1 class="-part">'
         }, {
            'tokenBegin' => '<h2 class="-part">',
            'level'      => 2,
         }, {
            'groupId'        => 'part',
            'tokenBegin'     => '<h1 class="part">',
            'level'          => 1,
            'doNumberToken'  => 1,
            'numberingStyle' => 'upper-alpha'
         }]
   });
      # Generate ToC
   $tocInsertor->insert($toc, <<HTML, {'output' => \$output});
<body>
   <h1 class="part">First Part</h1>
   <h1>Chapter</h1>
   <h2>Paragraph</h2>
   <h1 class="part">Second Part</h1>
   <h1>Chapter</h1>
   <h2>Paragraph</h2>
</body>
HTML
      # Test ToC
   eq_or_diff($output, <<HTML, 'TestMultipleGroupsPart', {max_width=>120});
<body>
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#part-A">First Part</a></li>
</ul>
<ul>
   <li><a href="#h-1">Chapter</a>
      <ul>
         <li><a href="#h-1.1">Paragraph</a></li>
      </ul>
   </li>
</ul>
<ul>
   <li><a href="#part-B">Second Part</a></li>
</ul>
<ul>
   <li><a href="#h-2">Chapter</a>
      <ul>
         <li><a href="#h-2.1">Paragraph</a></li>
      </ul>
   </li>
</ul>
<!-- End of generated Table of Contents -->

   <h1 class="part"><a name="part-A"></a>A &nbsp;First Part</h1>
   <h1><a name="h-1"></a>Chapter</h1>
   <h2><a name="h-1.1"></a>Paragraph</h2>
   <h1 class="part"><a name="part-B"></a>B &nbsp;Second Part</h1>
   <h1><a name="h-2"></a>Chapter</h1>
   <h2><a name="h-2.1"></a>Paragraph</h2>
</body>
HTML
}  # TestMultipleGroupsPart()


#--- TestMultipleTocs() -------------------------------------------------------
# function: Test multiple ToCs

sub TestMultipleTocs() {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<body>
   <h1>Header One</h1>
   <img src="test1.gif" alt="First picture"/>
   <h2>Paragraph One</h2>
   <img src="test2.gif" alt="Second picture"/>
</body>
EOT

      # Create objects
   my $toc1        = HTML::Toc->new();
   my $toc2        = HTML::Toc->new();
   my $tocInsertor = HTML::TocInsertor->new();
   my $output;

      # Set ToC options
   $toc2->setOptions({
      'tokenToToc'             => [{
         'groupId'    => 'image',
         'tokenBegin' => '<img alt=@>'
      }],
   });
      # Generate ToC
   $tocInsertor->insertIntoFile(
      [$toc1, $toc2], $filename, {'output' => \$output}
   );
      # Test ToC
   eq_or_diff($output, <<HTML, 'Test multiple ToCs', {max_width=>120});
<body>
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#h-1">Header One</a>
      <ul>
         <li><a href="#h-1.1">Paragraph One</a></li>
      </ul>
   </li>
</ul>
<!-- End of generated Table of Contents -->

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#image-1">First picture</a></li>
   <li><a href="#image-2">Second picture</a></li>
</ul>
<!-- End of generated Table of Contents -->

   <h1><a name="h-1"></a>Header One</h1>
   <a name="image-1"></a><img src="test1.gif" alt="First picture"/>
   <h2><a name="h-1.1"></a>Paragraph One</h2>
   <a name="image-2"></a><img src="test2.gif" alt="Second picture"/>
</body>
HTML
}  # TestMultipleTocs()


#--- TestSpecifyNumberedList() ------------------------------------------------
# function: Test specifying numbered list.

sub TestSpecifyNumberedList {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<body>
   <h1>Chapter</h1>
   <h2>Paragraph</h2>
</body>
EOT

      # Create objects
   my $toc        = HTML::Toc->new();
   my $tocGenerator = HTML::TocGenerator->new();

      # Set ToC options
   $toc->setOptions({
      'templateLevelBegin' => '"<ol>\n"',
      'templateLevelEnd'   => '"</ol>\n"',
   });
      # Generate ToC
   $tocGenerator->generateFromFile($toc, $filename);
      # Test ToC
   eq_or_diff($toc->format(), <<HTML, 'Test specifying numbered list', {max_width=>120});

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ol>
   <li><a href="#h-1">Chapter</a>
      <ol>
         <li><a href="#h-1.1">Paragraph</a></li>
      </ol>
   </li>
</ol>
<!-- End of generated Table of Contents -->
HTML
}  # TestSpecifyNumberedList()


#--- TestNumberTocUsingGeneratedNumbers() ------------------------------------------------
# function: Test number ToC using generated numbers

sub TestNumberTocUsingGeneratedNumbers {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<body>
   <h1>Chapter</h1>
   <h2>Paragraph</h2>
</body>
EOT

      # Create objects
   my $toc        = HTML::Toc->new();
   my $tocGenerator = HTML::TocGenerator->new();

      # Set ToC options
   $toc->setOptions({
      'templateLevel' => '"<li>$node &nbsp;$text"',
   });
      # Generate ToC
   $tocGenerator->generateFromFile($toc, $filename);
      # Test ToC
   eq_or_diff($toc->format(), <<HTML, 'Test number ToC using generated numbers', {max_width=>120});

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li>1 &nbsp;<a href="#h-1">Chapter</a>
      <ul>
         <li>1.1 &nbsp;<a href="#h-1.1">Paragraph</a></li>
      </ul>
   </li>
</ul>
<!-- End of generated Table of Contents -->
HTML
}  # TestNumberTocUsingGeneratedNumbers()


#--- TestUpdateFile() ---------------------------------------------------------
# function: Test HTML::TocUpdator->updateFile()

sub TestUpdateFile {
      # Assemble test file
   open(FILE, ">$filename") || die "Can't create $filename: $!";
   print FILE <<'EOT'; close(FILE);
<html>
<body><!-- #BeginToc -->
foo
<!-- #EndToc -->
   <h1><!-- #BeginTocAnchorNameBegin -->bar<!-- #EndTocAnchorNameBegin -->
   Chapter
   <!-- #BeginTocAnchorNameEnd -->foo<!-- #EndTocAnchorNameEnd --></h1>
</body>
</html>
EOT

      # Create objects
   my $toc        = HTML::Toc->new();
   my $tocUpdator = HTML::TocUpdator->new();
   my $output;

      # Generate ToC
   $tocUpdator->updateFile($toc, $filename, {'output' => \$output});
      # Test ToC
   eq_or_diff($output, <<HTML, 'Test HTML::TocUpdator->updateFile()', {max_width=>120});
<html>
<body><!-- #BeginToc -->
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#h-1"> Chapter </a></li>
</ul>
<!-- End of generated Table of Contents -->
<!-- #EndToc -->
   <h1><!-- #BeginTocAnchorNameBegin --><a name="h-1"></a><!-- #EndTocAnchorNameBegin -->
   Chapter
   </h1>
</body>
</html>
HTML
}  # TestUpdateFile()


#--- TestUsingCSS() -----------------------------------------------------------
# function: Test multiple ToCs

sub TestUsingCSS() {

      # Create objects
   my $toc          = new HTML::Toc;
   my $tocInsertor  = new HTML::TocInsertor;
   my $output;

   $toc->setOptions({
      'templateLevelBegin'   => '"<ol class=\"toc_$groupId$level\">\n"',
      'templateLevelEnd'     => '"</ol>\n"',
      'doNumberToken'        => 1,
      'tokenToToc' => [{
         'groupId'        => 'appendix',
         'tokenBegin'     => '<h1>',
         'numberingStyle' => 'upper-alpha'
      }, {
         'groupId'        => 'appendix',
         'tokenBegin'     => '<h2>',
         'level'          => 2,
        }]
   });
   $tocInsertor->insert($toc, <<EOT);
<html>
<head>
   <style type="text/css">
      ol.toc_appendix1 { list-style-type: upper-alpha }
   </style>
</head>
<body>
   <h1>Appendix</h1>
   <h2>Appendix Paragraph</h2>
   <h1>Appendix</h1>
   <h2>Appendix Paragraph</h2>
</body>
</html>
EOT
      # Insert ToC
   $tocInsertor->insert($toc, <<EOT, {'output' => \$output});
<html>
<head>
   <style type="text/css">
      ol.toc_appendix1 { list-style-type: upper-alpha }
   </style>
</head>
<body>
   <h1>Appendix</h1>
   <h2>Appendix Paragraph</h2>
   <h1>Appendix</h1>
   <h2>Appendix Paragraph</h2>
</body>
</html>
EOT
      # Test ToC
   eq_or_diff($output, <<HTML, 'TestUsingCSS', {max_width=>120});
<html>
<head>
   <style type="text/css">
      ol.toc_appendix1 { list-style-type: upper-alpha }
   </style>
</head>
<body>
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ol class="toc_appendix1">
   <li><a href="#appendix-A">Appendix</a>
      <ol class="toc_appendix2">
         <li><a href="#appendix-A.1">Appendix Paragraph</a></li>
      </ol>
   </li>
   <li><a href="#appendix-B">Appendix</a>
      <ol class="toc_appendix2">
         <li><a href="#appendix-B.1">Appendix Paragraph</a></li>
      </ol>
   </li>
</ol>
<!-- End of generated Table of Contents -->

   <h1><a name="appendix-A"></a>A &nbsp;Appendix</h1>
   <h2><a name="appendix-A.1"></a>A.1 &nbsp;Appendix Paragraph</h2>
   <h1><a name="appendix-B"></a>B &nbsp;Appendix</h1>
   <h2><a name="appendix-B.1"></a>B.1 &nbsp;Appendix Paragraph</h2>
</body>
</html>
HTML
}  # TestUsingCSS()



   # 2. Test 'generateFromFile()'
TestGenerateFromFile();
   # 4. Test 'doUseGroupsGlobal = 0'
TestGlobalGroups0();
   # 5. Test 'doUseGroupsGlobal = 1'
TestGlobalGroups1();
   # 6. Test 'tocInsertor->insertIntoFile'
TestInsertIntoFile();
   # 7. Test 'tocUpdator->insertIntoFile'
TestInsertIntoFileUsingTocUpdator();
   # 8. Test 'tocUpdator->updateFile'
TestUpdateFileUsingTocUpdator();
   # 9. Test using attribute value as ToC text
TestUsingAttributeValueAsTocText();
   # 3. Test 'generateFromFiles()'
TestGenerateFromFiles();
   # 1. Test 'extendFromFile()'
TestExtendFromFile();
   # 10. Test multiple ToCs
TestMultipleTocs();
   # 8. Test additional 'appendix' group
TestMultipleGroupsAppendix();
   # 9. Test additional 'part' group
TestMultipleGroupsPart();
   # 11. Test specifying numbered list
TestSpecifyNumberedList();
   # 11. Test specifying numbered list
TestNumberTocUsingGeneratedNumbers();
   # 12. Test 'updateFile()'
TestUpdateFile();
   # 13. Test using CSS
TestUsingCSS();
