/*
*   Copyright (C) 2001 PROTOS Project Consortium, 2002 OUSPG
*     [http://www.ee.oulu.fi/research/ouspg/protos]
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License version 2
*   as published by  the Free Software Foundation
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

package FI.protos.ouspg.wrapper;

import FI.protos.ouspg.wrapper.BugCatZero;

import java.io.*;
import java.net.*;
import java.util.*;
import java.text.*;

/**
 * An udp zerocase Bug cat.
 */
public class SIPBugCat extends BugCatZero
{

  /*
   * Internal variables
   */
  /* Remote address set by host command line option. */
  private InetAddress addr = null;

  /* destination port to connect to */
  private int dport = 5060;

  /* local port */
  private int lport = 5060;
  
  /* UDP Socket */
  private DatagramSocket socket = null;
  
  /* send maximum PDU size 64kB - header#*/ 
  private int maxpdusize = 65507;

  /* time to wait for reply in milliseconds */
  private int replyWaitDelay = 100;

  /* to wait between test-cases in milliseconds */
  private int testCaseDelay = 100;

  /* dump received packets ? */
  private boolean showreply = false;

  /* dump sent packets ? */
  private boolean showsent = false;

  /* send validcase between cases ? */
  private boolean validcase = false;

  /* send CANCEL/ACK */
  private boolean teardown = false;

  /* Tag replacements key = from, value = to */
  private HashMap replacements = null;

  byte[] validcaseInvite = null;
  byte[] validcaseTeardown = null;

  NumberFormat numberFormatBranch = null;

  /**
   * Default constructor. 
   */
  public SIPBugCat() {
    /* Initialize replacements */
    replacements = new HashMap();
    
    /* Inital value of <From> */
    try {
      InetAddress localaddr = InetAddress.getLocalHost();
      replacements.put("From-Name", "user");
      replacements.put("From-Address", localaddr.getCanonicalHostName());
      replacements.put("From-IP", localaddr.getHostAddress());
      replacements.put("From", replacements.get("From-Name") + "@" + replacements.get("From-Address") );
    } catch (UnknownHostException e) {
      System.out.println ("ERROR: Unknown local host: " + e.getMessage());
      System.exit (1);
    }
    
    /* Initial value of <Call-ID> */
    replacements.put("Call-ID", "0");
    /* Initial value of <Call-ID> */
    replacements.put("CSeq", "1");
    /* Initial value of <Local-Port> */
    replacements.put("Local-Port", "" + lport);
    /* Initial value of <Branch-ID> */
    replacements.put("Branch-ID", "0");
    numberFormatBranch = new DecimalFormat("000000");    
  }
  
  /**
   * Stub for Command-line help. Activated by -h command line option
   */
  public void h() {
    this.help ();
  }

  /**
   * Replacement command-line option. Activated by -r command line option
   */
  public void r(String from, String to) {
    replacements.put(from, to);
  }

  /*
    *Command-line help. Activated by -help command line option
   */
  public void help () {
    System.out.println("Usage java -jar <jarfile>.jar [ [OPTIONS] |"+
		       " -touri <SIP-URI> ]\n");
    System.out.println
      ("  -touri  <addr>        Recipient of the request\n" +
       "                        Example: <addr> : you@there.com");
    System.out.println
      ("  -fromuri <addr>       Initiator of the request\n"+ 
       "                        Default: " + replacements.get("From-Name") +
                                  "@" + replacements.get("From-Address"));
  
    System.out.println
      ("  -sendto <domain>      Send packets to <domain> instead of\n" +
       "                        domainname of -touri");
    System.out.println
      ("  -callid <callid>      Call id to start test-case call ids from\n" +
       "                        Default: "+replacements.get("Call-ID"));
    System.out.println
      ("  -dport <port>         Portnumber to send packets on host.\n" +
       "                        Default: "+dport);
    System.out.println
      ("  -lport <port>         Local portnumber to send packets from\n" +
       "                        Default: "+lport);
    System.out.println
      ("  -delay <ms>           Time to wait before sending new test-case\n"+ 
       "                        Defaults to "+testCaseDelay+" ms (milliseconds)");
    System.out.println
      ("  -replywait <ms>       Maximum time to wait for host to reply\n" +
       "                        Defaults to "+replyWaitDelay+" ms (milliseconds)");
    System.out.println
      ("  -file <file>          Send file <file> instead of test-case(s)");
    System.out.println
      ("  -help                 Display this help");
    System.out.println
      ("  -jarfile <file>       Get data from an alternate bugcat\n"+
       "                        JAR-file <file>");
    System.out.println 
      ("  -showreply            Show received packets");
    System.out.println 
      ("  -showsent             Show sent packets");
    System.out.println 
      ("  -teardown             Send CANCEL/ACK");
    System.out.println
      ("  -single <index>       Inject a single test-case <index>");
    System.out.println
      ("  -start <index>        Inject test-cases starting from <index>");
    System.out.println
      ("  -stop <index>         Stop test-case injection to <index>");
    System.out.println
      ("  -maxpdusize <int>     Maximum PDU size\n" + 
       "                        Default to "+maxpdusize+" bytes");
    System.out.println
      ("  -validcase            Send valid case (case #0) after each\n"+
       "                        test-case and wait for a response. May\n"+
       "                        be used to check if the target is still\n"+
       "                        responding. Default: off");
    System.out.print ("\n");
    System.exit(0);
  }
  
  /**
   * Set receiver by command line option '-touri'.
   * @param s The URI.
   */
  public void touri(String s) {
    try {
      int index = s.indexOf("@");
      if(index == -1) {
	System.out.println ("ERROR: Invalid to-URI!");
	System.exit (1);
      }

      replacements.put("To",s);

      String host = s.substring(index+1);
      String name = s.substring(0, index);

      replacements.put("To-Name", name);
      replacements.put("To-Address", host);

      addr = InetAddress.getByName(host);

      replacements.put("To-IP", addr.getHostAddress());    
    } catch (UnknownHostException e) {
      System.out.println ("ERROR: Unknown to-URI hostname: " + e.getMessage());
      System.exit (1);
    }
  }

  /**
   * Set to s instead of domainname of '-touri'.
   * @param s The address to be sent.
   */
  public void sendto(String s) {
    try {
      addr = InetAddress.getByName(s);
      replacements.put("Send-To", addr.getHostAddress());    
    } catch (UnknownHostException e) {
      System.out.println ("ERROR: Unknown sendto host: " + e.getMessage());
      System.exit (1);
    }
  }

  /**
   * Set fromuri by command line option '-fromuri'.
   * @param s The URI
   */
  public void fromuri(String s) {
    try {
      int index = s.indexOf("@");
      if(index < 0) {
	System.out.println ("ERROR: Invalid from-URI!");
	System.exit (1);
      }

      replacements.put("From", s);

      String host = s.substring(index+1);
      String name = s.substring(0, index);

      replacements.put("From-Name", name);
      replacements.put("From-Address", host);

      InetAddress sender_addr = InetAddress.getByName(host);

      replacements.put("From-IP", sender_addr.getHostAddress());    
    } catch (UnknownHostException e) {
      System.out.println ("ERROR: Unknown sender host: " + 
			  e.getMessage());
      System.exit (1);
    }
  }

  /**
   * Set the seed for call-ids by command line option -callid
   * @param s The call-id starting value.
   */
  public void callid(String s) {
    try {
      int iCallID = new Integer (s).intValue();
      replacements.put("Call-ID", "" + iCallID);
    
    } catch (NumberFormatException e) {
      System.out.println("ERROR: Invalid callid value: " + e.getMessage());
      System.exit (0);
    }
  }

  
  /**
   * Set destination port number by command line option '-dport'.
   * @param s The port number.
   */
  public void dport (String s) {
    try {
      dport = new Integer (s).intValue();
    } catch (NumberFormatException e) {
      System.out.println("ERROR: Invalid destination port number: "
			 + e.getMessage());
      System.exit (0);
    }
  }

  /**
   * Set local port number by command line option '-lport'.
   * @param s The port number.
   */
  public void lport (String s) {
    try {
      lport = Integer.parseInt(s);
    } catch (NumberFormatException e) {
      System.out.println("ERROR: Invalid local port number: " 
			+ e.getMessage());
      System.exit (0);
    }
    replacements.put("Local-Port", "" + lport);
  }
  
  /**
   * Set testCaseDelay by command line option '-delay'.
   */
  public void delay(String s) {
    try {
      testCaseDelay = new Integer (s).intValue();
    } catch (NumberFormatException e) {
      System.out.println("ERROR: Invalid injection delay: " 
			 + e.getMessage());
      System.exit (0);
    }
  }

  /**
   * Set replyWaitDelay delay by command line option '-replywait'.
   */
  public void replywait(String s) {
    try {
      replyWaitDelay = new Integer (s).intValue();
    } catch (NumberFormatException e) {
      System.out.println("ERROR: Invalid replywait: " 
			 + e.getMessage());
      System.exit (0);
    }
  }


  /**
   * Dump replies. Enabled by command line option '-showreply'.
   */
  public void showreply() {
    showreply = true;
  }

  /**
   * Dump sent packets. Enabled by command line option '-showsent'.
   */
  public void showsent() {
    showsent = true;
  }

  /**
   * Teardown connectin after INVITE. 
   * Enabled by command line option '-teardown'.
   */
  public void teardown() {
    teardown = true;
  }

  /**
   * Limits the maximum PDU size
   * @param maxpdusize maximum PDU size in bytes.
   */
  public void maxpdusize(String s) throws IllegalArgumentException {
    try {
      maxpdusize = new Integer (s).intValue();
    } catch (NumberFormatException e) {
      System.out.println("ERROR: Invalid maxpdusize " + e.getMessage());
      System.exit (0);
    }
 
  }

  /**
   * Send valid case between test-cases. 
   * Enabled by command line option '-validcase'.
   */
  public void validcase() {
    validcase = true;
  }

  /**
   * Main routine, parses command-line arguments and starts injection.
   * @param args Command-line arguments.
   */
  public static void main(String args[]) {
    SIPBugCat cat = new SIPBugCat();
    cat.run(args);
  }
    
  /**
   * Method redefined from super class
   */
  public void prepare() throws IOException {
    try {
      //Create socket and open on port lport
      socket = new DatagramSocket(lport);
    } catch (SocketException e) {
      System.out.println("Error while creating DatagramSocket!!");
      System.exit (1);
    }
   
    /* Check that some hostname is given. Otherwise give help */
    if (!(replacements.containsKey("To"))) {
      this.help();
      System.exit(0);
    }
  }

  /**
   * Method redefined from super class
   */  
  public void finish() throws IOException {
    socket.close();
    socket = null;

  }
  
  /**
   * Method redefined from super class
   */
  public void setZeroCase (byte[] b) {
    byte[][] tmp = parseTestCase(b);
    validcaseInvite = tmp[0];
    validcaseTeardown = tmp[1];
    
    /* check if subject and well doing well
    */
    if (validcase & startIndex != 0) {
      sendValidCase(0);
    }
  }

  /**
   * Method redefined from super class
   */
  public void inject(int index, byte[] metaData, byte[] data)
    throws IOException {
    
    // Extract INVITE / Teardown
    byte[][] pdus = parseTestCase(data);
    byte[] testCaseInvite = pdus[0];
    byte[] testCaseTeardown = pdus[1];

    /* And send it */
    System.out.println("Sending Test-Case #" + index);
    send(testCaseInvite,index);
    
    if (teardown) {
      showReply(replyWaitDelay);
      System.out.println(" Sending CANCEL");
      replacements.put("Teardown-Method","CANCEL");
      send(testCaseTeardown,index);
      showReply(replyWaitDelay);
      
      System.out.println(" Sending ACK");
      replacements.put("Teardown-Method","ACK");
      send(testCaseTeardown,index);
      showReply(replyWaitDelay);
    }

    
    delay(testCaseDelay);
    
    replacements.put("Call-ID",  "" 
		     + (Integer.parseInt( (String) replacements.get("Call-ID")) + 1));
    	
    replacements.put("Branch-ID",  ""
		     + (numberFormatBranch.format(Integer.parseInt((String) replacements.get("Branch-ID")) + 1)));


    /* Validcase */
    if (validcase) {
      sendValidCase(index);
    }

  }
  
  /* 
   * Private methods
   */ 
  private void send(byte[] data,int index) 
  {
    /* Prepare for empty data */
    if (data == null) {
      System.out.println ("WARNING: no data (NULL) to send.");
    } else {
      try {
	if (replacements.containsKey("Send-To")) {
	  addr = InetAddress.getByName((String)replacements.get("Send-To"));
	}
      } catch (UnknownHostException e) {
	System.out.println ("ERROR: Unknown sendto host " + e.getMessage());
	System.exit (1);
      }
      try {
	data = replace(data);
	data = limit(data, maxpdusize);
	DatagramPacket dpsend = new DatagramPacket(data, data.length, addr, dport);
	socket.send(dpsend);
	System.out.println("    test-case #" +index+", "+data.length+" bytes");
    
	if (showsent) {
	  System.out.print(dump(data, 0, data.length));
	}

      } catch (Exception e) {
	e.printStackTrace();
	System.exit (-1);
      }
    }
  }

  private void sendValidCase(int index) {
    
    System.out.println("Sending valid-case");
        
    int tReplyWaitDelay = replyWaitDelay;
    if (tReplyWaitDelay == 0) {
      tReplyWaitDelay = 1;
    }

    send(validcaseInvite,index);  

    while (showReply(tReplyWaitDelay).size() == 0) {
      System.out.println ("    test-case #" +index + ": No reply to" +
			  " valid INVITE packet within "+tReplyWaitDelay+" ms."+
			  " Retrying...");
      replacements.put("Call-ID",  "" 
		       + (Integer.parseInt( (String) replacements.get("Call-ID")) + 1));
      replacements.put("Branch-ID",  ""
		       + (numberFormatBranch.format(Integer.parseInt((String) replacements.get("Branch-ID")) + 1)));
      send(validcaseInvite,index);
      if (tReplyWaitDelay < Integer.MAX_VALUE / 2)
	tReplyWaitDelay = tReplyWaitDelay * 2;	  
    } 
    
    if (teardown) { 
      showReply(replyWaitDelay);
      System.out.println(" Sending CANCEL");
      replacements.put("Teardown-Method","CANCEL");
      send(validcaseTeardown,index);
      
      showReply(replyWaitDelay);
      System.out.println(" Sending ACK");
      replacements.put("Teardown-Method","ACK");
      send(validcaseTeardown,index);
      showReply(replyWaitDelay);
    }
    replacements.put("Call-ID",  "" 
		     + (Integer.parseInt( (String) replacements.get("Call-ID")) + 1));
    replacements.put("Branch-ID",  ""
		     + (numberFormatBranch.format(Integer.parseInt((String) replacements.get("Branch-ID")) + 1)));
    delay(testCaseDelay);
  }
  
  /**
   * Waits waitfor ms for checks all 2ms for new arrived packets.
   * @param  waitfor Time to wait
   * @return Vector of Returncodes. Code -1 : No successful parsing of \
             returncode 
   */  
  private Vector showReply(int waitfor) {
    DatagramPacket dprecv = new DatagramPacket(new byte[0xffff], 0xffff);
    byte[] recv_data = null;
    int returnCode = -1;
    Vector returnCodes = new Vector();
    
    long expires = System.currentTimeMillis() + waitfor;

    while (expires > System.currentTimeMillis()) {
      try {
      /* Read reply (one datagram, one datagram only..) */
	socket.setSoTimeout(2);
	socket.receive(dprecv);
	recv_data = new String(dprecv.getData()).substring(0, dprecv.getLength()).getBytes();

	if (showreply) {
	  System.out.print(dump(recv_data, 0, recv_data.length));
	}
      } catch (SocketTimeoutException e) { 
	continue;
      } catch (Exception e) {
	e.printStackTrace();
	System.exit (-1);
      }
      
      try {
	// and parse the Return code
	StringTokenizer st = new StringTokenizer(new String(recv_data));
	String token = st.nextToken();
	if ( "SIP/2.0".compareTo(token) != 0) {
	  returnCodes.add(new Integer(-1));
	} 
	else {
	  token = st.nextToken();
	  returnCode = Integer.parseInt(token);
	}
      } catch (NumberFormatException e) {
	returnCode = -1;
	returnCodes.add(new Integer(returnCode));
      }
      System.out.println("   Received Returncode: " + returnCode);
      returnCodes.add(new Integer(returnCode));
    }
    return returnCodes;
  }

  /**
   * Finds all tags between < and > and replaces them if some
   * replacement value has been given. After initial replacements,
   * calculates the content length of the data and substitutes tag
   * <Content-Length> with the actual content length value.
   * @param data data
   */
  private byte[] replace(byte[] data) {
    /* Put the data into a string buffer, so we can do the replacements
       conveniently. */
    StringBuffer buffer = new StringBuffer(new String(data));
    try {
      /* Iterate through all the given replacements and replace them */
      Iterator iter = replacements.keySet().iterator();
      
      while(iter.hasNext()) {
	String key = (String)(iter.next());
	String value = (String)(replacements.get(key));
	String tag = "<"+key+">";
	int i = buffer.indexOf(tag);
	
	while(i >= 0) {	
	  buffer.replace(i, i+tag.length(), value);
	  i = buffer.indexOf(tag);
	}
      }   
    } catch(Exception e) {
      System.out.println ("ERROR: " + e.getMessage());
      System.exit (-1);
    }

    /* Calculate and replace the content length */
    if (new String(data).indexOf("<Content-Length>") != -1) {
      String value = "0";
      try {
	/* Find the CRLFCRLF */
	String invite = new String(buffer.toString());
	String crlfcrlf = new String("\r\n\r\n");
	int headerlength = invite.indexOf(crlfcrlf);
      
	/* If CRLFCRLF exists */ 
	if(headerlength >= 0) {
	  /* Calculate the total header size (including CRLFCRLF) */
	  headerlength += crlfcrlf.length();
	
	  /* content length is data's length minus header's length */
	  value = ""+(invite.length()-headerlength);
	} 
      
	String tag = "<Content-Length>";
      
	/* Replace all occurrences of the string <Content-Length> */
	int i = buffer.indexOf(tag);	  
	while(i >= 0) {	
	  buffer.replace(i, i+tag.length(), value);
	  i = buffer.indexOf(tag);
	}	
      
      } catch(Exception e) {
	System.out.println ("ERROR: " + e.getMessage());
	System.exit (-1);
      }
    }
    
    return buffer.toString().getBytes();
  }


  /* parseTestCase
     @param byte[] testCase - the testCase in the form {1..n}(len(PDU) " " PDU)
     @return Byte[][]  
  */
  private byte[][] parseTestCase(byte[] testCase) {
    ArrayList al = new ArrayList();
    StringBuffer sb = new StringBuffer(new String(testCase));
    char ch;
        
    while (sb.length()>0) {
      StringTokenizer st = new StringTokenizer(sb.toString());
      
      String sLength = st.nextToken();
      sb.delete(0,sLength.length()+1);
      int iLength = Integer.parseInt(sLength);
            
      String sTemp = sb.substring(0,iLength);
      sb.delete(0,sTemp.length());
      al.add(sTemp.getBytes());
    }
    return (byte[][])al.toArray(new byte[0][0]);
  }

    
  /* Limits the byte array size to the maximum of max_length bytes.
     Used to limit UDP packets to 0xffff bytes. */
  private byte[] limit(byte[] data, int max_length) {
    if(data.length >= max_length) {
      String data_string = new String(data);
      data = data_string.substring(0, max_length).getBytes();      
    }
    return data;
  }

  /* Apply Delay */
  private void delay (int t) {
    try {
      Thread.currentThread().sleep(t);
    } catch (Exception e) { 
      System.out.println ("ERROR: " + e.getMessage());
      System.exit (-1);
    }
  }
} /* end of class */

// Local variables:
// c-basic-offset: 2
// End:


