#  holidays
#  --------
#  A fast, efficient Python library for generating country, province and state
#  specific sets of holidays on the fly. It aims to make determining whether a
#  specific date is a holiday as fast and flexible as possible.
#
#  Authors: Vacanza Team and individual contributors (see CONTRIBUTORS file)
#           dr-prodigy <dr.prodigy.github@gmail.com> (c) 2017-2023
#           ryanss <ryanssdev@icloud.com> (c) 2014-2017
#  Website: https://github.com/vacanza/holidays
#  License: MIT (see LICENSE file)

from unittest import TestCase

from holidays.countries.guam import Guam
from tests.common import CommonCountryTests


class TestGU(CommonCountryTests, TestCase):
    @classmethod
    def setUpClass(cls):
        super().setUpClass(Guam)

    def test_guam_discovery_day(self):
        name = "Guam Discovery Day"
        self.assertHolidayName(
            name,
            "2020-03-02",
            "2021-03-01",
            "2022-03-07",
            "2023-03-06",
            "2024-03-04",
            "2025-03-03",
        )
        self.assertHolidayName(name, range(1970, self.end_year))
        self.assertNoHolidayName(name, range(self.start_year, 1970))

    def test_good_friday(self):
        name = "Good Friday"
        self.assertHolidayName(
            name,
            "2020-04-10",
            "2021-04-02",
            "2022-04-15",
            "2023-04-07",
            "2024-03-29",
            "2025-04-18",
        )
        self.assertHolidayName(name, self.full_range)

    def test_liberation_day_guam(self):
        name = "Liberation Day (Guam)"
        self.assertHolidayName(name, (f"{year}-07-21" for year in range(1945, self.end_year)))
        self.assertNoHolidayName(name, range(self.start_year, 1945))

    def test_all_souls_day(self):
        self.assertHolidayName("All Souls' Day", (f"{year}-11-02" for year in self.full_range))

    def test_lady_of_camarin_day(self):
        self.assertHolidayName(
            "Lady of Camarin Day", (f"{year}-12-08" for year in self.full_range)
        )

    def test_l10n_default(self):
        self.assertLocalizedHolidays(
            ("2022-01-01", "New Year's Day"),
            ("2022-01-17", "Martin Luther King Jr. Day"),
            ("2022-02-14", "Valentine's Day"),
            ("2022-02-21", "Washington's Birthday"),
            ("2022-03-07", "Guam Discovery Day"),
            ("2022-03-17", "Saint Patrick's Day"),
            ("2022-04-15", "Good Friday"),
            ("2022-05-08", "Mother's Day"),
            ("2022-05-30", "Memorial Day"),
            ("2022-06-19", "Father's Day; Juneteenth National Independence Day"),
            ("2022-06-20", "Juneteenth National Independence Day (observed)"),
            ("2022-07-04", "Independence Day"),
            ("2022-07-21", "Liberation Day (Guam)"),
            ("2022-09-05", "Labor Day"),
            ("2022-10-31", "Halloween"),
            ("2022-11-02", "All Souls' Day"),
            ("2022-11-11", "Veterans Day"),
            ("2022-11-24", "Thanksgiving Day"),
            ("2022-12-08", "Lady of Camarin Day"),
            ("2022-12-25", "Christmas Day"),
            ("2022-12-26", "Christmas Day (observed)"),
        )

    def test_l10n_th(self):
        self.assertLocalizedHolidays(
            "th",
            ("2022-01-01", "วันขึ้นปีใหม่"),
            ("2022-01-17", "วันมาร์ติน ลูเทอร์ คิง จูเนียร์"),
            ("2022-02-14", "วันวาเลนไทน์"),
            ("2022-02-21", "วันเกิดวอชิงตัน"),
            ("2022-03-07", "วันค้นพบกวม"),
            ("2022-03-17", "วันนักบุญแพทริก"),
            ("2022-04-15", "วันศุกร์ประเสริฐ"),
            ("2022-05-08", "วันแม่"),
            ("2022-05-30", "วันรำลึก"),
            ("2022-06-19", "วันประกาศอิสรภาพแห่งชาติจูนทีนท์; วันพ่อ"),
            ("2022-06-20", "ชดเชยวันประกาศอิสรภาพแห่งชาติจูนทีนท์"),
            ("2022-07-04", "วันประกาศอิสรภาพ"),
            ("2022-07-21", "วันปลดปล่อย (กวม)"),
            ("2022-09-05", "วันแรงงาน"),
            ("2022-10-31", "วันฮาโลวีน"),
            ("2022-11-02", "วันภาวนาอุทิศแด่ผู้ล่วงลับ"),
            ("2022-11-11", "วันทหารผ่านศึก"),
            ("2022-11-24", "วันขอบคุณพระเจ้า"),
            ("2022-12-08", "วันแม่พระแห่งคามาริน"),
            ("2022-12-25", "วันคริสต์มาส"),
            ("2022-12-26", "ชดเชยวันคริสต์มาส"),
        )
