use core:lang;
use core:geometry;
use ui;
use progvis;
use progvis:data;
use progvis:view;
use lang:bs:macro;

/**
 * View hints.
 *
 * Provides visualizations that are aware of pointers and references as C++ does them, and also show
 * which objects are no longer reachable in the implementation.
 */
class CppViewHints extends ViewHints {
	// Don't pause on function entry. We won't have set up our formal parameters by then...
	Bool pauseOnEntry() : override { false; }

	// We have barriers!
	Bool barriersAvailable() : override { true; }

	// Default hints. We need to call them at times to wrap everything in our heap allocations.
	private progvis:data:DefaultViewHints defaultHints;

	// Create element for visualization.
	Data? createData(World:Traversal t, TypeInfo info, unsafe:RawPtr root) : override {
		if (info.isRef)
			return null;
		if (!info.isValue)
			return null;

		// See if this is an array allocation. That is only relevant to do for "topmost" objects.
		if (root.any) {
			// Is it an array allocation?
			if ((root.readFilled & AllocFlags:arrayAlloc.v) != 0) {
				return CppArray(info.type);
			}
		}

		if (c = createSpecial(t, info))
			return c;

		// Fall through to the default hints to make sure that "createAllocation" will be called for us.
		defaultHints.createData(t, info, root);
	}

	// Allow special-case for global variables. The original Named is passed in case the
	// implementation wishes to inspect it. The default implementation simply calls 'createData'.
	Data? createGlobal(World:Traversal t, TypeInfo info, GlobalVar global) : override {
		if (global as CppGlobalArray) {
			// Special case: this is stored as a pointer, but we should remove the indirection to be
			// consistent with the visualization of stack-allocated arrays.
			if (ptrType = info.type as PtrType) {
				if (inside = ptrType.inside) {
					return CppIndirectArray(inside);
				}
			}
		}

		super:createGlobal(t, info, global);
	}

	// Create a custom allocation.
	Allocation createAllocation(unsafe:RawPtr ptr, Data data) {
		HeapAllocation(ptr, data);
	}

	// Get a title for the type.
	Str? typeTitle(World world, TypeInfo info) : override {
		if (type = info.type as PtrType) {
			if (t = type.params[0].type) {
				TypeInfo info(t, true, false);

				Str result = world.typeTitle(info);
				if (type.isConst)
					result = "const " + result;

				if (type.isRef)
					return result + " &";
				else
					return result + " *";
			}
		} else if (info.type is named{Byte}) {
			return "char";
		}

		null;
	}

	// Check if this is a special type in some way.
	private Data? createSpecial(World:Traversal t, TypeInfo info) {
		if (ptr = info.type as PtrType) {
			return CppPointer(ptr);
		}

		if (info.type is named{runtime:semaphore}) {
			return createSync(t, info);
		}

		if (info.type is named{runtime:lock}) {
			return createSync(t, info);
		}

		if (info.type is named{runtime:condition}) {
			return createSync(t, info);
		}

		if (info.type is named{runtime:tid_t}) {
			// Hide that tid_t is a struct, and just output its contents.
			return progvis:data:ThreadId();
		}

		if (info.type is named{Byte}) {
			return CppChar();
		}

		null;
	}

	// Create a visualization for a synchronization primitive. This hides the "impl" variable and
	// wraps the allocation in a SyncData type that exposes the queue of waiting threads to the
	// model checker (but not to the user of the visualization).
	private Data createSync(World:Traversal t, TypeInfo info) {
		var members = t.findMembers(info.type);
		Str impl = "impl";

		// Find the "impl" member.
		Nat? implOffset;
		for (m in members) {
			if (m.name == impl) {
				implOffset = m.offset;
			}
		}

		Composite result = if (implOffset) {
			SyncData(info.type, implOffset);
		} else {
			Composite(info.type);
		};

		Bool first = true;
		for (m in members) {
			if (m.name == impl) {
				first = m.first;
			} else {
				result.add(m.name, m.offset, t.create(m.type), first | m.first);
				first = false;
			}
		}

		result;
	}

	Drawable? createView(World world, Data data) : override {
		if (data as CppChar) {
			return CppCharView(data);
		} else if (data as CppArray) {
			return WithCaption(CppArrayView(data), data.title(world));
		}

		null;
	}

	Drawable decorateAllocation(Allocation alloc, Drawable data) : override {
		if (alloc as HeapAllocation) {
			return inject(data, (x) => HeapView(alloc, x));
		}
		data;
	}
}


/**
 * C++ pointers/references.
 */
class CppPointer extends Pointer {
	init(PtrType type) {
		init(type) { targetType = type.inside; }
	}
	init(CppPointer original) {
		init(original) { targetType = original.targetType; }
	}

	// The type we point to.
	private core:lang:Type? targetType;

	void traverse(World:Traversal t, unsafe:RawPtr object, Nat offset) : override {
		if (!object.any) {
			to = null;
			return;
		}

		var base = object.readPtr(offset);
		if (!base.any) {
			to = null;
			return;
		}

		var sPtr = core:asm:sPtr.current;
		Nat ptrOffset = object.readNat(offset + sPtr) - 2*sPtr;

		Data baseData = t.findOrCreate(base);
		to = baseData.atOffset(targetType, ptrOffset);
	}

	// Copy.
	protected Data copyObject() : override {
		CppPointer(this);
	}
}


/**
 * Synchronization primitives.
 *
 * Is aware of the hidden "impl" data member, and extracts a list of waiting threads from it. This
 * list is only exposed to the model checker (through "summary"), and not to the visualization (as
 * the wake order is generally undefined).
 */
class SyncData extends Composite {
	// Create, and provide the offset of the "impl" member (expected to be a pointer).
	init(Type? type, Nat implOffset) {
		init(type) {
			implOffset = implOffset;
		}
	}

	// Copy.
	init(SyncData original) {
		init(original) {
			implOffset = original.implOffset;
			waiting = original.waiting;
		}
	}

	// Offset of the "impl" member.
	private Nat implOffset;

	// Threads currently waiting for this synchronization primitive.
	private Nat[] waiting;

	// Traverse.
	void traverse(World:Traversal t, unsafe:RawPtr object, Nat offset) : override {
		super:traverse(t, object, offset);

		if (x = object.readPtr(offset + implOffset).asObject() as runtime:Waitable) {
			waiting = x.waitingThreads();
		}
	}

	// Summary.
	protected void putSummary(StrBuf to, SummaryData data) : override {
		super:putSummary(to, data);

		// Also put the queue of waiting threads.
		for (t in waiting)
			to << "+" << data.threadId.get(t, 0);
	}

	// Copy.
	protected Data copyObject() : override {
		SyncData(this);
	}
	protected void copyData(Data->Data copies) : override {
		super:copyData(copies);
	}
}


/**
 * Custom logic for C++ heap allocations.
 *
 * Detects when allocations are deallocated, and makes sure they remain if they are not deallocated
 * properly.
 */
class HeapAllocation extends Allocation {
	// Create.
	init(unsafe:RawPtr src, Data data) {
		init(src, data) {}
	}

	// Is this allocation dead?
	Bool dead;

	// Is this a static allocation that we don't need to explicitly free.
	Bool staticAlloc;

	// Update our state.
	void update(World:Traversal t, Nat id) : override {
		super:update(t, id);

		// Update our state.
		Nat filled = src.readFilled();
		staticAlloc = (filled & AllocFlags:staticAlloc.v) != 0;
		dead = (filled & AllocFlags:sizeMask.v) == 0;
	}

	// Remove when we are deallocated, or if we are a static allocation (e.g. a C-string).
	Bool remove() : override {
		dead | staticAlloc;
	}
}


/**
 * Visualization of the heap allocations.
 */
class HeapView extends WrapDrawable {
	init(HeapAllocation alloc, Drawable wrap) {
		init(wrap) {
			alloc = alloc;
		}
	}

	// Allocation with data.
	HeapAllocation alloc;

	// Is this heap object dead?
	Bool dead;

	void draw(ui:Graphics g, Bool active) : override {
		super:draw(g, active);

		if (alloc.dead) {
			// Draw a big cross.
			Size s = size;
			g.line(pos, pos + s, errorFg);
			g.line(pos + Size(0, s.h), pos + Size(s.w, 0), errorFg);
		}
	}

	void update(Workspace:Update ws) : override {
		super:update(ws);
		dead = alloc.dead;
	}

}


/**
 * A single C++ character.
 */
class CppChar extends Primitive {
	init() {
		init(named{Byte}) {}
	}
	init(CppChar original) {
		init(original) {
			charVal = original.charVal;
		}
	}

	// Char value.
	Byte charVal;

	void traverse(World:Traversal t, unsafe:RawPtr object, Nat offset) : override {
		charVal = object.readByte(offset);
		value = charVal.toS;
	}

	// Copy.
	protected Data copyObject() : override {
		CppChar(this);
	}
}

class CppCharView extends AtomView {
	// Character value.
	private Text ch;

	// Numeric value.
	private Text num;

	// Data source.
	private CppChar src;

	// Last value.
	private Nat last;

	init(CppChar data) {
		init() {
			src = data;
			ch = Text("", codeFont);
			num = Text("", dataFont);
			last = 0x100; // Not possible to represent in a Byte.
		}
	}

	Data? srcData() : override { src; }

	void update(Workspace:Update ws) : override {
		ws.setVisual(src, this);
		super:update(ws);
		if (last != src.charVal.nat) {
			last = src.charVal.nat;
			ch = Text(Char(last).toS, codeFont);
			num = Text("(${last})", dataFont);
			invalidateSize();
		}
	}

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);

		Rect p = rect;
		Size numSize = num.size;
		Size margin = dataMargin;
		g.draw(num, dataFg, p.p1 - numSize - margin);
		g.draw(ch, dataFg, Point(p.p0.x + margin.w, p.p1.y - ch.size.h - margin.h));
	}

	void drawLinks(Graphics g, Bool active) : override {}

	Size computeSize() : override {
		Size a = ch.size;
		Size b = num.size;

		Size(a.w + b.w + dataMargin.w*3, max(a.h, b.h) + dataMargin.h*2);
	}
}

/**
 * C++ array data. For custom view of the array data.
 */
class CppArray extends Array {
	// Create.
	init(Type elemType) {
		init(elemType, true) {
			end = Pad();
		}

		contents.push(end);
	}

	// Copy.
	init(CppArray original) {
		init(original) {
			elemSize = original.elemSize;
			end = original.end;
		}
	}

	// Element size, cached.
	private Nat elemSize;

	// Padding element at the end of the array.
	private Data end;

	// Update the array.
	void update(World:Traversal t, unsafe:RawPtr array, Nat offset) {
		if (offset == 0) {
			contents.pop();
			updateArray(t, array, arraySize(array));
			contents.push(end);
		} else {
			print("WARNING: Invalid C++ array.");
		}
	}

	// Traverse.
	void traverse(World:Traversal t, unsafe:RawPtr array, Nat offset) {
		if (offset == 0) {
			contents.pop();
			traverseArray(t, array);
			contents.push(end);
		} else {
			print("WARNING: Invalid C++ array.");
		}
	}

	// Get the size of the array.
	private Nat arraySize(unsafe:RawPtr array) {
		elemSize = array.readSize;
		array.readCount & AllocFlags:sizeMask.v;
	}

	// Type.
	Type? type() : override {
		elemType;
	}

	// Title.
	Str title(World world) : override {
		world.typeTitle(elemType);
	}

	// Copy.
	protected Data copyObject() : override {
		CppArray(this);
	}
	protected void copyData(Data->Data copies) : override {
		super:copyData(copies);
		end = end.copy(copies);
	}

	// Find the data at a particular offset.
	Data atOffset(Type? type, Nat offset) {
		if (contents.empty)
			return this;
		if (elemSize == 0)
			return this;

		Nat element = min(offset / elemSize, contents.count - 1);
		return contents[element].atOffset(type, offset - element*elemSize);
	}
}

/**
 * Special case of the above to support global arrays.
 *
 * Global arrays are actually stored as a pointer to the array. We hide this fact to make the
 * visualization consistent with how arrays are visualized on the stack.
 */
class CppIndirectArray extends CppArray {
	// Create.
	init(Type elemType) {
		init(elemType) {}
	}

	// Copy.
	init(CppIndirectArray original) {
		init(original) {}
	}

	// Update.
	void update(World:Traversal t, unsafe:RawPtr ptr, Nat offset) {
		var array = ptr.readPtr(offset);
		if (array.any) {
			t.addData(array, this);
			super:update(t, array, 0);
		}
	}

	// Traverse.
	void traverse(World:Traversal t, unsafe:RawPtr ptr, Nat offset) {
		var array = ptr.readPtr(offset);
		if (array.any) {
			super:traverse(t, array, 0);
		}
	}

	// Copy. Note: copyData is fine to not override, we don't introduce extra data.
	protected Data copyObject() : override {
		CppIndirectArray(this);
	}
}


/**
 * View of the array in C++. It differs from the "original" visualization in that it draws each part
 * as its own box, without numbers.
 */
class CppArrayView extends DataView {
	init(CppArray data) {
		init {
			data = data;
		}
	}

	private CppArray data;

	Data? srcData() : override { data; }

	void draw(Graphics g, Bool active) : override {
		// We don't need to call the super class here, minimize overdraw.
		Point at = pos;
		Float width = size.w;
		for (x in content) {
			Size s = x.size;
			s.w = width;
			x.size = s;

			x.pos = at;
			x.draw(g, false);

			at.y += s.h;
		}
	}

	void drawLinks(Graphics g, Bool active) : override {
		for (v in content) {
			v.drawLinks(g, active);
		}
	}

	void update(Workspace:Update ws) {
		ws.setVisual(data, this);

		for (id, elem in data.contents) {
			if (id >= content.count) {
				content << create(ws, elem);
			}
			// Note: Types don't change.
			content[id].update(ws);
		}
		while (content.count > data.contents.count)
			content.pop();

		invalidateSize();
	}

private:

	Drawable[] content;

	Drawable create(Workspace:Update ws, Data data) {
		var c = ws.create(data);
		// Remove captions.
		if (c as WithCaption)
			return c.wrap;
		else
			return c;
	}

	Size computeSize() : override {
		Size s;
		for (x in content) {
			s.w = max(s.w, x.size.w);
			s.h += x.size.h;
		}
		s;
	}
}
