#pragma once
#include <panda/lib.h>
#include <panda/string.h>
#include <panda/time/basic.h>
#include <panda/string_view.h>

namespace panda { namespace time {

using panda::unlikely;

static const char GMT_FALLBACK[] = "GMT0";

const size_t TZNAME_MAX = 255; // max length of timezone name or POSIX rule (Europe/Moscow, ...)

const ptime_t EPOCH_NEGINF = std::numeric_limits<ptime_t>::min();

const int DAYS_PER_YEAR     = 365; // non-leap year only
const int DAYS_PER_QYEAR    = DAYS_PER_YEAR*4 + 1;
const int DAYS_PER_CENT     = DAYS_PER_QYEAR*25 - 1;
const int DAYS_PER_QCENT    = DAYS_PER_CENT*4 + 1;
const int EPOCH_CHRIST_DAYS = 719162; // christ_days(1970);
const int EPOCH_WDAY        = 4;

const ptime_t MAX_YEARS   = (ptime_t) 1 << 32;
const ptime_t MAX_MONTHS  = MAX_YEARS*12;
const ptime_t MAX_DAYS    = MAX_MONTHS*31;
const ptime_t MAX_HOURS   = MAX_DAYS*24;
const ptime_t MAX_MINUTES = MAX_HOURS*60;
const ptime_t MAX_EPOCH   = MAX_MINUTES*60;

const ptime_t OUTLIM_MONTH_BY_12    = ((MAX_MONTHS / 12) + 1)*12;
const ptime_t OUTLIM_EPOCH_BY_86400 = ((MAX_EPOCH / 86400) + 1)*86400;
const ptime_t OUTLIM_DAY_BY_7       = ((MAX_DAYS / 7) + 1)*7;
const ptime_t OUTLIM_DAY_BY_QCENT   = ((MAX_DAYS / DAYS_PER_QCENT) + 1)*DAYS_PER_QCENT;

const int YDAY2MON[][366] = {
    {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,-1},
    {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11}
};

const int YDAY2MDAY[][366] = {
    {1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,-1},
    {1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31}
};

const int MON2YDAY[][12] = {
    {0,31,59,90,120,151,181,212,243,273,304,334},
    {0,31,60,91,121,152,182,213,244,274,305,335}
};

panda::string readfile (const std::string_view& path);

inline int64_t char8_to_int64 (const char* source) {
    return *((int64_t*) source);
}

// GIVEN DAYS PASSED SINCE 1 Jan 0001 00:00:00 CALCULATES REAL YEAR AND DAYS REMAINDER - YDAY [0-365]
inline void christ_year (ptime_t days, int32_t &year, int32_t &remainder) {
    // 1-st step: separate FULL QUAD CENTURIES
    ptime_t tmp = (days + OUTLIM_DAY_BY_QCENT) % DAYS_PER_QCENT;
    year = (days - tmp)/DAYS_PER_QCENT * 400;
    days = tmp;
    
    // 2-nd step: separate FULL CENTURIES, condition fixes QCENT -> CENT border
    if (unlikely(days == DAYS_PER_CENT*4)) {
        year += 300;
        days = DAYS_PER_CENT;
    } else {
        year += days/DAYS_PER_CENT * 100;
        days %= DAYS_PER_CENT;
    }
    
    // 3-rd step: separate FULL QUAD YEARS, no border fix needed
    year += days/DAYS_PER_QYEAR * 4;
    days %= DAYS_PER_QYEAR;
    
    // 4-th step: separate FULL YEARS, condition fixes QYEAR -> YEAR border
    if (unlikely(days == DAYS_PER_YEAR*4)) {
        year += 4; // actually 3, but we must add 1 to result year, as the start is 1-st year, not 0-th
        remainder = 365;
    } else {
        year += days/DAYS_PER_YEAR + 1; // we must add 1 to result year, as the start is 1-st year, not 0-th
        remainder = days % DAYS_PER_YEAR;
    }
}

}}
