package DD;

our $DATE = '2017-02-10'; # DATE
our $VERSION = '0.001'; # VERSION

use Exporter qw(import);
our @EXPORT = qw(dd dmp);

our $BACKEND = $ENV{PERL_DD_BACKEND} || "Data::Dump";

sub dd {
    if ($BACKEND eq 'Data::Dump') {
        require Data::Dump;
        print Data::Dump::dump(@_), "\n";
        return @_;
    } elsif ($BACKEND eq 'Data::Dump::Color') {
        require Data::Dump::Color;
        print Data::Dump::Color::dump(@_), "\n";
        return @_;
    } elsif ($BACKEND eq 'Data::Dumper') {
        require Data::Dumper;
        local $Data::Dumper::Terse = 1;
        local $Data::Dumper::Indent = 1;
        local $Data::Dumper::Useqq = 1;
        local $Data::Dumper::Deparse = 1;
        local $Data::Dumper::Quotekeys = 0;
        local $Data::Dumper::Sortkeys = 1;
        print Data::Dumper::Dumper(@_);
        return @_;
    } elsif ($BACKEND eq 'Data::Dmp') {
        require Data::Dmp;
        goto \&Data::Dmp::dd;
    } else {
        die "DD: Unknown backend '$BACKEND'";
    }
}

1;
# ABSTRACT: Dump data structure for debugging

__END__

=pod

=encoding UTF-8

=head1 NAME

DD - Dump data structure for debugging

=head1 VERSION

This document describes version 0.001 of DD (from Perl distribution DD), released on 2017-02-10.

=head1 SYNOPSIS

In your code:

 use DD; # exports dd() and dmp()
 ...
 dd $data; # prints data to STDOUT
 ...
 my $foo = dd $data; # prints data to STDOUT, also returns it so $foo gets assigned

On the command-line:

 % perl -MDD -E'...; dd $data; ...'

=head1 DESCRIPTION

C<DD> is a module with a short name you can use for debugging. It provides
C<dd()> which dumps data structure to STDOUT, as well as return the original
data so you can insert C<dd> in the middle of expressions.

C<DD> can use several kinds of backends. The default is L<Data::Dump> which is
chosen because it's a mature module and produces visually nice dumps for
debugging. You can also use these other backends:

=over

=item * Data::Dumper

=item * Data::Dump::Color

Optional dependency.

=item * Data::Dmp

Optional dependency.

=back

=head1 PACKAGE VARIABLES

=head2 $BACKEND

The backend to use. The default is to use C<PERL_DD_BACKEND> environment
variable or "Data::Dump" as the fallback default.

=head1 FUNCTIONS

=head2 dd

=head1 ENVIRONMENT

=head2 PERL_DD_BACKEND

Can be used to set the default backend.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/DD>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-DD>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=DD>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<XXX> - basically the same thing but with a different name and defaults. I
happen to use "XXX" to mark todo items in source code, so I prefer C<DD>
instead.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
