package XML::GrammarBase::Role::DataDir;
$XML::GrammarBase::Role::DataDir::VERSION = '0.2.5';
use strict;
use warnings;


use MooX::Role 'late';

use File::ShareDir qw(dist_dir);

my $_component_re = qr/[A-Za-z_]\w*/;

has 'module_base' => (
    isa => sub {
        my ($dist_name) = @_;
        if (
            not(   ( ref($dist_name) eq '' )
                && ( $dist_name =~ m/\A$_component_re(?:-$_component_re)*\z/ ) )
            )
        {
            die
"module_base must be a distribution string of components separated by dashes";
        }
    },
    ,
    is => 'rw'
);
has 'data_dir' => (
    isa     => 'Str',
    is      => 'rw',
    default => sub { return shift->_calc_default_data_dir(); },
    lazy    => 1,
);

sub _calc_default_data_dir
{
    my ($self) = @_;

    return dist_dir( $self->module_base() );
}

sub _undefize
{
    my $class = shift;
    my $v     = shift;

    return defined($v) ? $v : "(undef)";
}

sub dist_path
{
    my ( $self, $basename ) = @_;

    return File::Spec->catfile( $self->data_dir, $basename );
}

sub dist_path_slot
{
    my ( $self, $slot ) = @_;

    return $self->dist_path( $self->$slot() );
}


1;    # End of XML::GrammarBase::RelaxNG::Validate

__END__

=pod

=encoding UTF-8

=head1 NAME

XML::GrammarBase::Role::DataDir - provide the data_dir accessor.

=head1 VERSION

version 0.2.5

=head1 SYNOPSIS

    package MyClass::WithDataDir;

    use MooX 'late';

    with ('XML::GrammarBase::Role::DataDir');

    has '+module_base' => (default => 'XML-Grammar-MyGrammar');

    package main;

    my $obj = MyClass::WithDataDir->new(
        data_dir => "/path/to/data-dir",
    );

=head1 VERSION

version 0.2.5

=head1 SLOTS

=head2 module_base

The basename of the distribution - used for dist dir.

=head2 data_dir

The data directory where the XML assets can be found (the RELAX NG schema, etc.)

=head1 METHODS

=head2 $self->dist_path($basename)

Returns the $basename relative to data_dir().

Utility method.

=head2 $self->dist_path_slot($slot)

Returns the basename of $self->$slot() relative to data_dir().

Utility method.

=head1 AUTHOR

Shlomi Fish, C<< <shlomif at cpan.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-xml-grammarbase at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=XML-GrammarBase>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc XML::GrammarBase

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=XML-GrammarBase>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/XML-GrammarBase>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/XML-GrammarBase>

=item * Search CPAN

L<http://search.cpan.org/dist/XML-GrammarBase/>

=back

=head1 ACKNOWLEDGEMENTS

=head1 COPYRIGHT & LICENSE

Copyright 2009 Shlomi Fish.

This program is distributed under the MIT (X11) License:
L<http://www.opensource.org/licenses/mit-license.php>

Permission is hereby granted, free of charge, to any person
obtaining a copy of this software and associated documentation
files (the "Software"), to deal in the Software without
restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following
conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

=head1 AUTHOR

Shlomi Fish <shlomif@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2018 by Shlomi Fish.

This is free software, licensed under:

  The MIT (X11) License

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
L<https://github.com/shlomif/perl-XML-GrammarBase/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=for :stopwords cpan testmatrix url annocpan anno bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Perldoc

You can find documentation for this module with the perldoc command.

  perldoc XML::GrammarBase::Role::DataDir

=head2 Websites

The following websites have more information about this module, and may be of help to you. As always,
in addition to those websites please use your favorite search engine to discover more resources.

=over 4

=item *

MetaCPAN

A modern, open-source CPAN search engine, useful to view POD in HTML format.

L<https://metacpan.org/release/XML-GrammarBase>

=item *

Search CPAN

The default CPAN search engine, useful to view POD in HTML format.

L<http://search.cpan.org/dist/XML-GrammarBase>

=item *

RT: CPAN's Bug Tracker

The RT ( Request Tracker ) website is the default bug/issue tracking system for CPAN.

L<https://rt.cpan.org/Public/Dist/Display.html?Name=XML-GrammarBase>

=item *

AnnoCPAN

The AnnoCPAN is a website that allows community annotations of Perl module documentation.

L<http://annocpan.org/dist/XML-GrammarBase>

=item *

CPAN Ratings

The CPAN Ratings is a website that allows community ratings and reviews of Perl modules.

L<http://cpanratings.perl.org/d/XML-GrammarBase>

=item *

CPANTS

The CPANTS is a website that analyzes the Kwalitee ( code metrics ) of a distribution.

L<http://cpants.cpanauthors.org/dist/XML-GrammarBase>

=item *

CPAN Testers

The CPAN Testers is a network of smoke testers who run automated tests on uploaded CPAN distributions.

L<http://www.cpantesters.org/distro/X/XML-GrammarBase>

=item *

CPAN Testers Matrix

The CPAN Testers Matrix is a website that provides a visual overview of the test results for a distribution on various Perls/platforms.

L<http://matrix.cpantesters.org/?dist=XML-GrammarBase>

=item *

CPAN Testers Dependencies

The CPAN Testers Dependencies is a website that shows a chart of the test results of all dependencies for a distribution.

L<http://deps.cpantesters.org/?module=XML::GrammarBase>

=back

=head2 Bugs / Feature Requests

Please report any bugs or feature requests by email to C<bug-xml-grammarbase at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/Public/Bug/Report.html?Queue=XML-GrammarBase>. You will be automatically notified of any
progress on the request by the system.

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/shlomif/perl-XML-GrammarBase>

  git clone git://github.com/shlomif/perl-XML-GrammarBase.git

=cut
