BEGIN { do "./t/lib/ANFANG.pm" or die ( $@ || $! ) }

use strict;
use warnings;

use Test::More;
use Test::Exception;
use Test::Warn;

use DBICTest;

my $schema = DBICTest->init_schema();

# Check the defined unique constraints
is_deeply(
  [ sort $schema->source('CD')->unique_constraint_names ],
  [ qw/cd_artist_title primary/ ],
  'CD source has an automatically named unique constraint'
);
is_deeply(
  [ sort $schema->source('Producer')->unique_constraint_names ],
  [ qw/primary prod_name/ ],
  'Producer source has a named unique constraint'
);
is_deeply(
  [ sort $schema->source('Track')->unique_constraint_names ],
  [ qw/primary track_cd_position track_cd_title/ ],
  'Track source has three unique constraints'
);
is_deeply(
  [ sort $schema->source('Tag')->unique_constraint_names ],
  [ qw/primary tagid_cd tagid_cd_tag tags_tagid_tag tags_tagid_tag_cd/ ],
  'Tag source has five unique constraints (from add_unique_constraings)'
);

my $artistid = 1;
my $title    = 'UNIQUE Constraint';

my $cd1 = $schema->resultset('CD')->find_or_create({
  artist => $artistid,
  title  => $title,
  year   => 2005,
});

my $cd2 = $schema->resultset('CD')->find(
  {
    artist => $artistid,
    title  => $title,
  },
  { key => 'cd_artist_title' }
);

is($cd2->get_column('artist'), $cd1->get_column('artist'), 'find by specific key: artist is correct');
is($cd2->title, $cd1->title, 'title is correct');
is($cd2->year, $cd1->year, 'year is correct');

my $cd3 = $schema->resultset('CD')->find($artistid, $title, { key => 'cd_artist_title' });

is($cd3->get_column('artist'), $cd1->get_column('artist'), 'find by specific key, ordered columns: artist is correct');
is($cd3->title, $cd1->title, 'title is correct');
is($cd3->year, $cd1->year, 'year is correct');

my $cd4 = $schema->resultset('CD')->update_or_create(
  {
    artist => $artistid,
    title  => $title,
    year   => 2007,
  },
);

ok(! $cd4->is_changed, 'update_or_create without key: row is clean');
is($cd4->cdid, $cd2->cdid, 'cdid is correct');
is($cd4->get_column('artist'), $cd2->get_column('artist'), 'artist is correct');
is($cd4->title, $cd2->title, 'title is correct');
is($cd4->year, 2007, 'updated year is correct');

my $cd5 = $schema->resultset('CD')->update_or_create(
  {
    artist => $artistid,
    title  => $title,
    year   => 2007,
  },
  { key => 'cd_artist_title' }
);

ok(! $cd5->is_changed, 'update_or_create by specific key: row is clean');
is($cd5->cdid, $cd2->cdid, 'cdid is correct');
is($cd5->get_column('artist'), $cd2->get_column('artist'), 'artist is correct');
is($cd5->title, $cd2->title, 'title is correct');
is($cd5->year, 2007, 'updated year is correct');

my $cd6 = $schema->resultset('CD')->update_or_create(
  {
    cdid   => $cd2->cdid,
    artist => 1,
    title  => $cd2->title,
    year   => 2005,
  },
  { key => 'primary' }
);

ok(! $cd6->is_changed, 'update_or_create by PK: row is clean');
is($cd6->cdid, $cd2->cdid, 'cdid is correct');
is($cd6->get_column('artist'), $cd2->get_column('artist'), 'artist is correct');
is($cd6->title, $cd2->title, 'title is correct');
is($cd6->year, 2005, 'updated year is correct');

my $cd7 = $schema->resultset('CD')->find_or_create(
  {
    artist => $artistid,
    title  => $title,
    year   => 2010,
  },
  { key => 'cd_artist_title' }
);

is($cd7->cdid, $cd1->cdid, 'find_or_create by specific key: cdid is correct');
is($cd7->get_column('artist'), $cd1->get_column('artist'), 'artist is correct');
is($cd7->title, $cd1->title, 'title is correct');
is($cd7->year, $cd1->year, 'year is correct');

my $artist = $schema->resultset('Artist')->find($artistid);
my $cd8 = $artist->find_or_create_related('cds',
  {
    title  => $title,
    year   => 2020,
  },
  { key => 'cd_artist_title' }
);

is($cd8->cdid, $cd1->cdid, 'find_or_create related by specific key: cdid is correct');
is($cd8->get_column('artist'), $cd1->get_column('artist'), 'artist is correct');
is($cd8->title, $cd1->title, 'title is correct');
is($cd8->year, $cd1->year, 'year is correct');

# Add an extra row to potentially confuse the query
$schema->resultset('CD')->create ({
  artist => 2,
  title => $title,
  year => 2022,
});
my $cd9 = $artist->cds->update_or_create(
  {
    cdid   => $cd1->cdid,
    title  => $title,
    year   => 2021,
  },
  { key => 'cd_artist_title' }
);

ok(! $cd9->is_changed, 'update_or_create by specific key: row is clean');
is($cd9->cdid, $cd1->cdid, 'cdid is correct');
is($cd9->get_column('artist'), $cd1->get_column('artist'), 'artist is correct');
is($cd9->title, $cd1->title, 'title is correct');
is($cd9->year, 2021, 'year is correct');

# Table with two unique constraints, and we're satisying one of them
my $track = $schema->resultset('Track')->find(
  {
    cd       => 1,
    position => 3,
  },
  { order_by => 'position' }
);

is($track->get_column('cd'), 1, 'track cd is correct');
is($track->get_column('position'), 3, 'track position is correct');

# Test a table with a unique constraint but no primary key
my $row = $schema->resultset('NoPrimaryKey')->update_or_create(
  {
    foo => 1,
    bar => 2,
    baz => 3,
  },
  { key => 'foo_bar' }
);

ok(! $row->is_changed, 'update_or_create on table without primary key: row is clean');
is($row->foo, 1, 'foo is correct');
is($row->bar, 2, 'bar is correct');
is($row->baz, 3, 'baz is correct');

# Test a unique condition with extra information in the where attr
{
  my $artist = $schema->resultset('Artist')->find({ artistid => 1 });
  my $cd = $artist->cds->find_or_new(
    {
      cdid  => 1,
      title => 'Not The Real Title',
      year  => 3000,
    },
    { key => 'primary' }
  );

  ok($cd->in_storage, 'find correctly grepped the key across a relationship');
  is($cd->cdid, 1, 'cdid is correct');
}

# Test update_or_new
{
    my $cd1 = $schema->resultset('CD')->update_or_new(
      {
        artist => $artistid,
        title  => "SuperHits $$",
        year   => 2007,
      },
      { key => 'cd_artist_title' }
    );

    is($cd1->in_storage, 0, 'CD is not in storage yet after update_or_new');
    $cd1->insert;
    ok($cd1->in_storage, 'CD got added to strage after update_or_new && insert');

    my $cd2 = $schema->resultset('CD')->update_or_new(
      {
        artist => $artistid,
        title  => "SuperHits $$",
        year   => 2008,
      },
      { key => 'cd_artist_title' }
    );
    ok($cd2->in_storage, 'Updating year using update_or_new was successful');
    is($cd2->id, $cd1->id, 'Got the same CD using update_or_new');
}

# make sure the ident condition is assembled sanely
{
  my $artist = $schema->resultset('Artist')->find(1);

  $schema->is_executed_sql_bind( sub { $artist->discard_changes }, [
    [
      'SELECT me.artistid, me.name, me.rank, me.charfield FROM artist me WHERE me.artistid = ?',
      [ { dbic_colname => "me.artistid", sqlt_datatype => "integer" } => 1 ],
    ]
  ], 'Expected query on discard_changes');
}

{
  throws_ok {
    eval <<'MOD' or die $@;
      package # hide from PAUSE
        DBICTest::Schema::UniqueConstraintWarningTest;

      use base qw/DBIx::Class::Core/;

      __PACKAGE__->table('dummy');

      __PACKAGE__->add_column(qw/ foo bar /);

      __PACKAGE__->add_unique_constraint(
        constraint1 => [qw/ foo /],
        constraint2 => [qw/ bar /],
      );

      1;
MOD
  } qr/\Qadd_unique_constraint() does not accept multiple constraints, use add_unique_constraints() instead\E/,
    'add_unique_constraint throws when more than one constraint specified';
}
# make sure NULL is not considered condition-deterministic
my $art_rs = $schema->resultset('Artist')->search({}, { order_by => 'artistid' });
$art_rs->create ({ artistid => $_ + 640, name => "Outranked $_" }) for (1..2);
warnings_are {
  is(
    $art_rs->find ({ artistid => 642, rank => 13, charfield => undef })->name,
    'Outranked 2',
    'Correct artist retrieved with find'
  );

  is (
    $art_rs->search({ charfield => undef })->find ({ artistid => 642, rank => 13 })->name,
    'Outranked 2',
    'Correct artist retrieved with find'
  );
} [], 'no warnings';

done_testing;

