/*
 * SourceStream.hpp
 *****************************************************************************
 * Copyright © 2015 - VideoLAN and VLC Authors
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston MA 02110-1301, USA.
 *****************************************************************************/
#ifndef SOURCESTREAM_HPP
#define SOURCESTREAM_HPP

#include "../tools/Macros.hpp"

#include <vlc_common.h>
#include <vlc_block_helper.h>

namespace adaptive
{
    class BlockStreamInterface;

    class AbstractSourceStream
    {
        PREREQ_INTERFACE(AbstractSourceStream);

        public:
            virtual stream_t *makeStream() = 0;
            virtual void Reset() = 0;
            virtual size_t Peek(const uint8_t **, size_t) = 0;
    };

    class AbstractChunksSourceStream : public AbstractSourceStream
    {
        PREREQ_VIRTUAL(AbstractChunksSourceStream);

        public:
            AbstractChunksSourceStream(vlc_object_t *, BlockStreamInterface *);
            virtual ~AbstractChunksSourceStream();
            void Reset() override;
            stream_t *makeStream() override;

        protected:
            virtual ssize_t Read(uint8_t *, size_t) = 0;
            virtual int     Seek(uint64_t) = 0;
            bool b_eof;
            vlc_object_t *p_obj;
            BlockStreamInterface *source;

        private:
            static ssize_t read_Callback(stream_t *, void *, size_t);
            static int seek_Callback(stream_t *, uint64_t);
            static int control_Callback( stream_t *, int i_query, va_list );
            static void delete_Callback( stream_t * );
    };

    class ChunksSourceStream : public AbstractChunksSourceStream
    {
        public:
            ChunksSourceStream(vlc_object_t *, BlockStreamInterface *);
            virtual ~ChunksSourceStream();
            void Reset() override;

        protected:
            ssize_t Read(uint8_t *, size_t) override;
            int     Seek(uint64_t) override;
            size_t  Peek(const uint8_t **, size_t) override;

        private:
            block_t *p_block;
    };

    class BufferedChunksSourceStream : public AbstractChunksSourceStream
    {
        public:
            BufferedChunksSourceStream(vlc_object_t *, BlockStreamInterface *);
            virtual ~BufferedChunksSourceStream();
            void Reset() override;

        protected:
            ssize_t Read(uint8_t *, size_t) override;
            int     Seek(uint64_t) override;
            size_t  Peek(const uint8_t **, size_t) override;

        private:
            ssize_t doRead(uint8_t *, size_t);
            void fillByteStream(size_t);
            void invalidatePeek();
            static const int MAX_BACKEND = 5 * 1024 * 1024;
            static const int MIN_BACKEND_CLEANUP = 50 * 1024;
            uint64_t i_global_offset;
            size_t i_bytestream_offset;
            block_bytestream_t bs;
            block_t *p_peekdata;
    };
}
#endif // SOURCESTREAM_HPP
