Example of illegal code:

```ada
with System;
procedure P (X : Integer) is
   A : System.Address := X'Address; -- error: attribute Address in expression
begin
   ...
end P;
```

This error is issued on the use of attribute `Address` to get the address of a
variable in an expression. If this was allowed, it would be possible to create
an unknown alias of the variable by converting back this address into a
pointer, which is problematic for following the value of the variable.

If the motivation for taking the address of the variable is to manipulate it
through a pointer, attribute `Access` should be used instead:

```ada
procedure P (X : aliased Integer) is
   A : access constant Integer := X'Access;
begin
   ...
end P;
```

In that case, the variable `X` cannot be read or written until the scope of `A`
ends, as the declaration of `A` has transferred ownership of the memory for `X`
to `A`.

If the motivation is instead to store or pass the address to some other
component not in SPARK, then the code taking the address of `X` should be
marked as not in SPARK with aspect `SPARK_Mode => Off`. To mark only part of a
subprogram as not in SPARK, it should be hoisted as another subprogram
(possibly local to the original subprogram). Note that, in order to call this
subprogram from SPARK code, it should have a declaration separate from its
body, and that declaration should be in SPARK. To model the effect of the
subprogram, consider adding a `Global` contract if it is a procedure, and
making it `Volatile_Function` if it is a function.
