// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/DecayedParticles.hh"

namespace Rivet {


  /// @brief eta -> 3 pions
  class BESIII_2023_I2633025 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2023_I2633025);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      UnstableParticles ufs = UnstableParticles(Cuts::pid==PID::ETA);
      declare(ufs, "UFS");
      DecayedParticles ETA(ufs);
      ETA.addStable(PID::PI0);
      ETA.addStable(PID::K0S);
      declare(ETA, "ETA");
      // histograms
      for(unsigned int ix=0;ix<2;++ix)
	book(_h[ix],1,1,1+ix);
      book(_h[2],2,1,1);
      book(_h[3],"TMP/phase",refData(3,1,1));
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      static const map<PdgId,unsigned int> & mode1  = { {211,1}, {-211,1}, {111,1} };
      static const map<PdgId,unsigned int> & mode2   = { {111,3} };
      DecayedParticles ETA = apply<DecayedParticles>(event, "ETA");
      // loop over particles
      for(unsigned int ix=0;ix<ETA.decaying().size();++ix) {
	if ( ETA.modeMatches(ix,3,mode1)) {
	  const Particle & pi0 = ETA.decayProducts()[ix].at( 111)[0];
	  const Particle & pip = ETA.decayProducts()[ix].at( 211)[0];
	  const Particle & pim = ETA.decayProducts()[ix].at(-211)[0];
	  double s1 = (pi0.momentum()+pim.momentum()).mass2();
	  double s2 = (pi0.momentum()+pip.momentum()).mass2();
	  double s3 = (pip.momentum()+pim.momentum()).mass2();
	  double mOut = pi0.mass()+pip.mass()+pim.mass();
	  double Q = ETA.decaying()[ix].mass()-mOut;
	  double X = sqrt(3.)/2./ETA.decaying()[ix].mass()/Q*(s1-s2);
	  double Y = 3.*(sqr(ETA.decaying()[ix].mass()-pi0.mass())-s3)/2./ETA.decaying()[ix].mass()/Q-1.;
	  _h[0]->fill(X);
	  _h[1]->fill(Y);
	}
	else if  ( ETA.modeMatches(ix,3,mode2)) {
	  const Particles & pi0 = ETA.decayProducts()[ix].at(111);
	  double s1 = (pi0[2].momentum()+pi0[1].momentum()).mass2();
	  double s2 = (pi0[2].momentum()+pi0[0].momentum()).mass2();
	  double s3 = (pi0[0].momentum()+pi0[1].momentum()).mass2();
	  double mOut = pi0[2].mass()+pi0[0].mass()+pi0[1].mass();
	  double Q = ETA.decaying()[ix].mass()-mOut;
	  double X = sqrt(3.)/2./ETA.decaying()[ix].mass()/Q*(s1-s2);
	  double Y = 3.*(sqr(ETA.decaying()[ix].mass()-pi0[2].mass())-s3)/2./ETA.decaying()[ix].mass()/Q-1.;
	  double Z = sqr(X)+sqr(Y);
	  _h[2]->fill(Z);
	  _h[3]->fill(sqrt(s1));
	  _h[3]->fill(sqrt(s2));
	  _h[3]->fill(sqrt(s3));
	}
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for(unsigned int ix=0;ix<4;++ix)
	normalize(_h[ix]);
      // normalize the last plot to the phase space volume (PR has program which computed these)
      vector<double> phsp = { 0.5053258491122877,1.998068177102547,2.8176054749830572,3.430879406441175,
			      3.9381854558400438,4.376823781582348,4.765764587726765,5.116251484948019,
			      5.435605198934152,5.7289080859973875,5.999857170942998,6.251238852604172,
			      6.4852123636098105,6.7034885806109825,6.907447897156746,7.098220785316871,
			      7.276744544004773,7.443804308576157,7.600063343138945,7.746085843780316,
			      7.882354388614016,8.009283483796587,8.127230210705884,8.236502685313326,
			      8.337366841525332,8.430051912586967,8.514754887764916,8.591644152215865,
			      8.660862467577541,8.722529413669271,8.776743383887757,8.823583205772895,
			      8.863109441939471,8.895365413803265,8.920377980320332,8.938158095601809,
			      8.948701162231394,8.951987190983493,8.947980772078905,8.936630857831052,
			      8.917870351261696,8.891615489734567,8.857765006582719,8.81619904676176,
			      8.766777804345876,8.70933983968699,8.643700021610368,8.569647024219863,
			      8.486940287493418,8.395306324128901,8.294434219585622,8.183970124393385,
			      8.063510472332702,7.932593567202544,7.790689052670059,7.637184595627547,
			      7.471368844155493,7.292409318726114,7.099323279572297,6.890938646004227,
			      6.66584047811349,6.422295909376109,6.158145849060354,5.870643429408017,
			      5.556203066842222,5.209990748295864,4.825211451020806,4.391762610087976,
			      3.893380654973374,3.300463567443619,2.5457539597853507};

      // convert to scatter
      Scatter2DPtr tmp;
      book(tmp,3,1,1);
      barchart(_h[3],tmp);
      double step = 0.002;
      // divide by phase space volume
      for(unsigned int ip=0;ip<tmp->points().size();++ip) {
	tmp->points()[ip].scaleY(1./phsp[ip]/step);
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    Histo1DPtr _h[4];
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESIII_2023_I2633025);

}
