; RTL - A Formal Theory of Register-Transfer Logic and Computer Arithmetic
; Copyright (C) 1995-2013 Advanced Mirco Devices, Inc.
;
; Contact:
;   David Russinoff
;   1106 W 9th St., Austin, TX 78703
;   http://www.russsinoff.com/
;
; See license file books/rtl/rel9/license.txt.
;
; Author: David M. Russinoff (david@russinoff.com)

(in-package "ACL2")

(include-book "lop2")
(local (include-book "../../arithmetic/top"))
(local (include-book "bitn"))
(local (include-book "bits"))

(local
 (defund C (k a b)
   (- (bitn a k) (bitn b k))))

(defund OLAMT (a b e)
  (logxor a (lnot b (1+ e))))

(defund OLAMG (a b e)
  (logand a (lnot b (1+ e))))

(defund OLAMZ (a b e)
  (lnot (logior a (lnot b (1+ e))) (1+ e)))

(defund OLAM1 (a b e)
  (logand (bits (olamt a b e) e 2)
	  (logand (bits (olamg a b e) (1- e) 1)
		  (lnot (bits (olamz a b e) (- e 2) 0) (1- e)))))

(defund OLAM2 (a b e)
  (logand (lnot (bits (olamt a b e) e 2) (1- e))
	  (logand (bits (olamz a b e) (1- e) 1)
		  (lnot (bits (olamz a b e) (- e 2) 0) (1- e)))))

(defund OLAM3 (a b e)
  (logand (bits (olamt a b e) e 2)
	  (logand (bits (olamz a b e) (1- e) 1)
		  (lnot (bits (olamg a b e) (- e 2) 0) (1- e)))))

(defund OLAM4 (a b e)
  (logand (lnot (bits (olamt a b e) e 2) (1- e))
	  (logand (bits (olamg a b e) (1- e) 1)
		  (lnot (bits (olamg a b e) (- e 2) 0) (1- e)))))

(defund OLAM0 (a b e)
  (logior (olam1 a b e)
	  (logior (olam2 a b e)
		  (logior (olam3 a b e)
			  (olam4 a b e)))))

(defund OLAMB (a b e)
  (+ (* 2 (olam0 a b e))
     (lnot (bitn (olamt a b e) 0) 1)))

(local (defthm bitn-olamt
    (implies (and (integerp a)
		  (>= a 0) ;(> a 0)
		  (integerp b)
		  (>= b 0) ;(> b 0)
;		  (not (= a b))
;		  (= e (expo a))
		  (= e (expo b))
;		  (> e 0)
		  (integerp k)
		  (>= k 0)
		  (<= k e)
                  )
	     (iff (= (bitn (olamt a b e) k) 1)
		  (= (c k a b) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable c olamt)
		  :use ((:instance bitn-lnot-not-equal (x b) (n (1+ e)))
			(:instance bitn-0-1 (x a) (n k))
			(:instance bitn-0-1 (x b) (n k))
			(:instance bitn-0-1 (x (lnot b (1+ e))) (n k))
			(:instance expo-upper-bound (x b))
			(:instance bitn-logxor (x a) (y (lnot b (1+ e))) (n k)))))))

(defthm OLAMT-NAT
  (implies (and (integerp a)
                (>= a 0) ;(> a 0)
;                (integerp b)
 ;               (> b 0)
  ;              (not (= a b))
   ;             (= e (expo a))
    ;            (= e (expo b))
     ;           (> e 0)
                )
           (and (integerp (olamt a b e))
                (>= (olamt a b e) 0)))
;  :rule-classes ()
  :hints (("Goal" :in-theory (enable olamt)
           )))

(local (defthm bitn-olamg
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 0)
		  (integerp k)
		  (>= k 0)
		  (<= k e))
	     (iff (= (bitn (olamg a b e) k) 1)
		  (= (c k a b) 1)))
  :rule-classes ()
  :hints (("Goal" :in-theory (e/d ( c olamg) (EXPO-BOUND-ERIC
                                              EXPO-COMPARISON-REWRITE-TO-BOUND
                                              EXPO-COMPARISON-REWRITE-TO-BOUND-2))
		  :use ((:instance bitn-lnot-not-equal (x b) (n (1+ e)))
			(:instance bitn-0-1 (x a) (n k))
			(:instance bitn-0-1 (x b) (n k))
			(:instance bitn-0-1 (x (lnot b (1+ e))) (n k))
			(:instance expo-upper-bound (x b)))))))

(defthm OLAMG-NAT
    (implies (and (integerp a)
		  (> a 0)
                  )
	     (and (integerp (olamg a b e))
		  (>= (olamg a b e) 0)))
;  :rule-classes ()
  :hints (("Goal" :in-theory (enable olamg))))

(local (defthm bitn-olamz-1
         (implies (and (integerp a)
                       (> a 0)
                       (integerp b)
                       (> b 0)
                       (not (= a b))
                       (= e (expo a))
                       (= e (expo b))
                       (> e 0)
                       (integerp k)
                       (>= k 0)
                       (<= k e))
                  (iff (= (bitn (logior a (lnot b (1+ e))) k) 0)
                       (= (c k a b) -1)))
         :rule-classes ()
         :hints (("Goal" :in-theory (e/d (c) (EXPO-BOUND-ERIC
                                              EXPO-COMPARISON-REWRITE-TO-BOUND
                                              EXPO-COMPARISON-REWRITE-TO-BOUND-2
                                              BITN-KNOWN-NOT-0-REPLACE-WITH-1))
		  :use ((:instance bitn-lnot-not-equal (x b) (n (1+ e)))
			(:instance bitn-0-1 (x a) (n k))
			(:instance bitn-0-1 (x b) (n k))
			(:instance bitn-0-1 (x (lnot b (1+ e))) (n k))
			(:instance expo-upper-bound (x b)))))))

(local (defthm bitn-olamz
         (implies (and (integerp a)
                       (> a 0)
                       (integerp b)
                       (> b 0)
                       (not (= a b))
                       (= e (expo a))
                       (= e (expo b))
                       (> e 0)
                       (integerp k)
                       (>= k 0)
                       (<= k e))
                  (iff (= (bitn (olamz a b e) k) 1)
                       (= (c k a b) -1)))
         :rule-classes ()
         :hints (("Goal" :in-theory (set-difference-theories
                                     (enable  olamz
                                              bits-reduce)
                                     '(BITN-KNOWN-NOT-0-REPLACE-WITH-1
                                       EXPO-BOUND-ERIC
                                       EXPO-COMPARISON-REWRITE-TO-BOUND
                                       EXPO-COMPARISON-REWRITE-TO-BOUND-2))
		  :use (bitn-olamz-1
			(:instance bitn-lnot-not-equal (x (logior a (lnot b (1+ e)))) (n (1+ e)))

			(:instance bitn-0-1 (x (olamz a b e)) (n k))
			(:instance bitn-0-1 (x (logior a (lnot b (1+ e)))) (n k))
			(:instance expo-upper-bound (x b))
                    ;			(:instance or-dist-a (x a) (y (lnot b (1+ e))) (n (1+ e)))
			(:instance expo-upper-bound (x a)))))))

(defthm OLAMZ-NAT
    (implies (and (integerp a)
		  (> a 0)
;		  (integerp b)
	;	  (> b 0)
		;  (not (= a b))
		 ; (= e (expo a))
		 ; (= e (expo b))
		  ;(> e 0)
                  )
	     (and (integerp (olamz a b e))
		  (>= (olamz a b e) 0))))

(local (defthm bitn-olam1-1
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (>= k 0))
	     (equal (bitn (olam1 a b e) k)
		    (logand (bitn (bits (olamt a b e) e 2) k)
			    (logand (bitn (bits (olamg a b e) (1- e) 1) k)
				    (bitn (lnot (bits (olamz a b e) (- e 2) 0) (1- e)) k)))))
  :hints (("Goal" :in-theory (enable olam1)))))

(local (defthm bitn-olam1-2
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (>= k 0))
	     (iff (= (bitn (olam1 a b e) k) 1)
		  (and (= (bitn (bits (olamt a b e) e 2) k) 1)
		       (= (bitn (bits (olamg a b e) (1- e) 1) k) 1)
		       (= (bitn (lnot (bits (olamz a b e) (- e 2) 0) (1- e)) k) 1))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable bits olam1 logand)
		  :use ((:instance bitn-0-1 (x (bits (olamt a b e) e 2)) (n k))
			(:instance bitn-0-1 (x (bits (olamg a b e) (1- e) 1)) (n k))
			(:instance bitn-0-1 (x (lnot (bits (olamz a b e) (- e 2) 0) (1- e))) (n k)))))))

(local (in-theory (disable bitn-olam1-1)))

(local (defthm bitn-olam1-3
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam1 a b e) k) 1)
		  (and (= (bitn (bits (olamt a b e) e 2) k) 1)
		       (= (bitn (bits (olamg a b e) (1- e) 1) k) 1)
		       (= (bitn (bits (olamz a b e) (- e 2) 0) k) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable olam1 bits lnot logand)
		  :use (;olamz-nat
			bitn-olam1-2
;			(:instance bits< (x (olamz a b e)) (i (- e 2)) (j 0))
			(:instance bitn-0-1 (x (lnot (bits (olamz a b e) (- e 2) 0) (1- e))) (n k))
			(:instance bitn-lnot-not-equal (x (bits (olamz a b e) (- e 2) 0)) (n (1- e)))
			(:instance bitn-0-1 (x (bits (olamz a b e) (- e 2) 0)) (n k)))))))

(local (defthm bitn-olam1
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam1 a b e) k) 1)
		  (and (= (bitn (olamt a b e) (+ 2 k)) 1)
		       (= (bitn (olamg a b e) (+ 1 k)) 1)
		       (= (bitn (olamz a b e) k) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable olam1 bits logand)
		  :use (;olamz-nat
			;olamt-nat
			;olamg-nat
			bitn-olam1-3
;			(:instance bits< (x (olamz a b e)) (i (- e 2)) (j 0))
			(:instance bitn-0-1 (x (olamz a b e)) (n k))
;			(:instance bitn-bits (x (olamt a b e)) (i e) (j 2))
;			(:instance bitn-bits (x (olamg a b e)) (i (- e 1)) (j 1))
	;		(:instance bitn-bits (x (olamz a b e)) (i (- e 2)) (j 0))
                        )))))

(defthm OLAM1-NAT
    (implies (and (integerp a)
		  (> a 0)
		 ; (integerp b)
		  ;(> b 0)
	;	  (not (= a b))
		;  (= e (expo a))
		 ; (= e (expo b))
		 ; (> e 1)
		 ; (integerp k)
		 ; (<= k (- e 2))
		 ; (>= k 0)
                  )
	     (and (integerp (olam1 a b e))
		  (>= (olam1 a b e) 0)))
;  :rule-classes ()
  :hints (("Goal" :in-theory (disable bits logand)
		  :use (;olamz-nat
			;olamt-nat
			;olamg-nat
;			(:instance bits< (x (olamz a b e)) (i (- e 2)) (j 0))
                        ))))

(local (defthm bitn-olam3-1
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (>= k 0))
	     (equal (bitn (olam3 a b e) k)
		    (logand (bitn (bits (olamt a b e) e 2) k)
			    (logand (bitn (bits (olamz a b e) (1- e) 1) k)
				    (bitn (lnot (bits (olamg a b e) (- e 2) 0) (1- e)) k)))))
  :hints (("Goal" :in-theory (enable olam3)))))

(local (defthm bitn-olam3-2
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (>= k 0))
	     (iff (= (bitn (olam3 a b e) k) 1)
		  (and (= (bitn (bits (olamt a b e) e 2) k) 1)
		       (= (bitn (bits (olamz a b e) (1- e) 1) k) 1)
		       (= (bitn (lnot (bits (olamg a b e) (- e 2) 0) (1- e)) k) 1))))
  :rule-classes ()
  :hints (("Goal"; :in-theory (disable bits olam3 logand)
		  :use ((:instance bitn-0-1 (x (bits (olamt a b e) e 2)) (n k))
			(:instance bitn-0-1 (x (bits (olamz a b e) (1- e) 1)) (n k))
			(:instance bitn-0-1 (x (lnot (bits (olamg a b e) (- e 2) 0) (1- e))) (n k)))))))

(local (in-theory (disable bitn-olam3-1)))

(local (defthm bitn-olam3-3
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam3 a b e) k) 1)
		  (and (= (bitn (bits (olamt a b e) e 2) k) 1)
		       (= (bitn (bits (olamz a b e) (1- e) 1) k) 1)
		       (= (bitn (bits (olamg a b e) (- e 2) 0) k) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable olam3 bits lnot logand)
		  :use (;olamg-nat
			bitn-olam3-2
;			(:instance bits< (x (olamg a b e)) (i (- e 2)) (j 0))
			(:instance bitn-0-1 (x (lnot (bits (olamg a b e) (- e 2) 0) (1- e))) (n k))
			(:instance bitn-lnot-not-equal (x (bits (olamg a b e) (- e 2) 0)) (n (1- e)))
			(:instance bitn-0-1 (x (bits (olamg a b e) (- e 2) 0)) (n k)))))))

(local (defthm bitn-olam3
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam3 a b e) k) 1)
		  (and (= (bitn (olamt a b e) (+ 2 k)) 1)
		       (= (bitn (olamz a b e) (+ 1 k)) 1)
		       (= (bitn (olamg a b e) k) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable olam3 bits logand)
		  :use (;olamg-nat
			;olamt-nat
			;olamz-nat
			bitn-olam3-3
;			(:instance bits< (x (olamg a b e)) (i (- e 2)) (j 0))
			(:instance bitn-0-1 (x (olamg a b e)) (n k))
;			(:instance bitn-bits (x (olamt a b e)) (i e) (j 2))
	;		(:instance bitn-bits (x (olamz a b e)) (i (- e 1)) (j 1))
		;	(:instance bitn-bits (x (olamg a b e)) (i (- e 2)) (j 0))
                        )))))

(defthm OLAM3-NAT
    (implies (and (integerp a)
		  (> a 0)
;		  (integerp b)
	;	  (> b 0)
;		  (not (= a b))
	;	  (= e (expo a))
		;  (= e (expo b))
		 ; (> e 1)
		 ; (integerp k)
		 ; (<= k (- e 2))
		 ; (>= k 0)
                  )
	     (and (integerp (olam3 a b e))
		  (>= (olam3 a b e) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable bits logand)
		  :use (;olamg-nat
			;olamt-nat
			;olamz-nat
;			(:instance bits< (x (olamg a b e)) (i (- e 2)) (j 0))
                        ))))

(local (defthm bitn-olam2-1
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (>= k 0))
	     (equal (bitn (olam2 a b e) k)
		    (logand (bitn (lnot (bits (olamt a b e) e 2) (1- e)) k)
			    (logand (bitn (bits (olamz a b e) (1- e) 1) k)
				    (bitn (lnot (bits (olamz a b e) (- e 2) 0) (1- e)) k)))))
  :hints (("Goal" :in-theory (enable olam2)))))

(local (defthm bitn-olam2-2
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (>= k 0))
	     (iff (= (bitn (olam2 a b e) k) 1)
		  (and (= (bitn (lnot (bits (olamt a b e) e 2) (1- e)) k) 1)
		       (= (bitn (bits (olamz a b e) (1- e) 1) k) 1)
		       (= (bitn (lnot (bits (olamz a b e) (- e 2) 0) (1- e)) k) 1))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable bits olam2 logand)
		  :use ((:instance bitn-0-1 (x (lnot (bits (olamt a b e) e 2) (1- e))) (n k))
			(:instance bitn-0-1 (x (bits (olamz a b e) (1- e) 1)) (n k))
			(:instance bitn-0-1 (x (lnot (bits (olamz a b e) (- e 2) 0) (1- e))) (n k)))))))

(local (in-theory (disable bitn-olam2-1)))

(local (defthm bitn-olam2-3
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam2 a b e) k) 1)
		  (and (= (bitn (bits (olamt a b e) e 2) k) 0)
		       (= (bitn (bits (olamz a b e) (1- e) 1) k) 1)
		       (= (bitn (bits (olamz a b e) (- e 2) 0) k) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable olam2 bits lnot logand)
		  :use (;olamz-nat
			;olamt-nat
			bitn-olam2-2
;			(:instance bits< (x (olamt a b e)) (i e) (j 2))
			(:instance bitn-0-1 (x (lnot (bits (olamt a b e) e 2) (1- e))) (n k))
			(:instance bitn-lnot-not-equal (x (bits (olamt a b e) e 2)) (n (1- e)))
			(:instance bitn-0-1 (x (bits (olamt a b e) e 2)) (n k))
;			(:instance bits< (x (olamz a b e)) (i (- e 2)) (j 0))
			(:instance bitn-0-1 (x (lnot (bits (olamz a b e) (- e 2) 0) (1- e))) (n k))
			(:instance bitn-lnot-not-equal (x (bits (olamz a b e) (- e 2) 0)) (n (1- e)))
			(:instance bitn-0-1 (x (bits (olamz a b e) (- e 2) 0)) (n k)))))))

(local (defthm bitn-olam2
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam2 a b e) k) 1)
		  (and (= (bitn (olamt a b e) (+ 2 k)) 0)
		       (= (bitn (olamz a b e) (+ 1 k)) 1)
		       (= (bitn (olamz a b e) k) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable olam2 bits logand)
		  :use (;olamz-nat
			;olamt-nat
			bitn-olam2-3
;			(:instance bits< (x (olamz a b e)) (i (- e 2)) (j 0))
			(:instance bitn-0-1 (x (olamz a b e)) (n k))
;			(:instance bitn-bits (x (olamt a b e)) (i e) (j 2))
	;		(:instance bitn-bits (x (olamz a b e)) (i (- e 1)) (j 1))
		;	(:instance bitn-bits (x (olamz a b e)) (i (- e 2)) (j 0))
                        )))))

(defthm OLAM2-NAT
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
;		  (> b 0)
	;	  (not (= a b))
		;  (= e (expo a))
		 ; (= e (expo b))
;		  (> e 1)
	;	  (integerp k)
		;  (<= k (- e 2))
		 ; (>= k 0)
                  )
	     (and (integerp (olam2 a b e))
		  (>= (olam2 a b e) 0)))
;  :rule-classes ()
  :hints (("Goal" :in-theory (disable bits logand)
		  :use (;olamz-nat
			;olamt-nat
;			(:instance bits< (x (olamt a b e)) (i e) (j 2))
	;		(:instance bits< (x (olamz a b e)) (i (- e 2)) (j 0))
))))

(local (defthm bitn-olam4-1
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (>= k 0))
	     (equal (bitn (olam4 a b e) k)
		    (logand (bitn (lnot (bits (olamt a b e) e 2) (1- e)) k)
			    (logand (bitn (bits (olamg a b e) (1- e) 1) k)
				    (bitn (lnot (bits (olamg a b e) (- e 2) 0) (1- e)) k)))))
  :hints (("Goal" :in-theory (enable olam4)))))

(local (defthm bitn-olam4-2
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (>= k 0))
	     (iff (= (bitn (olam4 a b e) k) 1)
		  (and (= (bitn (lnot (bits (olamt a b e) e 2) (1- e)) k) 1)
		       (= (bitn (bits (olamg a b e) (1- e) 1) k) 1)
		       (= (bitn (lnot (bits (olamg a b e) (- e 2) 0) (1- e)) k) 1))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable bits olam4 logand)
		  :use ((:instance bitn-0-1 (x (lnot (bits (olamt a b e) e 2) (1- e))) (n k))
			(:instance bitn-0-1 (x (bits (olamg a b e) (1- e) 1)) (n k))
			(:instance bitn-0-1 (x (lnot (bits (olamg a b e) (- e 2) 0) (1- e))) (n k)))))))

(local (in-theory (disable bitn-olam4-1)))

(local (defthm bitn-olam4-3
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam4 a b e) k) 1)
		  (and (= (bitn (bits (olamt a b e) e 2) k) 0)
		       (= (bitn (bits (olamg a b e) (1- e) 1) k) 1)
		       (= (bitn (bits (olamg a b e) (- e 2) 0) k) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable olam4 bits lnot logand)
		  :use (;olamg-nat
			;olamt-nat
			bitn-olam4-2
;			(:instance bits< (x (olamt a b e)) (i e) (j 2))
			(:instance bitn-0-1 (x (lnot (bits (olamt a b e) e 2) (1- e))) (n k))
			(:instance bitn-lnot-not-equal (x (bits (olamt a b e) e 2)) (n (1- e)))
			(:instance bitn-0-1 (x (bits (olamt a b e) e 2)) (n k))
;			(:instance bits< (x (olamg a b e)) (i (- e 2)) (j 0))
			(:instance bitn-0-1 (x (lnot (bits (olamg a b e) (- e 2) 0) (1- e))) (n k))
			(:instance bitn-lnot-not-equal (x (bits (olamg a b e) (- e 2) 0)) (n (1- e)))
			(:instance bitn-0-1 (x (bits (olamg a b e) (- e 2) 0)) (n k)))))))

(local (defthm bitn-olam4
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam4 a b e) k) 1)
		  (and (= (bitn (olamt a b e) (+ 2 k)) 0)
		       (= (bitn (olamg a b e) (+ 1 k)) 1)
		       (= (bitn (olamg a b e) k) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable olam4 bits logand)
		  :use (;olamg-nat
			;olamt-nat
			bitn-olam4-3
;			(:instance bits< (x (olamg a b e)) (i (- e 2)) (j 0))
			(:instance bitn-0-1 (x (olamg a b e)) (n k))
;			(:instance bitn-bits (x (olamt a b e)) (i e) (j 2))
	;		(:instance bitn-bits (x (olamg a b e)) (i (- e 1)) (j 1))
		;	(:instance bitn-bits (x (olamg a b e)) (i (- e 2)) (j 0))
                        )))))

(defthm OLAM4-NAT
    (implies (and (integerp a)
		  (> a 0)
                  )
	     (and (integerp (olam4 a b e))
		  (>= (olam4 a b e) 0)))
;  :rule-classes ()
)

(local (defthm bitn-olam0-1
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (equal (bitn (olam0 a b e) k)
		    (logior (bitn (olam1 a b e) k)
			    (logior (bitn (olam2 a b e) k)
				    (logior (bitn (olam3 a b e) k)
					    (bitn (olam4 a b e) k))))))
  :hints (("Goal" :in-theory (enable olam0)))))

(local (defthm bitn-olam0-2
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam0 a b e) k) 1)
		  (or (= (bitn (olam1 a b e) k) 1)
		      (= (bitn (olam2 a b e) k) 1)
		      (= (bitn (olam3 a b e) k) 1)
		      (= (bitn (olam4 a b e) k) 1))))
  :hints (("Goal" :in-theory (disable BITN-KNOWN-NOT-0-REPLACE-WITH-1)
		  :use (;olam1-nat
			;olam2-nat
			;olam3-nat
			;olam4-nat
			(:instance bitn-0-1 (x (olam1 a b e)) (n k))
			(:instance bitn-0-1 (x (olam2 a b e)) (n k))
			(:instance bitn-0-1 (x (olam3 a b e)) (n k))
			(:instance bitn-0-1 (x (olam4 a b e)) (n k)))))))

(local (in-theory (disable bitn-olam0-1)))

(local (defthm bitn-olam0-3
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam0 a b e) k) 1)
		  (or (and (= (bitn (olamt a b e) (+ 2 k)) 1)
			   (= (bitn (olamg a b e) (+ 1 k)) 1)
			   (= (bitn (olamz a b e) k) 0))
		      (and (= (bitn (olamt a b e) (+ 2 k)) 0)
			   (= (bitn (olamz a b e) (+ 1 k)) 1)
			   (= (bitn (olamz a b e) k) 0))
		      (and (= (bitn (olamt a b e) (+ 2 k)) 1)
			   (= (bitn (olamz a b e) (+ 1 k)) 1)
			   (= (bitn (olamg a b e) k) 0))
		      (and (= (bitn (olamt a b e) (+ 2 k)) 0)
			   (= (bitn (olamg a b e) (+ 1 k)) 1)
			   (= (bitn (olamg a b e) k) 0)))))
  :hints (("Goal" :in-theory (disable olam0)
		  :use (bitn-olam0-2 bitn-olam1 bitn-olam2 bitn-olam3 bitn-olam4)))))

(local (defthm bitn-olam0-4
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam0 a b e) k) 1)
		  (or (and (= (bitn (olamt a b e) (+ 2 k)) 1)
			   (= (bitn (olamg a b e) (+ 1 k)) 1)
			   (not (= (bitn (olamz a b e) k) 1)))
		      (and (not (= (bitn (olamt a b e) (+ 2 k)) 1))
			   (= (bitn (olamz a b e) (+ 1 k)) 1)
			   (not (= (bitn (olamz a b e) k) 1)))
		      (and (= (bitn (olamt a b e) (+ 2 k)) 1)
			   (= (bitn (olamz a b e) (+ 1 k)) 1)
			   (not (= (bitn (olamg a b e) k) 1)))
		      (and (not (= (bitn (olamt a b e) (+ 2 k)) 1))
			   (= (bitn (olamg a b e) (+ 1 k)) 1)
			   (not (= (bitn (olamg a b e) k) 1))))))
  :hints (("Goal" :in-theory (disable olam0)
		  :use (bitn-olam0-3
			(:instance bitn-0-1 (x (olamz a b e)) (n k))
			(:instance bitn-0-1 (x (olamg a b e)) (n k))
			(:instance bitn-0-1 (x (olamt a b e)) (n (+ 2 k))))))))

(local (defthm c-0-1
         (or (= (c k a b) 0)
             (= (c k a b) 1)
             (= (c k a b) -1))
         :rule-classes ()
         :hints (("Goal" :in-theory (enable c)
                  :use ((:instance bitn-0-1 (x a) (n k))
			(:instance bitn-0-1 (x b) (n k)))))))

(local (defthm bitn-olam0
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (iff (= (bitn (olam0 a b e) k) 1)
		  (and (not (= (c (1+ k) a b) 0))
		       (not (= (c (1+ k) a b)
			       (if (= (c (+ 2 k) a b) 0)
				   (- (c k a b))
				 (c k a b)))))))
  :hints (("Goal" :in-theory (disable c olam0
                                      EXPO-COMPARISON-REWRITE-TO-BOUND-2
                                      EXPO-COMPARISON-REWRITE-TO-BOUND)
		  :use (bitn-olam0-4
			bitn-olamg
			bitn-olamz
			c-0-1
			(:instance c-0-1 (k (+ 1 k)))
			(:instance c-0-1 (k (+ 2 k)))
			(:instance bitn-olamt (k (+ 2 k)))
			(:instance bitn-olamz (k (1+ k)))
			(:instance bitn-olamg (k (1+ k))))))))

(defthm OLAM0-NAT
    (implies (and (integerp a)
		  (> a 0)
                  )
	     (and (integerp (olam0 a b e))
		  (>= (olam0 a b e) 0)))
;  :rule-classes ()
  :hints (("Goal" :in-theory (disable logior))))

(defthm OLAMB-NAT
    (implies (and (integerp a)
		  (> a 0)
;		  (integerp b)
	;	  (> b 0)
		;  (not (= a b))
		 ; (= e (expo a))
		 ; (= e (expo b))
		 ; (> e 1)
                  )
	     (and (integerp (olamb a b e))
		  (>= (olamb a b e) 0)))
;  :rule-classes ()
  :hints (("Goal" :use (;olam0-nat
			(:instance bitn-0-1 (x (olamt a b e)) (n 0))))))

(local (defthm bitn-olamb-1
         (implies (and (integerp a)
                       (> a 0)
                       (integerp b)
                       (> b 0)
                       (not (= a b))
                       (= e (expo a))
                       (= e (expo b))
                       (> e 1))
                  (= (fl (/ (olamb a b e) 2))
                     (olam0 a b e)))
         :hints (("Goal" :in-theory (enable olamb)
                  :use ( ;olam0-nat
			(:instance fl-unique (x (/ (olamb a b e) 2)) (n (olam0 a b e)))
			(:instance bitn-0-1 (x (olamt a b e)) (n 0)))))))


(local (defthm bitn-olamb-2
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp n)
		  (> n 0))
	     (= (bitn (olamb a b e) n)
		(bitn (olam0 a b e) (1- n))))
  :hints (("Goal" :use (;olamb-nat
			bitn-olamb-1
			(:instance bitn-rec-pos (x (olamb a b e)) (n n)))))))

(local (defthm bitn-olamb
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp n)
		  (<= n (- e 1))
		  (> n 0))
	     (iff (= (bitn (olamb a b e) n) 1)
		  (and (not (= (c n a b) 0))
		       (not (= (c n a b)
			       (if (= (c (+ n 1) a b) 0)
				   (- (c (1- n) a b))
				 (c (1- n) a b)))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable EXPO-COMPARISON-REWRITE-TO-BOUND
                                      EXPO-COMPARISON-REWRITE-TO-BOUND-2)
		  :use (;bitn-olamb-2
			(:instance bitn-olam0 (k (1- n))))))))


(local (defthm lop3-1
    (implies (and (integerp a)
		  (>= a 0)
		  (integerp b)
		  (>= b 0)
		  (integerp n)
		  (> n 0)
		  (= (mod a (expt 2 n)) (mod b (expt 2 n)))
		  (= (bitn a n) (bitn b n)))
	     (= (mod a (expt 2 (1+ n)))
		(mod b (expt 2 (1+ n)))))
  :rule-classes ()
  :hints (("Goal" :use (mod-n+1
			(:instance mod-n+1 (a b)))))))

(local (defthm lop3-2
    (implies (and (integerp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0)
		  (= (bitn x n) 1))
	     (= (expo (mod x (expt 2 (1+ n))))
		n))
  :rule-classes ()
  :hints (("Goal" :use (;(:instance mod>=0 (m x) (n (expt 2 (1+ n))))
			(:instance mod-bnd-1 (m x) (n (expt 2 (1+ n))))
			(:instance expo-unique (x (mod x (expt 2 (1+ n)))))
			(:instance bitn-mod (n (1+ n)) (k n))
			(:instance bit-expo-a (x (mod x (expt 2 (1+ n))))))))))

(local (defthm lop3-3
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (= (c n a b) 0)
			 (IMPLIES (AND (INTEGERP A)
				       (< 0 A)
				       (INTEGERP B)
				       (< 0 B)
				       (= E (EXPO A))
				       (= E (EXPO B))
				       (< 0 E)
				       (INTEGERP (+ -1 N))
				       (<= 0 (+ -1 N))
				       (< (+ -1 N) E)
				       (NOT (= (MOD A (EXPT 2 (+ 1 -1 N)))
					       (MOD B (EXPT 2 (+ 1 -1 N))))))
				  (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))) 0))
				       (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))))
					  (IF (OR (= (C (+ -1 N) A B) 0)
						  (= (C (+ 1 -1 N) A B) 0))
					      (LOP A B (C (+ -1 N) A B) (+ -1 N))
					      (LOP A B (- (C (+ -1 N) A B))
						   (+ -1 N))))))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (MOD (OLAMB A B E) (EXPT 2 N)) 0))
			 (= (EXPO (MOD (OLAMB A B E) (EXPT 2 N)))
			    (lop a b (c (1- n) a b) (1- n)))))
	 :rule-classes ()
	 :hints (("Goal" :in-theory (e/d ( lop c) (EXPO-COMPARISON-REWRITE-TO-BOUND-2))
                  :use (lop3-1)))))

(local (defthm lop3-4
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (= (c n a b) 0)
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (bitn (olamb a b e) n) 0))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable EXPO-COMPARISON-REWRITE-TO-BOUND-2)
           :use (bitn-olamb
			(:instance bitn-0-1 (x (olamb a b e))))))))

(local (defthm lop3-5
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (= (c n a b) 0)
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (mod (olamb a b e) (expt 2 (1+ n)))
		(mod (olamb a b e) (expt 2 n))))
  :rule-classes ()
  :hints (("Goal" :use (lop3-4
			;olamb-nat
			(:instance mod-n+1 (a (olamb a b e))))))))

(local (defthm lop3-6
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (= (c n a b) 0)
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (lop a b (c n a b) n)
		(lop a b (c (1- n) a b) (1- n))))
    :hints (("Goal" :in-theory (enable lop c)))
  :rule-classes ()))

(local (defthm lop3-7
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (= (c n a b) 0)
			 (IMPLIES (AND (INTEGERP A)
				       (< 0 A)
				       (INTEGERP B)
				       (< 0 B)
				       (= E (EXPO A))
				       (= E (EXPO B))
				       (< 0 E)
				       (INTEGERP (+ -1 N))
				       (<= 0 (+ -1 N))
				       (< (+ -1 N) E)
				       (NOT (= (MOD A (EXPT 2 (+ 1 -1 N)))
					       (MOD B (EXPT 2 (+ 1 -1 N))))))
				  (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))) 0))
				       (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))))
					  (IF (OR (= (C (+ -1 N) A B) 0)
						  (= (C (+ 1 -1 N) A B) 0))
					      (LOP A B (C (+ -1 N) A B) (+ -1 N))
					      (LOP A B (- (C (+ -1 N) A B))
						   (+ -1 N))))))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 N))) 0))
			 (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 N))))
			    (IF (OR (= (C N A B) 0)
				    (= (C (+ 1 N) A B) 0))
				(LOP A B (C N A B) N)
				(LOP A B (- (C N A B)) N)))))
	 :rule-classes ()
	 :hints (("Goal" :in-theory (disable lop)
			 :use (lop3-3 lop3-5 lop3-6)))))

(local (defthm lop3-8
    (implies (and (integerp a)
		  (>= a 0)
		  (integerp b)
		  (>= b 0)
		  (integerp n)
		  (>= n 0)
		  (= (mod a (expt 2 (1+ n))) (mod b (expt 2 (1+ n)))))
	     (= (bitn a n) (bitn b n)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bitn-mod (x a) (n (1+ n)) (k n))
			(:instance bitn-mod (x b) (n (1+ n)) (k n)))))))

(local (defthm lop3-8-1
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (integerp n)
		  (> n 0)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (not (= (mod (olamb a b e) (expt 2 n)) 0)))
	     (not (= (mod (olamb a b e) (expt 2 (1+ n))) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable olamb)
		  :use (;olamb-nat
			(:instance mod-does-nothing (m 0) (n (expt 2 n)))
			(:instance mod-of-mod-cor (x (olamb a b e)) (a (1+ n)) (b n)))))))

(local (defthm lop3-9
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (not (= (c n a b) 0))
			 (= (c (1+ n) a b) 0)
			 (= (c (1- n) a b) (- (c n a b)))
			 (IMPLIES (AND (INTEGERP A)
				       (< 0 A)
				       (INTEGERP B)
				       (< 0 B)
				       (= E (EXPO A))
				       (= E (EXPO B))
				       (< 0 E)
				       (INTEGERP (+ -1 N))
				       (<= 0 (+ -1 N))
				       (< (+ -1 N) E)
				       (NOT (= (MOD A (EXPT 2 (+ 1 -1 N)))
					       (MOD B (EXPT 2 (+ 1 -1 N))))))
				  (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))) 0))
				       (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))))
					  (IF (OR (= (C (+ -1 N) A B) 0)
						  (= (C (+ 1 -1 N) A B) 0))
					      (LOP A B (C (+ -1 N) A B) (+ -1 N))
					      (LOP A B (- (C (+ -1 N) A B))
						   (+ -1 N))))))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 N))) 0))
			 (= (EXPO (MOD (OLAMB A B E) (EXPT 2 N)))
			    (lop a b (- (c (1- n) a b)) (1- n)))))
	 :rule-classes ()
	 :hints (("Goal" :in-theory (e/d ( lop c) (EXPO-COMPARISON-REWRITE-TO-BOUND-2))
                  :use (lop3-8-1
			       (:instance lop3-8 (n (1- n))))))))

(local (defthm lop3-10
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (= (c (1+ n) a b) 0)
		  (= (c (1- n) a b) (- (c n a b)))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (bitn (olamb a b e) n) 0))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable EXPO-COMPARISON-REWRITE-TO-BOUND-2)
           :use (bitn-olamb
			(:instance bitn-0-1 (x (olamb a b e))))))))

(local (defthm lop3-11
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (= (c (1+ n) a b) 0)
		  (= (c (1- n) a b) (- (c n a b)))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (mod (olamb a b e) (expt 2 (1+ n)))
		(mod (olamb a b e) (expt 2 n))))
  :rule-classes ()
  :hints (("Goal" :use (lop3-10
			;olamb-nat
			(:instance mod-n+1 (a (olamb a b e))))))))

(local (defthm lop3-12
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (= (c (1+ n) a b) 0)
		  (= (c (1- n) a b) (- (c n a b)))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (lop a b (- (c (1- n) a b)) n)
		(lop a b (- (c (1- n) a b)) (1- n))))
    :hints (("Goal" :in-theory (enable lop c)))
  :rule-classes ()))

(local (defthm lop3-13
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (= (c (1+ n) a b) 0)
		  (= (c (1- n) a b) (- (c n a b)))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (INTEGERP N)
		  (<= 0 N))
	     (EQUAL (LOP A B (* -1 (C (+ -1 N) A B)) (+ -1 N))
		    (LOP A B (C N A B) N)))
    :hints (("Goal" :in-theory (enable lop c)))))

(local (defthm lop3-14
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (not (= (c n a b) 0))
			 (= (c (1+ n) a b) 0)
			 (= (c (1- n) a b) (- (c n a b)))
			 (IMPLIES (AND (INTEGERP A)
				       (< 0 A)
				       (INTEGERP B)
				       (< 0 B)
				       (= E (EXPO A))
				       (= E (EXPO B))
				       (< 0 E)
				       (INTEGERP (+ -1 N))
				       (<= 0 (+ -1 N))
				       (< (+ -1 N) E)
				       (NOT (= (MOD A (EXPT 2 (+ 1 -1 N)))
					       (MOD B (EXPT 2 (+ 1 -1 N))))))
				  (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))) 0))
				       (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))))
					  (IF (OR (= (C (+ -1 N) A B) 0)
						  (= (C (+ 1 -1 N) A B) 0))
					      (LOP A B (C (+ -1 N) A B) (+ -1 N))
					      (LOP A B (- (C (+ -1 N) A B))
						   (+ -1 N))))))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 N))) 0))
			 (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 N))))
			    (IF (OR (= (C N A B) 0)
				    (= (C (+ 1 N) A B) 0))
				(LOP A B (C N A B) N)
				(LOP A B (- (C N A B)) N)))))
	 :rule-classes ()
	 :hints (("Goal" :use (lop3-9 lop3-11 lop3-12)))))

(local (defthm lop3-15
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (= (c (1+ n) a b) 0)
		  (not (= (c (1- n) a b) (- (c n a b))))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (bitn (olamb a b e) n) 1))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable EXPO-COMPARISON-REWRITE-TO-BOUND-2)
           :use (bitn-olamb)))))

(local (defthm lop3-16
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (= (c (1+ n) a b) 0)
		  (not (= (c (1- n) a b) (- (c n a b))))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (and (not (= (mod (olamb a b e) (expt 2 (1+ n))) 0))
		  (= (expo (mod (olamb a b e) (expt 2 (1+ n)))) n)))
  :rule-classes ()
  :hints (("Goal" :use (lop3-15
			;olamb-nat
			(:instance lop3-2 (x (olamb a b e))))))))

(local (defthm lop3-17
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (= (c (1+ n) a b) 0)
		  (not (= (c (1- n) a b) (- (c n a b))))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (lop a b (c n a b) n) n))
    :hints (("Goal" :in-theory (enable lop c)))
  :rule-classes ()))

(local (defthm lop3-18
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (not (= (c n a b) 0))
			 (= (c (1+ n) a b) 0)
			 (not (= (c (1- n) a b) (- (c n a b))))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (mod (olamb a b e) (expt 2 (1+ n))) 0))
			 (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 N))))
			    (IF (OR (= (C N A B) 0)
				    (= (C (+ 1 N) A B) 0))
				(LOP A B (C N A B) N)
				(LOP A B (- (C N A B)) N)))))
	 :rule-classes ()
	 :hints (("Goal" :in-theory (disable lop c)
			 :use (lop3-16 lop3-17)))))

(local (defthm lop3-19
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (not (= (c n a b) 0))
			 (= (c (1+ n) a b) 0)
			 (IMPLIES (AND (INTEGERP A)
				       (< 0 A)
				       (INTEGERP B)
				       (< 0 B)
				       (= E (EXPO A))
				       (= E (EXPO B))
				       (< 0 E)
				       (INTEGERP (+ -1 N))
				       (<= 0 (+ -1 N))
				       (< (+ -1 N) E)
				       (NOT (= (MOD A (EXPT 2 (+ 1 -1 N)))
					       (MOD B (EXPT 2 (+ 1 -1 N))))))
				  (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))) 0))
				       (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))))
					  (IF (OR (= (C (+ -1 N) A B) 0)
						  (= (C (+ 1 -1 N) A B) 0))
					      (LOP A B (C (+ -1 N) A B) (+ -1 N))
					      (LOP A B (- (C (+ -1 N) A B))
						   (+ -1 N))))))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (mod (olamb a b e) (expt 2 (1+ n))) 0))
			 (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 N))))
			    (IF (OR (= (C N A B) 0)
				    (= (C (+ 1 N) A B) 0))
				(LOP A B (C N A B) N)
				(LOP A B (- (C N A B)) N)))))
	 :rule-classes ()
	 :hints (("Goal" :in-theory (disable lop c)
			 :use (lop3-14 lop3-18)))))

(local (defthm lop3-20
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (not (= (c n a b) 0))
			 (not (= (c (1+ n) a b) 0))
			 (= (c (1- n) a b) (c n a b))
			 (IMPLIES (AND (INTEGERP A)
				       (< 0 A)
				       (INTEGERP B)
				       (< 0 B)
				       (= E (EXPO A))
				       (= E (EXPO B))
				       (< 0 E)
				       (INTEGERP (+ -1 N))
				       (<= 0 (+ -1 N))
				       (< (+ -1 N) E)
				       (NOT (= (MOD A (EXPT 2 (+ 1 -1 N)))
					       (MOD B (EXPT 2 (+ 1 -1 N))))))
				  (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))) 0))
				       (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))))
					  (IF (OR (= (C (+ -1 N) A B) 0)
						  (= (C (+ 1 -1 N) A B) 0))
					      (LOP A B (C (+ -1 N) A B) (+ -1 N))
					      (LOP A B (- (C (+ -1 N) A B))
						   (+ -1 N))))))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (mod (olamb a b e) (expt 2 (1+ n))) 0))
			 (= (EXPO (MOD (OLAMB A B E) (EXPT 2 N)))
			    (lop a b (- (c (1- n) a b)) (1- n)))))
	 :rule-classes ()
	 :hints (("Goal" :in-theory (e/d ( lop c) (EXPO-COMPARISON-REWRITE-TO-BOUND-2))
                  :use (lop3-8-1
			       (:instance lop3-8 (n (1- n))))))))

(local (in-theory (disable EXPO-COMPARISON-REWRITE-TO-BOUND-2))) ;yuck

(local (defthm lop3-21
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (not (= (c (1+ n) a b) 0))
		  (= (c (1- n) a b) (c n a b))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (bitn (olamb a b e) n) 0))
  :rule-classes ()
  :hints (("Goal" :use (bitn-olamb
			(:instance bitn-0-1 (x (olamb a b e))))))))

(local (defthm lop3-22
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (not (= (c (1+ n) a b) 0))
		  (= (c (1- n) a b) (c n a b))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (mod (olamb a b e) (expt 2 (1+ n)))
		(mod (olamb a b e) (expt 2 n))))
  :rule-classes ()
  :hints (("Goal" :use (lop3-21
			;olamb-nat
			(:instance mod-n+1 (a (olamb a b e))))))))

(local (defthm lop3-23
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (not (= (c (1+ n) a b) 0))
		  (= (c (1- n) a b) (c n a b))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (lop a b (- (c (1- n) a b)) n)
		(lop a b (- (c (1- n) a b)) (1- n))))
    :hints (("Goal" :in-theory (enable lop c)))
  :rule-classes ()))

(local (defthm lop3-24
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (not (= (c (1+ n) a b) 0))
		  (= (c (1- n) a b) (c n a b))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (INTEGERP N)
		  (<= 0 N))
	     (EQUAL (LOP A B (* -1 (C (+ -1 N) A B)) (+ -1 N))
		    (LOP A B (- (C N A B)) N)))
    :hints (("Goal" :in-theory (enable lop c)))))

(local (defthm lop3-25
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (not (= (c n a b) 0))
			 (not (= (c (1+ n) a b) 0))
			 (= (c (1- n) a b) (c n a b))
			 (IMPLIES (AND (INTEGERP A)
				       (< 0 A)
				       (INTEGERP B)
				       (< 0 B)
				       (= E (EXPO A))
				       (= E (EXPO B))
				       (< 0 E)
				       (INTEGERP (+ -1 N))
				       (<= 0 (+ -1 N))
				       (< (+ -1 N) E)
				       (NOT (= (MOD A (EXPT 2 (+ 1 -1 N)))
					       (MOD B (EXPT 2 (+ 1 -1 N))))))
				  (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))) 0))
				       (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))))
					  (IF (OR (= (C (+ -1 N) A B) 0)
						  (= (C (+ 1 -1 N) A B) 0))
					      (LOP A B (C (+ -1 N) A B) (+ -1 N))
					      (LOP A B (- (C (+ -1 N) A B))
						   (+ -1 N))))))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (mod (olamb a b e) (expt 2 (1+ n))) 0))
			 (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 N))))
			    (IF (OR (= (C N A B) 0)
				    (= (C (+ 1 N) A B) 0))
				(LOP A B (C N A B) N)
				(LOP A B (- (C N A B)) N)))))
	 :rule-classes ()
	 :hints (("Goal" :in-theory (disable lop c)
			 :use (lop3-20 lop3-22 lop3-23)))))

(local (defthm lop3-26
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (not (= (c (1+ n) a b) 0))
		  (not (= (c (1- n) a b) (c n a b)))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (bitn (olamb a b e) n) 1))
  :rule-classes ()
  :hints (("Goal" :use (bitn-olamb)))))

(local (defthm lop3-27
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (not (= (c n a b) 0))
			 (not (= (c (1+ n) a b) 0))
			 (not (= (c (1- n) a b) (c n a b)))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (mod (olamb a b e) (expt 2 (1+ n))) 0))
			 (= (expo (mod (olamb a b e) (expt 2 (1+ n)))) n)))
	 :rule-classes ()
	 :hints (("Goal" :use (lop3-26
			       ;olamb-nat
			       (:instance lop3-2 (x (olamb a b e))))))))

(local (defthm lop3-28
    (IMPLIES (AND (INTEGERP N)
		  (< 0 N)
		  (not (= (c n a b) 0))
		  (not (= (c (1+ n) a b) 0))
		  (not (= (c (1- n) a b) (c n a b)))
		  (INTEGERP A)
		  (< 0 A)
		  (INTEGERP B)
		  (< 0 B)
		  (= E (EXPO A))
		  (= E (EXPO B))
		  (< 0 E)
		  (INTEGERP N)
		  (<= 0 N)
		  (< N E)
		  (NOT (= (MOD A (EXPT 2 (+ 1 N)))
			  (MOD B (EXPT 2 (+ 1 N))))))
	     (= (lop a b (- (c n a b)) n) n))
    :hints (("Goal" :in-theory (enable lop c)))
    :rule-classes ()))

(local (defthm lop3-29
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (not (= (c n a b) 0))
			 (not (= (c (1+ n) a b) 0))
			 (not (= (c (1- n) a b) (c n a b)))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (mod (olamb a b e) (expt 2 (1+ n))) 0))
			 (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 N))))
			    (IF (OR (= (C N A B) 0)
				    (= (C (+ 1 N) A B) 0))
				(LOP A B (C N A B) N)
				(LOP A B (- (C N A B)) N)))))
	 :rule-classes ()
	 :hints (("Goal" :in-theory (disable lop c)
			 :use (lop3-27 lop3-28)))))

(local (defthm lop3-30
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (not (= (c n a b) 0))
			 (not (= (c (1+ n) a b) 0))
			 (IMPLIES (AND (INTEGERP A)
				       (< 0 A)
				       (INTEGERP B)
				       (< 0 B)
				       (= E (EXPO A))
				       (= E (EXPO B))
				       (< 0 E)
				       (INTEGERP (+ -1 N))
				       (<= 0 (+ -1 N))
				       (< (+ -1 N) E)
				       (NOT (= (MOD A (EXPT 2 (+ 1 -1 N)))
					       (MOD B (EXPT 2 (+ 1 -1 N))))))
				  (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))) 0))
				       (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))))
					  (IF (OR (= (C (+ -1 N) A B) 0)
						  (= (C (+ 1 -1 N) A B) 0))
					      (LOP A B (C (+ -1 N) A B) (+ -1 N))
					      (LOP A B (- (C (+ -1 N) A B))
						   (+ -1 N))))))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (mod (olamb a b e) (expt 2 (1+ n))) 0))
			 (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 N))))
			    (IF (OR (= (C N A B) 0)
				    (= (C (+ 1 N) A B) 0))
				(LOP A B (C N A B) N)
				(LOP A B (- (C N A B)) N)))))
	 :rule-classes ()
	 :hints (("Goal" :in-theory (disable lop c)
			 :use (lop3-25 lop3-29)))))

(local (defthm lop3-31
	   (IMPLIES (AND (INTEGERP N)
			 (< 0 N)
			 (IMPLIES (AND (INTEGERP A)
				       (< 0 A)
				       (INTEGERP B)
				       (< 0 B)
				       (= E (EXPO A))
				       (= E (EXPO B))
				       (< 0 E)
				       (INTEGERP (+ -1 N))
				       (<= 0 (+ -1 N))
				       (< (+ -1 N) E)
				       (NOT (= (MOD A (EXPT 2 (+ 1 -1 N)))
					       (MOD B (EXPT 2 (+ 1 -1 N))))))
				  (and (not (= (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))) 0))
				       (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 -1 N))))
					  (IF (OR (= (C (+ -1 N) A B) 0)
						  (= (C (+ 1 -1 N) A B) 0))
					      (LOP A B (C (+ -1 N) A B) (+ -1 N))
					      (LOP A B (- (C (+ -1 N) A B))
						   (+ -1 N))))))
			 (INTEGERP A)
			 (< 0 A)
			 (INTEGERP B)
			 (< 0 B)
			 (= E (EXPO A))
			 (= E (EXPO B))
			 (< 0 E)
			 (INTEGERP N)
			 (<= 0 N)
			 (< N E)
			 (NOT (= (MOD A (EXPT 2 (+ 1 N)))
				 (MOD B (EXPT 2 (+ 1 N))))))
		    (and (not (= (mod (olamb a b e) (expt 2 (1+ n))) 0))
			 (= (EXPO (MOD (OLAMB A B E) (EXPT 2 (+ 1 N))))
			    (IF (OR (= (C N A B) 0)
				    (= (C (+ 1 N) A B) 0))
				(LOP A B (C N A B) N)
				(LOP A B (- (C N A B)) N)))))
	 :rule-classes ()
	 :hints (("Goal" :use (lop3-30 lop3-19
                                       lop3-7)))))

(local (defthm lop3-32
    (implies (and (integerp x)
		  (> x 0)
		  (= (expo x) 1))
	     (or (= x 2) (= x 3)))
  :rule-classes ()
  :hints (("Goal" :use (expo-upper-bound expo-lower-bound)))))



(local (defthm lop3-33
	   (implies (and (integerp a)
			 (> a 0)
			 (integerp b)
			 (> b 0)
			 (= e (expo a))
			 (= e (expo b))
			 (> e 0)
			 (not (= (mod a 2) (mod b 2))))
		    (equal (expo (mod (olamb a b e) 2)) 0))
	 :hints (("Goal" :use (;olamb-natural
			       (:instance mod012 (m (olamb a b e))))))))

(local (in-theory (disable BITN-KNOWN-NOT-0-REPLACE-WITH-1)))

(local (defthm lop3-33-1
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
	          (not (= (bitn a 0) (bitn b 0))))
	     (= (bitn a 0) (bitn (lnot b (1+ e)) 0)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance expo-upper-bound (x b))
			(:instance bitn-lnot-not-equal (x b) (n (1+ e)) (k 0))
			(:instance bitn-0-1 (x a) (n 0))
			(:instance bitn-0-1 (x b) (n 0))
			(:instance bitn-0-1 (x (lnot b (1+ e))) (n 0)))))))

(local (defthm lop3-33-2
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
	          (not (= (mod a 2) (mod b 2))))
	     (= (bitn a 0) (bitn (lnot b (1+ e)) 0)))
  :rule-classes ()
  :hints (("Goal" :use (lop3-33-1
			(:instance bitn-def (x a) (n 0))
			(:instance bitn-def (x b) (n 0)))))))

(local (defthm lop3-33-3
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
	          (not (= (mod a 2) (mod b 2))))
	     (= (bitn (olamt a b e) 0) 0))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable olamt)
		  :use (lop3-33-2
			(:instance expo-upper-bound (x b))
			(:instance bitn-logxor (x a) (y (lnot b (1+ e))) (n 0))
			(:instance bitn-0-1 (x a) (n 0)))))))

(local (defthm lop3-33-4
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
	          (not (= (mod a 2) (mod b 2))))
	     (not (= (mod (olamb a b e) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable olamb)
		  :use (lop3-33-3
;			olam0-nat
;			(:instance mod+-thm (m 1) (n 2) (a (olam0 a b e)))
                             )))))

(local (defthm lop3-34
	   (implies (and (integerp a)
			 (> a 0)
			 (integerp b)
			 (> b 0)
			 (= e (expo a))
			 (= e (expo b))
			 (> e 1)
			 (integerp n)
			 (>= n 0)
			 (< n e)
			 (not (= (mod a (expt 2 (1+ n)))
				 (mod b (expt 2 (1+ n))))))
		    (and (not (= (mod (olamb a b e) (expt 2 (1+ n))) 0))
			 (= (expo (mod (olamb a b e) (expt 2 (1+ n))))
			    (if (or (= (c n a b) 0)
				    (= (c (1+ n) a b) 0))
				(lop a b (c n a b) n)
			      (lop a b (- (c n a b)) n)))))
	 :rule-classes ()
	 :hints (("Goal" :in-theory (e/d (lop) (olamb))
			 :induct (natp-induct n))
		 ("Subgoal *1/2"; :in-theory (disable lop c)
				 :use (lop3-31))
		 ("Subgoal *1/1" :use lop3-33-4))))

(defthm OLAM1-BND
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (< (olam1 a b e) (expt 2 (- e 1))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable olam1)
		  :use (;olamz-nat
			;olamt-nat
			;olamg-nat
;			(:instance lnot-bnds (x (bits (olamz a b e) (- e 2) 0)) (n (1- e)))
;			(:instance bits< (x (olamt a b e)) (i e) (j 2))
	;		(:instance bits< (x (olamz a b e)) (i (- e 2)) (j 0))
			(:instance logand-bnd
				   (x (bits (olamt a b e) e 2))
				   (y (logand (bits (olamg a b e) (1- e) 1)
					      (lnot (bits (olamz a b e) (- e 2) 0) (1- e)))))))))

(defthm OLAM3-BND
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (< (olam3 a b e) (expt 2 (- e 1))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable olam3)
		  :use (;olamg-nat
			;olamt-nat
			;olamz-nat
;			(:instance lnot-bnds (x (bits (olamg a b e) (- e 2) 0)) (n (1- e)))
;			(:instance bits< (x (olamt a b e)) (i e) (j 2))
	;		(:instance bits< (x (olamg a b e)) (i (- e 2)) (j 0))
			(:instance logand-bnd ;and-dist-a
				   (x (bits (olamt a b e) e 2))
				   (y (logand (bits (olamz a b e) (1- e) 1)
					      (lnot (bits (olamg a b e) (- e 2) 0) (1- e)))))
))))

(defthm OLAM2-BND
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (< (olam2 a b e) (expt 2 (- e 1))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable olam2)
		  :use (;olamz-nat
			;olamt-nat
;			(:instance lnot-bnds (x (bits (olamt a b e) e 2)) (n (1- e)))
	;		(:instance lnot-bnds (x (bits (olamz a b e) (- e 2) 0)) (n (1- e)))
;			(:instance bits< (x (olamt a b e)) (i e) (j 2))
	;		(:instance bits< (x (olamz a b e)) (i (- e 2)) (j 0))
			(:instance logand-bnd
				   (x (lnot (bits (olamt a b e) e 2) (1- e)))
				   (y (logand (bits (olamz a b e) (1- e) 1)
					      (lnot (bits (olamz a b e) (- e 2) 0) (1- e)))))
))))

(defthm OLAM4-BND
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (integerp k)
		  (<= k (- e 2))
		  (>= k 0))
	     (< (olam4 a b e) (expt 2 (- e 1))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable olam4)
		  :use (;olamg-nat
			;olamt-nat
;			(:instance lnot-bnds (x (bits (olamt a b e) e 2)) (n (1- e)))
	;		(:instance lnot-bnds (x (bits (olamg a b e) (- e 2) 0)) (n (1- e)))
;			(:instance bits< (x (olamt a b e)) (i e) (j 2))
	;		(:instance bits< (x (olamg a b e)) (i (- e 2)) (j 0))
			(:instance logand-bnd
				   (x (lnot (bits (olamt a b e) e 2) (1- e)))
				   (y (logand (bits (olamg a b e) (1- e) 1)
					      (lnot (bits (olamg a b e) (- e 2) 0) (1- e)))))
))))

(local (in-theory (disable EXPO-COMPARISON-REWRITE-TO-BOUND)))

(local (defthm olam0-bnd-1
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (and (integerp (logior (olam3 a b e) (olam4 a b e)))
		  (>= (logior (olam3 a b e) (olam4 a b e))
		      0)
		  (< (logior (olam3 a b e) (olam4 a b e))
		     (expt 2 (- e 1)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable)
		  :use ((:instance olam3-bnd (k 0))
			(:instance olam4-bnd (k 0))
;			(:instance olam3-nat)
	;		(:instance olam4-nat)
;			(:instance or-dist-a (n (- e 1)) (x (olam3 a b e)) (y (olam4 a b e)))
)))))

(local (defthm olam0-bnd-2
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (and (integerp (logior (olam2 a b e)
				    (logior (olam3 a b e)
					    (olam4 a b e))))
		  (>= (logior (olam2 a b e)
			      (logior (olam3 a b e)
				      (olam4 a b e)))
		      0)
		  (< (logior (olam2 a b e)
			     (logior (olam3 a b e)
				     (olam4 a b e)))
		     (expt 2 (- e 1)))))
  :rule-classes ()
  :hints (("Goal" :use (olam0-bnd-1
			(:instance olam2-bnd (k 0))
;			(:instance olam2-nat)
;			(:instance or-dist-a
	;			   (n (- e 1))
		;		   (x (olam2 a b e))
			;	   (y (logior (olam3 a b e) (olam4 a b e))))
                        )))))

(defthm OLAM0-BND
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (< (olam0 a b e) (expt 2 (- e 1))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable olam0)
		  :use (olam0-bnd-2
			;(:instance olam1-nat)
			(:instance olam1-bnd (k 0))
;			(:instance or-dist-a
	;			   (n (- e 1))
		;		   (x (olam1 a b e))
			;	   (y (logior (olam2 a b e)
				;	      (logior (olam3 a b e)
					;	      (olam4 a b e)))))
                        ))))

(defthm OLAMB-BND
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (< (olamb a b e) (expt 2 e)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable olamb expt ;yuck!
                                     )
           :use (olam0-bnd
			;olam0-nat
			(:instance bitn-0-1 (x (olamt a b e)) (n 0))))))

(local (defthm olamb-mod
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (= (mod (olamb a b e) (expt 2 e))
		(olamb a b e)))
  :rule-classes ()
  :hints (("Goal" :use (olamb-bnd
			;olamb-nat
			(:instance mod-does-nothing (m (olamb a b e)) (n (expt 2 e))))))))

(local (defthm lop3-35
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (= (c e a b) 0)
		  (not (= (mod a (expt 2 e))
			  (mod b (expt 2 e)))))
	     (and (not (= (mod (olamb a b e) (expt 2 e)) 0))
		  (= (expo (mod (olamb a b e) (expt 2 e)))
		     (lop a b (c (- e 1) a b) (- e 1)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance lop3-34 (n (- e 1))))))))

(local (defthm lop3-36
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (= (c e a b) 0)
		  (not (= (mod a (expt 2 e))
			  (mod b (expt 2 e)))))
	     (and (not (= (olamb a b e) 0))
		  (= (expo (olamb a b e))
		     (lop a b (c (- e 1) a b) (- e 1)))))
  :rule-classes ()
  :hints (("Goal" :use (lop3-35 olamb-mod)))))

(local (defthm lop3-37
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1)
		  (not (= (mod a (expt 2 e))
			  (mod b (expt 2 e)))))
	     (and (not (= (olamb a b e) 0))
		  (= (expo (olamb a b e))
		     (lop a b (c (- e 1) a b) (- e 1)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable lop c)
           :use (lop3-36
			(:instance expo-upper-bound (x a))
			(:instance expo-upper-bound (x b))
			(:instance expo-lower-bound (x a))
			(:instance expo-lower-bound (x b))
			(:instance bit-expo-b (x a) (n e))
			(:instance bit-expo-b (x b) (n e)))))))

(local (defthm lop3-38
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (and (not (= (olamb a b e) 0))
		  (= (expo (olamb a b e))
		     (lop a b (c (- e 1) a b) (- e 1)))))
  :rule-classes ()
  :hints (("Goal" :use (lop3-37
			(:instance expo-upper-bound (x a))
			(:instance expo-upper-bound (x b))
			(:instance expo-lower-bound (x a))
			(:instance expo-lower-bound (x b))
			(:instance bit-expo-b (x a) (n e))
			(:instance bit-expo-b (x b) (n e))
			(:instance mod-does-nothing (m a) (n (expt 2 (1+ e))))
			(:instance mod-does-nothing (m b) (n (expt 2 (1+ e))))
			(:instance lop3-1 (n e)))))))

(local (defthm lop3-39
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (= (lop a b 0 (1+ e))
		(lop a b (c (- e 1) a b) (- e 1))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable lop c)
           :use ((:instance expo-upper-bound (x a))
			(:instance expo-upper-bound (x b))
			(:instance expo-lower-bound (x a))
			(:instance expo-lower-bound (x b))
			(:instance bit-expo-b (x a) (n e))
			(:instance bit-expo-b (x b) (n e)))))))

(local (defthm lop3-40
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (= (expo (olamb a b e))
		(lop a b 0 (1+ e))))
  :rule-classes ()
  :hints (("Goal" :use (lop3-38 lop3-39)))))

(defthm olop-thm-2
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (and (not (= (olamb a b e) 0))
		  (or (= (expo (- a b)) (expo (olamb a b e)))
		      (= (expo (- a b)) (1- (expo (olamb a b e)))))))
  :rule-classes ()
  :hints (("Goal" :use (lop3-40
			lop3-38
			(:instance lop-bnds (n (1+ e)))
			(:instance expo-upper-bound (x a))
			(:instance expo-upper-bound (x b))))))







(include-book "land0")
(include-book "lior0")
(include-book "lxor0")

(defun lamt-0 (a b e)
  (lxor0 a (lnot b (1+ e)) (1+ e)))

(defun lamg-0 (a b e)
  (land0 a (lnot b (1+ e)) (1+ e)))

(defun lamz-0 (a b e)
  (lnot (lior0 a (lnot b (1+ e)) (1+ e)) (1+ e)))

(defun lam1-0 (a b e)
  (land0 (bits (lamt-0 a b e) e 2)
	(land0 (bits (lamg-0 a b e) (1- e) 1)
	      (lnot (bits (lamz-0 a b e) (- e 2) 0) (1- e))
	      (1- e))
	(1- e)))

(defun lam2-0 (a b e)
  (land0 (lnot (bits (lamt-0 a b e) e 2) (1- e))
	(land0 (bits (lamz-0 a b e) (1- e) 1)
	      (lnot (bits (lamz-0 a b e) (- e 2) 0) (1- e))
	      (1- e))
	(1- e)))

(defun lam3-0 (a b e)
  (land0 (bits (lamt-0 a b e) e 2)
	(land0 (bits (lamz-0 a b e) (1- e) 1)
	      (lnot (bits (lamg-0 a b e) (- e 2) 0) (1- e))
	      (1- e))
	(1- e)))

(defun lam4-0 (a b e)
  (land0 (lnot (bits (lamt-0 a b e) e 2) (1- e))
	(land0 (bits (lamg-0 a b e) (1- e) 1)
	      (lnot (bits (lamg-0 a b e) (- e 2) 0) (1- e))
	      (1- e))
	(1- e)))

(defun lam0-0 (a b e)
  (lior0 (lam1-0 a b e)
	(lior0 (lam2-0 a b e)
	      (lior0 (lam3-0 a b e)
		    (lam4-0 a b e)
		    (1- e))
	      (1- e))
	(1- e)))

(defun lamb-0 (a b e)
  (+ (* 2 (lam0-0 a b e))
     (lnot (bitn (lamt-0 a b e) 0) 1)))

(in-theory (enable bits-tail bvecp-expo))

(local
(defthm bvecp-lamt-0
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (bvecp (lamt-0 a b e) (+ 1 e))))
)

(local
(defthm lamt-0-olamt
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (equal (lamt-0 a b e) (olamt a b e)))
  :hints (("Goal" :in-theory (enable olamt land0 lior0 lxor0))))
)

(local
(defthm bvecp-lamg-0
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (bvecp (lamg-0 a b e) (+ 1 e))))
)

(local
(defthm lamg-0-olamg
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (equal (lamg-0 a b e) (olamg a b e)))
  :hints (("Goal" :in-theory (enable olamg land0 lior0 lxor0))))
)

(local
(defthm bvecp-lamz-0
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (bvecp (lamz-0 a b e) (+ 1 e))))
)

(local
(defthm lamz-0-olamz
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (equal (lamz-0 a b e) (olamz a b e)))
  :hints (("Goal" :in-theory (enable olamz land0 lior0 lxor0))))
)

(local
(defthm bvecp-lam1-0
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (bvecp (lam1-0 a b e) (+ -1 e))))
)

(local
(defthm lam1-0-olam1
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (equal (lam1-0 a b e) (olam1 a b e)))
  :hints (("Goal" :in-theory (enable olam1 land0 lior0 lxor0)
		  :use (bvecp-lamt-0 bvecp-lamz-0 bvecp-lamg-0))))
)

(local
(defthm bvecp-lam2-0
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (bvecp (lam2-0 a b e) (+ -1 e))))
)

(local
(defthm lam2-0-olam2
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (equal (lam2-0 a b e) (olam2 a b e)))
  :hints (("Goal" :in-theory (enable olam2 land0 lior0 lxor0))))
)

(local
(defthm bvecp-lam3-0
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (bvecp (lam3-0 a b e) (+ -1 e))))
)

(local
(defthm lam3-0-olam3
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (equal (lam3-0 a b e) (olam3 a b e)))
  :hints (("Goal" :in-theory (enable olam3 land0 lior0 lxor0))))
)

(local
(defthm bvecp-lam4-0
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (bvecp (lam4-0 a b e) (+ -1 e))))
)

(local
(defthm lam4-0-olam4
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (equal (lam4-0 a b e) (olam4 a b e)))
  :hints (("Goal" :in-theory (enable olam4 land0 lior0 lxor0))))
)

(local
(defthm bvecp-lam0-0
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (bvecp (lam0-0 a b e) (+ -1 e))))
)

(local
(defthm lam0-0-olam0
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (equal (lam0-0 a b e) (olam0 a b e)))
  :hints (("Goal" :in-theory (enable olam0 land0 lior0 lxor0)
		  :use (bvecp-lam1-0 bvecp-lam2-0 bvecp-lam3-0 bvecp-lam4-0))))
)

(local
(defthm lamb-0-olamb
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (equal (lamb-0 a b e) (olamb a b e)))
  :hints (("Goal" :in-theory (enable olamb))))
)

(defthm lop-thm-2-original
    (implies (and (integerp a)
		  (> a 0)
		  (integerp b)
		  (> b 0)
		  (not (= a b))
		  (= e (expo a))
		  (= e (expo b))
		  (> e 1))
	     (and (not (= (lamb-0 a b e) 0))
		  (or (= (expo (- a b)) (expo (lamb-0 a b e)))
		      (= (expo (- a b)) (1- (expo (lamb-0 a b e)))))))
  :rule-classes ()
  :hints (("Goal" :use olop-thm-2)))


