# Json Serializer for PHP

<p align="center">
    <a href="LICENSE.txt" target="_blank">
        <img alt="Software License" src="https://img.shields.io/badge/license-MIT-brightgreen.svg?style=flat-square">
    </a>
    <img alt="Build Status" src="https://github.com/zumba/json-serializer/actions/workflows/php.yml/badge.svg?branch=master">
    <a href="https://packagist.org/packages/zumba/json-serializer" target="_blank">
        <img alt="Total Downloads" src="https://img.shields.io/packagist/dt/zumba/json-serializer.svg?style=flat-square">
    </a>
    <a href="https://packagist.org/packages/zumba/json-serializer" target="_blank">
        <img alt="Latest Stable Version" src="https://img.shields.io/packagist/v/zumba/json-serializer.svg?style=flat-square&label=stable">
    </a>
</p>

This is a library to serialize PHP variables in JSON format. It is similar of the `serialize()` function in PHP,
but the output is a string JSON encoded. You can also unserialize the JSON generated by this tool and have you
PHP content back.

Supported features:

- Encode/Decode of scalar, null, array
- Encode/Decode of objects
- Encode/Decode of binary data
- Support nested serialization
- Support not declared properties on the original class definition (ie, properties in `stdClass`)
- Support object recursion
- Closures (via 3rd party library. See details below)

Unsupported serialization content:

- Resource (ie, `fopen()` response)
- NAN, INF constants

Limitations:

- Binary data containing null bytes (\u0000) as array keys cannot be properly decoded because of a json extension bug:
    - https://github.com/remicollet/pecl-json-c/issues/7
    - https://github.com/json-c/json-c/issues/108

This project should not be confused with `JsonSerializable` interface added on PHP 5.4. This interface is used on
`json_encode` to encode the objects. There is no unserialization with this interface, differently from this project.

*Json Serializer requires PHP >= 7.0 and tested until PHP 7.4*

## Example

```php

class MyCustomClass {
	public $isItAwesome = true;
	protected $nice = 'very!';
}

$instance = new MyCustomClass();

$serializer = new Zumba\JsonSerializer\JsonSerializer();
$json = $serializer->serialize($instance);
// $json will contain the content {"@type":"MyCustomClass","isItAwesome":true,"nice":"very!"}

$restoredInstance = $serializer->unserialize($json);
// $restoredInstance will be an instance of MyCustomClass
```

## How to Install

If you are using composer, install the package [`zumba/json-serializer`](https://packagist.org/packages/zumba/json-serializer).

```
$ composer require zumba/json-serializer
```

Or add the `zumba/json-serializer` directly in your `composer.json` file.

If you are not using composer, you can just copy the files from `src` folder in your project.

## Serializing Binary Strings

Binary strings introduce two special identifiers in the final json: `@utf8encoded` and `@scalar`.
`@utf8encoded` is an array of keys from the original data which have their value (or the keys themselves)
encoded from 8bit to UTF-8. This is how the serializer knows what to encode back from UTF-8 to 8bit when deserializing.
Example:

```php
$data = ['key' => '<binaryvalue>', 'anotherkey' => 'nonbinaryvalue'];
$serializer = new Zumba\JsonSerializer\JsonSerializer();
$json = $serializer->serialize($data);
// $json will contain the content {"key":"<utf8encodedbinaryvalue>","anotherkey":"nonbinaryvalue","@utf8encoded":{"key":1}}
```

`@scalar` is used only when the value to be encoded is not an array or an object but a binary string. Example:

```php
$data = '<binaryvalue>';
$serializer = new Zumba\JsonSerializer\JsonSerializer();
$json = $serializer->serialize($data);
// $json will contain the content {"@scalar":"<utf8encodedbinaryvalue>","@utf8encoded":1}
```


## Serializing Closures

For serializing PHP closures you have to pass an object implementing `SuperClosure\SerializerInterface`.
This interface is provided by the project [SuperClosure](https://github.com/jeremeamia/super_closure). This
project also provides a closure serializer that implements this interface.

Closure serialization has some limitations. Please check the SuperClosure project to check if it fits your
needs.

To use the SuperClosure with JsonSerializer, just pass the SuperClosure object as the first parameter
on JsonSerializer constructor. Example:

```php
$toBeSerialized = [
	'data' => [1, 2, 3],
	'worker' => function ($data) {
		$double = [];
		foreach ($data as $i => $number) {
			$double[$i] = $number * 2;
		}
		return $double;
	}
];

$superClosure = new SuperClosure\Serializer();
$jsonSerializer = new Zumba\JsonSerializer\JsonSerializer($superClosure);
$serialized = $jsonSerializer->serialize($toBeSerialized);
```

PS: JsonSerializer does not have a hard dependency of SuperClosure. If you want to use both projects
make sure you add both on your composer requirements.

## Custom Serializers

Some classes may not be suited to be serialized and unserialized using the default reflection methods.

Custom serializers provide the ability to define ```serialize``` and ```unserialize``` methods for specific classes.

```php
class MyType {
    public $field1;
    public $field2;
}

class MyTypeSerializer {
    public function serialize(MyType $obj) {
        return array('fields' => $obj->field1 . ' ' . $obj->field2);
    }

    public function unserialize($values) {
        list($field1, $field2) = explode(' ', $values['fields']);
        $obj = new MyType();
        $obj->field1 = $field1;
        $obj->field2 = $field2;
        return $obj;
    }
}

// map of "class name" => Custom serializer
$customObjectSerializers['MyType'] = new MyTypeSerializer();
$jsonSerializer = new Zumba\JsonSerializer\JsonSerializer(null, $customObjectSerializers);

$toBeSerialized = new MyType();
$toBeSerialized->field1 = 'x';
$toBeSerialized->field2 = 'y';
$json = $jsonSerializer->serialize($toBeSerialized);
// $json == {"@type":"Zumba\\\\JsonSerializer\\\\Test\\\\SupportClasses\\\\MyType","fields":"x y"}

$myType = $jsonSerializer->unserialize($json);
// $myType == $toBeSerialized
```
