% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/zoned-time.R
\name{zoned-parsing}
\alias{zoned-parsing}
\alias{zoned_time_parse_complete}
\alias{zoned_time_parse_abbrev}
\title{Parsing: zoned-time}
\usage{
zoned_time_parse_complete(
  x,
  ...,
  format = NULL,
  precision = "second",
  locale = clock_locale()
)

zoned_time_parse_abbrev(
  x,
  zone,
  ...,
  format = NULL,
  precision = "second",
  locale = clock_locale()
)
}
\arguments{
\item{x}{\verb{[character]}

A character vector to parse.}

\item{...}{These dots are for future extensions and must be empty.}

\item{format}{\verb{[character / NULL]}

A format string. A combination of the following commands, or \code{NULL},
in which case a default format string is used.

A vector of multiple format strings can be supplied. They will be tried in
the order they are provided.

\strong{Year}
\itemize{
\item \verb{\%C}: The century as a decimal number. The modified command \verb{\%NC} where
\code{N} is a positive decimal integer specifies the maximum number of
characters to read. If not specified, the default is \code{2}. Leading zeroes
are permitted but not required.
\item \verb{\%y}: The last two decimal digits of the year. If the century is not
otherwise specified (e.g. with \verb{\%C}), values in the range \verb{[69 - 99]} are
presumed to refer to the years \verb{[1969 - 1999]}, and values in the range
\verb{[00 - 68]} are presumed to refer to the years \verb{[2000 - 2068]}. The
modified command \verb{\%Ny}, where \code{N} is a positive decimal integer, specifies
the maximum number of characters to read. If not specified, the default is
\code{2}. Leading zeroes are permitted but not required.
\item \verb{\%Y}: The year as a decimal number. The modified command \verb{\%NY} where \code{N}
is a positive decimal integer specifies the maximum number of characters to
read. If not specified, the default is \code{4}. Leading zeroes are permitted
but not required.
}

\strong{Month}
\itemize{
\item \verb{\%b}, \verb{\%B}, \verb{\%h}: The \code{locale}'s full or abbreviated case-insensitive
month name.
\item \verb{\%m}: The month as a decimal number. January is \code{1}. The modified command
\verb{\%Nm} where \code{N} is a positive decimal integer specifies the maximum number
of characters to read. If not specified, the default is \code{2}. Leading zeroes
are permitted but not required.
}

\strong{Day}
\itemize{
\item \verb{\%d}, \verb{\%e}: The day of the month as a decimal number. The modified
command \verb{\%Nd} where \code{N} is a positive decimal integer specifies the maximum
number of characters to read. If not specified, the default is \code{2}. Leading
zeroes are permitted but not required.
}

\strong{Day of the week}
\itemize{
\item \verb{\%a}, \verb{\%A}: The \code{locale}'s full or abbreviated case-insensitive weekday
name.
\item \verb{\%w}: The weekday as a decimal number (\code{0-6}), where Sunday is \code{0}. The
modified command \verb{\%Nw} where \code{N} is a positive decimal integer specifies
the maximum number of characters to read. If not specified, the default is
\code{1}. Leading zeroes are permitted but not required.
}

\strong{ISO 8601 week-based year}
\itemize{
\item \verb{\%g}: The last two decimal digits of the ISO week-based year. The
modified command \verb{\%Ng} where \code{N} is a positive decimal integer specifies
the maximum number of characters to read. If not specified, the default is
\code{2}. Leading zeroes are permitted but not required.
\item \verb{\%G}: The ISO week-based year as a decimal number. The modified command
\verb{\%NG} where \code{N} is a positive decimal integer specifies the maximum number
of characters to read. If not specified, the default is \code{4}. Leading zeroes
are permitted but not required.
\item \verb{\%V}: The ISO week-based week number as a decimal number. The modified
command \verb{\%NV} where \code{N} is a positive decimal integer specifies the maximum
number of characters to read. If not specified, the default is \code{2}. Leading
zeroes are permitted but not required.
\item \verb{\%u}: The ISO weekday as a decimal number (\code{1-7}), where Monday is \code{1}.
The modified command \verb{\%Nu} where \code{N} is a positive decimal integer
specifies the maximum number of characters to read. If not specified, the
default is \code{1}. Leading zeroes are permitted but not required.
}

\strong{Week of the year}
\itemize{
\item \verb{\%U}: The week number of the year as a decimal number. The first Sunday
of the year is the first day of week \code{01}. Days of the same year prior to
that are in week \code{00}. The modified command \verb{\%NU} where \code{N} is a positive
decimal integer specifies the maximum number of characters to read. If not
specified, the default is \code{2}. Leading zeroes are permitted but not
required.
\item \verb{\%W}: The week number of the year as a decimal number. The first Monday
of the year is the first day of week \code{01}. Days of the same year prior to
that are in week \code{00}. The modified command \verb{\%NW} where \code{N} is a positive
decimal integer specifies the maximum number of characters to read. If not
specified, the default is \code{2}. Leading zeroes are permitted but not
required.
}

\strong{Day of the year}
\itemize{
\item \verb{\%j}: The day of the year as a decimal number. January 1 is \code{1}. The
modified command \verb{\%Nj} where \code{N} is a positive decimal integer specifies
the maximum number of characters to read. If not specified, the default is
\code{3}. Leading zeroes are permitted but not required.
}

\strong{Date}
\itemize{
\item \verb{\%D}, \verb{\%x}: Equivalent to \verb{\%m/\%d/\%y}.
\item \verb{\%F}: Equivalent to \verb{\%Y-\%m-\%d}. If modified with a width (like \verb{\%NF}),
the width is applied to only \verb{\%Y}.
}

\strong{Time of day}
\itemize{
\item \verb{\%H}: The hour (24-hour clock) as a decimal number. The modified command
\verb{\%NH} where \code{N} is a positive decimal integer specifies the maximum number
of characters to read. If not specified, the default is \code{2}. Leading zeroes
are permitted but not required.
\item \verb{\%I}: The hour (12-hour clock) as a decimal number. The modified command
\verb{\%NI} where \code{N} is a positive decimal integer specifies the maximum number
of characters to read. If not specified, the default is \code{2}. Leading zeroes
are permitted but not required.
\item \verb{\%M}: The minutes as a decimal number. The modified command \verb{\%NM} where
\code{N} is a positive decimal integer specifies the maximum number of
characters to read. If not specified, the default is \code{2}. Leading zeroes
are permitted but not required.
\item \verb{\%S}: The seconds as a decimal number. Leading zeroes are permitted but
not required. If encountered, the \code{locale} determines the decimal point
character. Generally, the maximum number of characters to read is
determined by the precision that you are parsing at. For example, a
precision of \code{"second"} would read a maximum of 2 characters, while a
precision of \code{"millisecond"} would read a maximum of 6 (2 for the values
before the decimal point, 1 for the decimal point, and 3 for the values
after it). The modified command \verb{\%NS}, where \code{N} is a positive decimal
integer, can be used to exactly specify the maximum number of characters to
read. This is only useful if you happen to have seconds with more than 1
leading zero.
\item \verb{\%p}: The \code{locale}'s equivalent of the AM/PM designations associated with
a 12-hour clock. The command \verb{\%I} must precede \verb{\%p} in the format string.
\item \verb{\%R}: Equivalent to \verb{\%H:\%M}.
\item \verb{\%T}, \verb{\%X}: Equivalent to \verb{\%H:\%M:\%S}.
\item \verb{\%r}: Equivalent to \verb{\%I:\%M:\%S \%p}.
}

\strong{Time zone}
\itemize{
\item \verb{\%z}: The offset from UTC in the format \verb{[+|-]hh[mm]}. For example
\code{-0430} refers to 4 hours 30 minutes behind UTC. And \code{04} refers to 4 hours
ahead of UTC. The modified command \verb{\%Ez} parses a \code{:} between the hours and
minutes and leading zeroes on the hour field are optional:
\verb{[+|-]h[h][:mm]}. For example \code{-04:30} refers to 4 hours 30 minutes behind
UTC. And \code{4} refers to 4 hours ahead of UTC.
\item \verb{\%Z}: The full time zone name or the time zone abbreviation, depending on
the function being used. A single word is parsed. This word can only
contain characters that are alphanumeric, or one of \code{'_'}, \code{'/'}, \code{'-'} or
\code{'+'}.
}

\strong{Miscellaneous}
\itemize{
\item \verb{\%c}: A date and time representation. Equivalent to
\verb{\%a \%b \%d \%H:\%M:\%S \%Y}.
\item \code{\%\%}: A \verb{\%} character.
\item \verb{\%n}: Matches one white space character. \verb{\%n}, \verb{\%t}, and a space can be
combined to match a wide range of white-space patterns. For example \code{"\%n "}
matches one or more white space characters, and \code{"\%n\%t\%t"} matches one to
three white space characters.
\item \verb{\%t}: Matches zero or one white space characters.
}}

\item{precision}{\verb{[character(1)]}

A precision for the resulting zoned-time. One of:
\itemize{
\item \code{"second"}
\item \code{"millisecond"}
\item \code{"microsecond"}
\item \code{"nanosecond"}
}

Setting the \code{precision} determines how much information \verb{\%S} attempts
to parse.}

\item{locale}{\verb{[clock_locale]}

A locale object created from \code{\link[=clock_locale]{clock_locale()}}.}

\item{zone}{\verb{[character(1)]}

A full time zone name.}
}
\value{
A zoned-time.
}
\description{
There are two parsers into a zoned-time, \code{zoned_time_parse_complete()} and
\code{zoned_time_parse_abbrev()}.
\subsection{zoned_time_parse_complete()}{

\code{zoned_time_parse_complete()} is a parser for \emph{complete} date-time strings,
like \code{"2019-01-01T00:00:00-05:00[America/New_York]"}. A complete date-time
string has both the time zone offset and full time zone name in the string,
which is the only way for the string itself to contain all of the information
required to construct a zoned-time. Because of this,
\code{zoned_time_parse_complete()} requires both the \verb{\%z} and \verb{\%Z} commands to be
supplied in the \code{format} string.

The default options assume that \code{x} should be parsed at second precision,
using a \code{format} string of \code{"\%Y-\%m-\%dT\%H:\%M:\%S\%Ez[\%Z]"}. This matches the
default result from calling \code{format()} on a zoned-time. Additionally, this
format matches the de-facto standard extension to RFC 3339 for creating
completely unambiguous date-times.
}

\subsection{zoned_time_parse_abbrev()}{

\code{zoned_time_parse_abbrev()} is a parser for date-time strings containing only
a time zone abbreviation, like \code{"2019-01-01 00:00:00 EST"}. The time zone
abbreviation is not enough to identify the full time zone name that the
date-time belongs to, so the full time zone name must be supplied as the
\code{zone} argument. However, the time zone abbreviation can help with resolving
ambiguity around daylight saving time fallbacks.

For \code{zoned_time_parse_abbrev()}, \verb{\%Z} must be supplied and is interpreted as
the time zone abbreviation rather than the full time zone name.

If used, the \verb{\%z} command must parse correctly, but its value will be
completely ignored.

The default options assume that \code{x} should be parsed at second precision,
using a \code{format} string of \code{"\%Y-\%m-\%d \%H:\%M:\%S \%Z"}. This matches the default
result from calling \code{print()} or \code{format(usetz = TRUE)} on a POSIXct
date-time.
}
}
\details{
If \code{zoned_time_parse_complete()} is given input that is length zero, all
\code{NA}s, or completely fails to parse, then no time zone will be able to be
determined. In that case, the result will use \code{"UTC"}.

If your date-time strings contain time zone offsets (like \code{-04:00}), but
not the full time zone name, you might need \code{\link[=sys_time_parse]{sys_time_parse()}}.

If your date-time strings don't contain time zone offsets or the full time
zone name, you might need to use \code{\link[=naive_time_parse]{naive_time_parse()}}. From there, if you
know the time zone that the date-times are supposed to be in, you can convert
to a zoned-time with \code{\link[=as_zoned_time]{as_zoned_time()}}.
}
\section{Full Precision Parsing}{


It is highly recommended to parse all of the information in the date-time
string into a type at least as precise as the string. For example, if your
string has fractional seconds, but you only require seconds, specify a
sub-second \code{precision}, then round to seconds manually using whatever
convention is appropriate for your use case. Parsing such a string directly
into a second precision result is ambiguous and undefined, and is unlikely to
work as you might expect.
}

\examples{
library(magrittr)

zoned_time_parse_complete("2019-01-01T01:02:03-05:00[America/New_York]")

zoned_time_parse_complete(
  "January 21, 2019 -0500 America/New_York",
  format = "\%B \%d, \%Y \%z \%Z"
)

# Nanosecond precision
x <- "2019/12/31 01:05:05.123456700-05:00[America/New_York]"
zoned_time_parse_complete(
  x,
  format = "\%Y/\%m/\%d \%H:\%M:\%S\%Ez[\%Z]",
  precision = "nanosecond"
)

# The `\%z` offset must correspond to the true offset that would be used
# if the input was parsed as a naive-time and then converted to a zoned-time
# with `as_zoned_time()`. For example, the time that was parsed above used an
# offset of `-05:00`. We can confirm that this is correct with:
year_month_day(2019, 1, 1, 1, 2, 3) \%>\%
  as_naive_time() \%>\%
  as_zoned_time("America/New_York")

# So the following would not parse correctly
zoned_time_parse_complete("2019-01-01T01:02:03-04:00[America/New_York]")

# `\%z` is useful for breaking ties in otherwise ambiguous times. For example,
# these two times are on either side of a daylight saving time fallback.
# Without the `\%z` offset, you wouldn't be able to tell them apart!
x <- c(
  "1970-10-25T01:30:00-04:00[America/New_York]",
  "1970-10-25T01:30:00-05:00[America/New_York]"
)

zoned_time_parse_complete(x)

# If you have date-time strings with time zone abbreviations,
# `zoned_time_parse_abbrev()` should be able to help. The `zone` must be
# provided, because multiple countries may use the same time zone
# abbreviation. For example:
x <- "1970-01-01 02:30:30 IST"

# IST = India Standard Time
zoned_time_parse_abbrev(x, "Asia/Kolkata")

# IST = Israel Standard Time
zoned_time_parse_abbrev(x, "Asia/Jerusalem")

# The time zone abbreviation is mainly useful for resolving ambiguity
# around daylight saving time fallbacks. Without the abbreviation, these
# date-times would be ambiguous.
x <- c(
  "1970-10-25 01:30:00 EDT",
  "1970-10-25 01:30:00 EST"
)
zoned_time_parse_abbrev(x, "America/New_York")
}
