% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RNG.R
\name{getRNG}
\alias{getRNG}
\alias{hasRNG}
\alias{nextRNG}
\alias{setRNG}
\title{Getting/Setting RNGs}
\usage{
getRNG(object, ..., num.ok = FALSE, extract = TRUE, recursive = TRUE)

hasRNG(object)

nextRNG(object, ..., ndraw = 0L)

setRNG(object, ..., verbose = FALSE, check = TRUE)
}
\arguments{
\item{object}{an R object from which RNG settings can be extracted, e.g. an
integer vector containing a suitable value for \code{.Random.seed} or embedded
RNG data, e.g., in S3/S4 slot \code{rng} or \code{rng$noise}.}

\item{...}{extra arguments to allow extension and passed to a suitable S4 method
\code{.getRNG} or \code{.setRNG}.}

\item{num.ok}{logical that indicates if single numeric (not integer) RNG data should be
considered as a valid RNG seed (\code{TRUE}) or passed to \code{\link{set.seed}}
into a proper RNG seed (\code{FALSE}) (See details and examples).}

\item{extract}{logical that indicates if embedded RNG data should be looked for and
extracted (\code{TRUE}) or if the object itself should be considered as an
RNG specification.}

\item{recursive}{logical that indicates if embedded RNG data should be extracted
recursively (\code{TRUE}) or only once (\code{FASE}).}

\item{ndraw}{number of draws to perform before returning the RNG seed.}

\item{verbose}{a logical that indicates if the new RNG settings should
be displayed.}

\item{check}{logical that indicates if only valid RNG kinds should be
accepted, or if invalid values should just throw a warning.
Note that this argument is used only on R >= 3.0.2.}
}
\value{
\code{getRNG}, \code{getRNG1}, \code{nextRNG} and \code{setRNG}
usually return an integer vector of length > 2L, like \code{\link{.Random.seed}}.

\code{getRNG} and \code{getRNG1} return \code{NULL} if no RNG data was found.

\code{setRNG} invisibly returns the old RNG settings as
they were before changing them.
}
\description{
\code{getRNG} returns the Random Number Generator (RNG) settings used for
computing an object, using a suitable \code{.getRNG} S4 method to extract
these settings.
For example, in the case of objects that result from multiple model fits,
it would return the RNG settings used to compute the best fit.
}
\details{
This function handles single number RNG specifications in the following way:
\describe{
\item{integers}{Return them unchanged, considering them as encoded RNG kind
specification (see \code{\link{RNG}}). No validity check is performed.}
\item{real numbers}{If \code{num.ok=TRUE} return them unchanged.
Otherwise, consider them as (pre-)seeds and pass them to \code{\link{set.seed}}
to get a proper RNG seed.
Hence calling \code{getRNG(1234)} is equivalent to \code{set.seed(1234); getRNG()}
(See examples).
}
}
}
\examples{
#--- getRNG ---
# get current RNG settings
s <- getRNG()
head(s)
showRNG(s)

# get RNG from a given single numeric seed
s1234 <- getRNG(1234)
head(s1234)
showRNG(s1234)
# this is identical to the RNG seed as after set.seed()
set.seed(1234)
identical(s1234, .Random.seed)
# but if num.ok=TRUE the object is returned unchanged
getRNG(1234, num.ok=TRUE)

# single integer RNG data = encoded kind 
head(getRNG(1L))

# embedded RNG data
s <- getRNG(list(1L, rng=1234))
identical(s, s1234)
 

#--- hasRNG ---
# test for embedded RNG data
hasRNG(1)
hasRNG( structure(1, rng=1:3) )
hasRNG( list(1, 2, 3) )
hasRNG( list(1, 2, 3, rng=1:3) )
hasRNG( list(1, 2, 3, noise=list(1:3, rng=1)) )


#--- nextRNG ---
head(nextRNG())
head(nextRNG(1234))
head(nextRNG(1234, ndraw=10))


#--- setRNG ---

obj <- list(x=1000, rng=123)
setRNG(obj)
rng <- getRNG()
runif(10)
set.seed(123)
rng.equal(rng)


}
\seealso{
\code{\link{.Random.seed}}, \code{\link{showRNG}}
}
