"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.openIssueCreationLink = exports.maybePrintIssueCreationPrompt = exports.shouldRecommendIssue = void 0;
const chalk_1 = __importDefault(require("chalk"));
const open_1 = __importDefault(require("open"));
const querystring_1 = require("querystring");
const path_1 = require("./path");
const repoSpecifier = /^([\w.-]+)\/([\w.-]+)$/;
const githubURL = /github.com(:|\/)([\w.-]+\/[\w.-]+?)(.git|\/.*)?$/;
function parseRepoString(repository) {
    if (repository.startsWith("github:")) {
        repository = repository.replace(/^github:/, "");
    }
    const urlMatch = repository.match(githubURL);
    if (urlMatch) {
        repository = urlMatch[2];
    }
    const specMatch = repository.match(repoSpecifier);
    if (!specMatch) {
        return null;
    }
    const [, org, repo] = specMatch;
    return { org, repo, provider: "GitHub" };
}
function getPackageVCSDetails(packageDetails) {
    const repository = require(path_1.resolve(path_1.join(packageDetails.path, "package.json")))
        .repository;
    if (!repository) {
        return null;
    }
    if (typeof repository === "string") {
        return parseRepoString(repository);
    }
    else if (typeof repository === "object" &&
        typeof repository.url === "string") {
        return parseRepoString(repository.url);
    }
}
function shouldRecommendIssue(vcsDetails) {
    if (!vcsDetails) {
        return true;
    }
    const { repo, org } = vcsDetails;
    if (repo === "DefinitelyTyped" && org === "DefinitelyTyped") {
        return false;
    }
    return true;
}
exports.shouldRecommendIssue = shouldRecommendIssue;
function maybePrintIssueCreationPrompt(packageDetails, packageManager) {
    const vcs = getPackageVCSDetails(packageDetails);
    if (vcs && shouldRecommendIssue(vcs)) {
        console.log(`💡 ${chalk_1.default.bold(packageDetails.name)} is on ${vcs.provider}! To draft an issue based on your patch run

    ${packageManager === "yarn" ? "yarn" : "npx"} patch-package ${packageDetails.pathSpecifier} --create-issue
`);
    }
}
exports.maybePrintIssueCreationPrompt = maybePrintIssueCreationPrompt;
function openIssueCreationLink({ packageDetails, patchFileContents, packageVersion, }) {
    const vcs = getPackageVCSDetails(packageDetails);
    if (!vcs) {
        console.error(`Error: Couldn't find VCS details for ${packageDetails.pathSpecifier}`);
        process.exit(1);
    }
    // trim off trailing newline since we add an extra one in the markdown block
    if (patchFileContents.endsWith("\n")) {
        patchFileContents = patchFileContents.slice(0, -1);
    }
    open_1.default(`https://github.com/${vcs.org}/${vcs.repo}/issues/new?${querystring_1.stringify({
        title: "",
        body: `Hi! 👋 
      
Firstly, thanks for your work on this project! 🙂

Today I used [patch-package](https://github.com/ds300/patch-package) to patch \`${packageDetails.name}@${packageVersion}\` for the project I'm working on.

<!-- 🔺️🔺️🔺️ PLEASE REPLACE THIS BLOCK with a description of your problem, and any other relevant context 🔺️🔺️🔺️ -->

Here is the diff that solved my problem:

\`\`\`diff
${patchFileContents}
\`\`\`

<em>This issue body was [partially generated by patch-package](https://github.com/ds300/patch-package/issues/296).</em>
`,
    })}`);
}
exports.openIssueCreationLink = openIssueCreationLink;
//# sourceMappingURL=data:application/json;base64,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