// Copyright 2025 Specter Ops, Inc.
//
// Licensed under the Apache License, Version 2.0
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//	http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0
package generator

import (
	"fmt"
	"io"
	"os"
	"path"
	"strings"

	"github.com/specterops/bloodhound/packages/go/schemagen/model"
)

type nodeIcon struct {
	Icon  string
	Color string
}

var nodeIcons = map[string]nodeIcon{
	// Active Directory Node Types
	"User": {
		Icon:  "fa-user",
		Color: "#17E625",
	},
	"Group": {
		Icon:  "fa-users",
		Color: "#DBE617",
	},
	"Computer": {
		Icon:  "fa-desktop",
		Color: "#E67873",
	},
	"Domain": {
		Icon:  "fa-globe",
		Color: "#17E6B9",
	},
	"GPO": {
		Icon:  "fa-list",
		Color: "#998EFD",
	},
	"AIACA": {
		Icon:  "fa-arrows-left-right-to-line",
		Color: "#9769F0",
	},
	"RootCA": {
		Icon:  "fa-landmark",
		Color: "#6968E8",
	},
	"EnterpriseCA": {
		Icon:  "fa-building",
		Color: "#4696E9",
	},
	"NTAuthStore": {
		Icon:  "fa-store",
		Color: "#D575F5",
	},
	"CertTemplate": {
		Icon:  "fa-id-card",
		Color: "#B153F3",
	},
	"IssuancePolicy": {
		Icon:  "fa-clipboard-check",
		Color: "#99B2DD",
	},
	"OU": {
		Icon:  "fa-sitemap",
		Color: "#FFAA00",
	},
	"Container": {
		Icon:  "fa-box",
		Color: "#F79A78",
	},
	// Azure Node Types
	"AZUser": {
		Icon:  "fa-user",
		Color: "#34D2EB",
	},
	"AZGroup": {
		Icon:  "fa-users",
		Color: "#F57C9B",
	},
	"AZTenant": {
		Icon:  "fa-cloud",
		Color: "#54F2F2",
	},
	"AZSubscription": {
		Icon:  "fa-key",
		Color: "#D2CCA1",
	},
	"AZResourceGroup": {
		Icon:  "fa-cube",
		Color: "#89BD9E",
	},
	"AZVM": {
		Icon:  "fa-desktop",
		Color: "#F9ADA0",
	},
	"AZWebApp": {
		Icon:  "fa-object-group",
		Color: "#4696E9",
	},
	"AZLogicApp": {
		Icon:  "fa-sitemap",
		Color: "#9EE047",
	},
	"AZAutomationAccount": {
		Icon:  "fa-cog",
		Color: "#F4BA44",
	},
	"AZFunctionApp": {
		Icon:  "fa-bolt",
		Color: "#F4BA44",
	},
	"AZContainerRegistry": {
		Icon:  "fa-box-open",
		Color: "#0885D7",
	},
	"AZManagedCluster": {
		Icon:  "fa-cubes",
		Color: "#326CE5",
	},
	"AZDevice": {
		Icon:  "fa-desktop",
		Color: "#B18FCF",
	},
	"AZKeyVault": {
		Icon:  "fa-lock",
		Color: "#ED658C",
	},
	"AZApp": {
		Icon:  "fa-window-restore",
		Color: "#03FC84",
	},
	"AZVMScaleSet": {
		Icon:  "fa-server",
		Color: "#007CD0",
	},
	"AZServicePrincipal": {
		Icon:  "fa-robot",
		Color: "#C1D6D6",
	},
	"AZRole": {
		Icon:  "fa-clipboard-list",
		Color: "#ED8537",
	},
	"AZManagementGroup": {
		Icon:  "fa-sitemap",
		Color: "#BD93D8",
	},
}

func GenerateExtensionSQLActiveDirectory(dir string, adSchema model.ActiveDirectory) error {
	return GenerateExtensionSQL("AD", "Active Directory", "v0.0.1", "AD", dir, "ad_graph_schema.sql", adSchema.NodeKinds, adSchema.RelationshipKinds, adSchema.PathfindingRelationships)
}

func GenerateExtensionSQLAzure(dir string, azSchema model.Azure) error {
	return GenerateExtensionSQL("AZ", "Azure", "v0.0.1", "AZ", dir, "az_graph_schema.sql", azSchema.NodeKinds, azSchema.RelationshipKinds, azSchema.PathfindingRelationships)
}

func GenerateExtensionSQL(name string, displayName string, version string, namespace, dir string, fileName string, nodeKinds []model.StringEnum, relationshipKinds []model.StringEnum, pathfindingRelationshipKinds []model.StringEnum) error {
	var sb strings.Builder

	sb.WriteString(fmt.Sprintf("-- Code generated by Cuelang code gen. DO NOT EDIT!\n-- Cuelang source: %s/", SchemaSourceName))

	sb.WriteString(`
CREATE OR REPLACE FUNCTION genscript_upsert_kind(node_kind_name TEXT) RETURNS void AS $$
BEGIN
    IF NOT EXISTS (SELECT id FROM kind WHERE kind.name = node_kind_name) THEN
        INSERT INTO kind (name) VALUES (node_kind_name);
    END IF;
END $$ LANGUAGE plpgsql;
	`)

	sb.WriteString(`
CREATE OR REPLACE FUNCTION genscript_upsert_source_kind(node_kind_name TEXT) RETURNS void AS $$
BEGIN
    IF NOT EXISTS (SELECT id FROM source_kinds WHERE source_kinds.name = node_kind_name) THEN
        INSERT INTO source_kinds (name) VALUES (node_kind_name);
    END IF;
END $$ LANGUAGE plpgsql;
	`)

	sb.WriteString(`
CREATE OR REPLACE FUNCTION genscript_upsert_schema_node_kind(v_extension_id INT, v_kind_name VARCHAR(256), v_display_name TEXT, v_description TEXT, v_is_display_kind BOOLEAN, v_icon TEXT, v_icon_color TEXT) RETURNS void AS $$
DECLARE
	retreived_kind_id SMALLINT;
BEGIN
	SELECT id INTO retreived_kind_id FROM kind WHERE name = v_kind_name;
	IF retreived_kind_id IS NULL THEN
		RAISE EXCEPTION 'couldn''t find matching kind_id';
	END IF;
	
	IF NOT EXISTS (SELECT id FROM schema_node_kinds nk WHERE nk.kind_id = retreived_kind_id) THEN
		INSERT INTO schema_node_kinds (schema_extension_id, kind_id, display_name, description, is_display_kind, icon, icon_color) VALUES (v_extension_id, retreived_kind_id, v_display_name, v_description, v_is_display_kind, v_icon, v_icon_color);
	ELSE
		UPDATE schema_node_kinds SET display_name = v_display_name, description = v_description, is_display_kind = v_is_display_kind, icon = v_icon, icon_color = v_icon_color WHERE kind_id = retreived_kind_id;
	END IF;
END;
$$ LANGUAGE plpgsql;
`)

	sb.WriteString(`
CREATE OR REPLACE FUNCTION genscript_upsert_schema_relationship_kind(v_extension_id INT, v_kind_name VARCHAR(256), v_description TEXT, v_is_traversable BOOLEAN) RETURNS void AS $$
DECLARE
	retreived_kind_id SMALLINT;
BEGIN
	SELECT id INTO retreived_kind_id FROM kind WHERE name = v_kind_name;
	IF retreived_kind_id IS NULL THEN
		RAISE EXCEPTION 'couldn''t find matching kind_id';
	END IF;
	
	IF NOT EXISTS (SELECT id FROM schema_relationship_kinds ek WHERE ek.kind_id = retreived_kind_id) THEN
		INSERT INTO schema_relationship_kinds (schema_extension_id, kind_id, description, is_traversable) VALUES (v_extension_id, retreived_kind_id, v_description, v_is_traversable);
	ELSE
		UPDATE schema_relationship_kinds SET description = v_description, is_traversable = v_is_traversable WHERE kind_id = retreived_kind_id;
	END IF;
END;
$$ LANGUAGE plpgsql;
`)

	sb.WriteString(`
CREATE OR REPLACE FUNCTION genscript_upsert_schema_environments(v_extension_id INT, v_environment_kind_name VARCHAR(256), v_source_kind_name VARCHAR(256)) RETURNS INTEGER AS $$
DECLARE
	retreived_environment_kind_id SMALLINT;
	retreived_source_kind_id SMALLINT;
	schema_environment_id INTEGER;
BEGIN
	SELECT id INTO retreived_environment_kind_id FROM kind WHERE name = v_environment_kind_name;
	IF retreived_environment_kind_id IS NULL THEN
		RAISE EXCEPTION 'couldn''t find matching kind_id';
	END IF;

	SELECT id INTO retreived_source_kind_id FROM source_kinds WHERE name = v_source_kind_name;
	IF retreived_source_kind_id IS NULL THEN
		RAISE EXCEPTION 'couldn''t find matching kind_id';
	END IF;
	
	IF NOT EXISTS (SELECT id FROM schema_environments se WHERE se.schema_extension_id = v_extension_id) THEN
		INSERT INTO schema_environments (schema_extension_id, environment_kind_id, source_kind_id) VALUES (v_extension_id, retreived_environment_kind_id, retreived_source_kind_id) RETURNING id INTO schema_environment_id;
	ELSE
		UPDATE schema_environments SET environment_kind_id = retreived_environment_kind_id, source_kind_id = retreived_source_kind_id WHERE schema_extension_id = v_extension_id RETURNING id INTO schema_environment_id;
	END IF;

	RETURN schema_environment_id;
END;
$$ LANGUAGE plpgsql;
`)

	sb.WriteString(`
CREATE OR REPLACE FUNCTION genscript_upsert_schema_environments_principal_kinds(v_environment_id INT, v_principal_kind_name VARCHAR(256)) RETURNS void AS $$
DECLARE
	retreived_kind_id SMALLINT;
BEGIN
	SELECT id INTO retreived_kind_id FROM kind WHERE name = v_principal_kind_name;
	IF retreived_kind_id IS NULL THEN
		RAISE EXCEPTION 'couldn''t find matching kind_id';
	END IF;
	
	IF NOT EXISTS (SELECT 1 FROM schema_environments_principal_kinds pk WHERE pk.principal_kind = retreived_kind_id) THEN
		INSERT INTO schema_environments_principal_kinds (environment_id, principal_kind) VALUES (v_environment_id, retreived_kind_id);
	END IF;
END;
$$ LANGUAGE plpgsql;
`)

	sb.WriteString("\nDO $$\nDECLARE\n\textension_id INT;\n\tenvironment_id INT;\nBEGIN\n\tLOCK schema_extensions, schema_node_kinds, schema_relationship_kinds, kind;\n\n")

	sb.WriteString(fmt.Sprintf("\tIF NOT EXISTS (SELECT id FROM schema_extensions WHERE name = '%s') THEN\n", name))
	sb.WriteString(fmt.Sprintf("\t\tINSERT INTO schema_extensions (name, display_name, version, is_builtin, namespace) VALUES ('%s', '%s', '%s', true, '%s') RETURNING id INTO extension_id;\n", name, displayName, version, namespace))
	sb.WriteString("\tELSE\n")
	sb.WriteString(fmt.Sprintf("\t\tUPDATE schema_extensions SET display_name = '%s', version = '%s', namespace = '%s' WHERE name = '%s' RETURNING id INTO extension_id;\n", displayName, version, namespace, name))
	sb.WriteString("\tEND IF;\n\n")

	sb.WriteString("\t-- Insert Node Kinds\n")
	for _, kind := range nodeKinds {
		sb.WriteString(fmt.Sprintf("\tPERFORM genscript_upsert_kind('%s');\n", kind.GetRepresentation()))
	}

	sb.WriteString("\n\t-- Insert Relationship Kinds\n")
	for _, kind := range relationshipKinds {
		sb.WriteString(fmt.Sprintf("\tPERFORM genscript_upsert_kind('%s');\n", kind.GetRepresentation()))
	}

	sb.WriteString("\n")

	for _, kind := range nodeKinds {
		iconInfo, found := nodeIcons[kind.GetRepresentation()]

		sb.WriteString(fmt.Sprintf("\tPERFORM genscript_upsert_schema_node_kind(extension_id, '%s', '%s', '', %t, '%s', '%s');\n", kind.GetRepresentation(), kind.GetRepresentation(), found, iconInfo.Icon, iconInfo.Color))
	}

	traversableMap := make(map[string]struct{})

	for _, kind := range pathfindingRelationshipKinds {
		traversableMap[kind.GetRepresentation()] = struct{}{}
	}

	sb.WriteString("\n")

	for _, kind := range relationshipKinds {
		_, traversable := traversableMap[kind.GetRepresentation()]

		sb.WriteString(fmt.Sprintf("\tPERFORM genscript_upsert_schema_relationship_kind(extension_id, '%s', '', %t);\n", kind.GetRepresentation(), traversable))
	}

	sb.WriteString("\n")

	switch name {
	case "AD":
		GenerateADSpecifics(&sb)
	case "AZ":
		GenerateAZSpecifics(&sb)
	}

	sb.WriteString("END $$;\n")
	sb.WriteString(`
DROP FUNCTION IF EXISTS genscript_upsert_kind;
DROP FUNCTION IF EXISTS genscript_upsert_source_kind;
DROP FUNCTION IF EXISTS genscript_upsert_schema_node_kind;
DROP FUNCTION IF EXISTS genscript_upsert_schema_relationship_kind;
DROP FUNCTION IF EXISTS genscript_upsert_schema_environments;
DROP FUNCTION IF EXISTS genscript_upsert_schema_environments_principal_kinds;`)

	if _, err := os.Stat(dir); err != nil {
		if !os.IsNotExist(err) {
			return err
		}

		if err := os.MkdirAll(dir, defaultPackageDirPermission); err != nil {
			return err
		}
	}

	if fout, err := os.OpenFile(path.Join(dir, fileName), fileOpenMode, defaultSourceFilePermission); err != nil {
		return err
	} else {
		defer fout.Close()

		_, err := fout.WriteString(sb.String())
		return err
	}
}

func GenerateADSpecifics(sb io.StringWriter) {
	sb.WriteString("\tPERFORM genscript_upsert_source_kind('Base');\n")
	sb.WriteString("\tPERFORM genscript_upsert_kind('Domain');\n")
	sb.WriteString("\tSELECT genscript_upsert_schema_environments(extension_id, 'Domain', 'Base') INTO environment_id;\n")
	sb.WriteString("\tPERFORM genscript_upsert_schema_environments_principal_kinds(environment_id, 'User');\n")
	sb.WriteString("\tPERFORM genscript_upsert_schema_environments_principal_kinds(environment_id, 'Computer');\n")
}

func GenerateAZSpecifics(sb io.StringWriter) {
	sb.WriteString("\tPERFORM genscript_upsert_source_kind('AZBase');\n")
	sb.WriteString("\tPERFORM genscript_upsert_kind('Tenant');\n")
	sb.WriteString("\tSELECT genscript_upsert_schema_environments(extension_id, 'Tenant', 'AZBase') INTO environment_id;\n")
	sb.WriteString("\tPERFORM genscript_upsert_schema_environments_principal_kinds(environment_id, 'AZUser');\n")
	sb.WriteString("\tPERFORM genscript_upsert_schema_environments_principal_kinds(environment_id, 'AZVM');\n")
	sb.WriteString("\tPERFORM genscript_upsert_schema_environments_principal_kinds(environment_id, 'AZServicePrincipal');\n")
}
